/*
 * JCusparse - Java bindings for CUSPARSE, the NVIDIA CUDA sparse
 * matrix library, to be used with JCuda
 *
 * Copyright (c) 2010-2012 Marco Hutter - http://www.jcuda.org
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#include "JCusparse.hpp"
#include "JCusparse_common.hpp"
#include <iostream>
#include <string>
#include <map>

jfieldID cuComplex_x; // float
jfieldID cuComplex_y; // float

jfieldID cuDoubleComplex_x; // double
jfieldID cuDoubleComplex_y; // double


/**
 * Called when the library is loaded. Will initialize all
 * required field and method IDs
 */
JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM *jvm, void *reserved)
{
    JNIEnv *env = NULL;
    if (jvm->GetEnv((void **)&env, JNI_VERSION_1_4))
    {
        return JNI_ERR;
    }

    Logger::log(LOG_TRACE, "Initializing JCusparse\n");

    jclass cls = NULL;

    // Initialize the JNIUtils and PointerUtils
    if (initJNIUtils(env) == JNI_ERR) return JNI_ERR;
    if (initPointerUtils(env) == JNI_ERR) return JNI_ERR;


    // Obtain the fieldIDs for cuComplex#x and cuComplex#y
    if (!init(env, cls, "jcuda/cuComplex")) return JNI_ERR;
    if (!init(env, cls, cuComplex_x, "x", "F")) return JNI_ERR;
    if (!init(env, cls, cuComplex_y, "y", "F")) return JNI_ERR;


    // Obtain the fieldIDs for cuDoubleComplex#x and cuDoubleComplex#y
    if (!init(env, cls, "jcuda/cuDoubleComplex")) return JNI_ERR;
    if (!init(env, cls, cuDoubleComplex_x, "x", "D")) return JNI_ERR;
    if (!init(env, cls, cuDoubleComplex_y, "y", "D")) return JNI_ERR;

    return JNI_VERSION_1_4;
}


/*
 * Set the log level
 *
 * Class:     jcuda_jcusparse_JCusparse
 * Method:    setLogLevelNative
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_jcuda_jcusparse_JCusparse_setLogLevelNative
  (JNIEnv *env, jclass cla, jint logLevel)
{
    Logger::setLogLevel((LogLevel)logLevel);
}





/**
 * Converts the given jobject
 * into a cuComplex
 * and returns whether this conversion succeeded.
 */
bool initNative(JNIEnv *env, jobject &input, cuComplex &output)
{
    output.x = env->GetFloatField(input, cuComplex_x);
    output.y = env->GetFloatField(input, cuComplex_y);
    if (env->ExceptionCheck())
    {
        return false;
    }
    return true;
}

/**
 * Converts the given jobject
 * into a cuDoubleComplex
 * and returns whether this conversion succeeded.
 */
bool initNative(JNIEnv *env, jobject &input, cuDoubleComplex &output)
{
    output.x = env->GetDoubleField(input, cuDoubleComplex_x);
    output.y = env->GetDoubleField(input, cuDoubleComplex_y);
    if (env->ExceptionCheck())
    {
        return false;
    }
    return true;
}

/**
 * Converts the given cuComplex
 * into a jobject
 * and returns whether this conversion succeeded.
 */
bool releaseNative(JNIEnv *env, cuComplex &input, jobject &output)
{
    env->SetFloatField(output, cuComplex_x, input.x);
    env->SetFloatField(output, cuComplex_y, input.y);
    if (env->ExceptionCheck())
    {
        return false;
    }
    return true;
}

/**
 * Converts the given cuDoubleComplex
 * into a jobject
 * and returns whether this conversion succeeded.
 */
bool releaseNative(JNIEnv *env, cuDoubleComplex &input, jobject &output)
{
    env->SetDoubleField(output, cuDoubleComplex_x, input.x);
    env->SetDoubleField(output, cuDoubleComplex_y, input.y);
    if (env->ExceptionCheck())
    {
        return false;
    }
    return true;
}





/** CUSPARSE initialization and managment routines */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateNative(JNIEnv *env, jclass cls, jobject handle)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCreate");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreate(handle=%p)\n",
        handle);

    // Native variable declarations
    cusparseHandle_t handle_native;

    // Obtain native variable values
    // handle is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreate(&handle_native);

    // Write back native variable values
    setNativePointerValue(env, handle, (jlong)handle_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyNative(JNIEnv *env, jclass cls, jobject handle)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDestroy");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroy(handle=%p)\n",
        handle);

    // Native variable declarations
    cusparseHandle_t handle_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroy(handle_native);

    // Write back native variable values
    // handle is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetVersionNative(JNIEnv *env, jclass cls, jobject handle, jintArray version)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseGetVersion");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (version == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'version' is null for cusparseGetVersion");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetVersion(handle=%p, version=%p)\n",
        handle, version);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int version_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    // version is set here

    // Native function call
    cusparseStatus_t jniResult_native = cusparseGetVersion(handle_native, &version_native);

    // Write back native variable values
    // handle is a read-only native pointer
    if (!set(env, version, 0, (jint)version_native)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSetStreamNative(JNIEnv *env, jclass cls, jobject handle, jobject streamId)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSetStream");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // streamId may be NULL

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSetStream(handle=%p, streamId=%p)\n",
        handle, streamId);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cudaStream_t streamId_native = NULL;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    if (streamId != NULL)
    {
        streamId_native = (cudaStream_t)getNativePointerValue(env, streamId);
    }

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSetStream(handle_native, streamId_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // streamId is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** CUSPARSE type creation, destruction, set and get routines */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetPointerModeNative(JNIEnv *env, jclass cls, jobject handle, jintArray mode)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseGetPointerMode");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (mode == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'mode' is null for cusparseGetPointerMode");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetPointerMode(handle=%p, mode=%p)\n",
        handle, mode);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparsePointerMode_t mode_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    // mode is set here

    // Native function call
    cusparseStatus_t jniResult_native = cusparseGetPointerMode(handle_native, &mode_native);

    // Write back native variable values
    // handle is a read-only native pointer
    if (!set(env, mode, 0, (jint)mode_native)) return JCUSPARSE_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSetPointerModeNative(JNIEnv *env, jclass cls, jobject handle, jint mode)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSetPointerMode");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // mode is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSetPointerMode(handle=%p, mode=%d)\n",
        handle, mode);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparsePointerMode_t mode_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    mode_native = (cusparsePointerMode_t)mode;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSetPointerMode(handle_native, mode_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // mode is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** sparse matrix descriptor */
/**
* <pre>
* When the matrix descriptor is created, its fields are initialized to:
CUSPARSE_MATRIX_TYPE_GENERAL
CUSPARSE_INDEX_BASE_ZERO
All other fields are uninitialized
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateMatDescrNative(JNIEnv *env, jclass cls, jobject descrA)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCreateMatDescr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateMatDescr(descrA=%p)\n",
        descrA);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;

    // Obtain native variable values
    // descrA is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateMatDescr(&descrA_native);

    // Write back native variable values
    setNativePointerValue(env, descrA, (jlong)descrA_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyMatDescrNative(JNIEnv *env, jclass cls, jobject descrA)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDestroyMatDescr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyMatDescr(descrA=%p)\n",
        descrA);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyMatDescr(descrA_native);

    // Write back native variable values
    // descrA is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSetMatTypeNative(JNIEnv *env, jclass cls, jobject descrA, jint type)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSetMatType");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // type is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSetMatType(descrA=%p, type=%d)\n",
        descrA, type);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;
    cusparseMatrixType_t type_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    type_native = (cusparseMatrixType_t)type;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSetMatType(descrA_native, type_native);

    // Write back native variable values
    // descrA is a read-only native pointer
    // type is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetMatTypeNative(JNIEnv *env, jclass cls, jobject descrA)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseGetMatType");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetMatType(descrA=%p)\n",
        descrA);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);

    // Native function call
    cusparseMatrixType_t jniResult_native = cusparseGetMatType(descrA_native);

    // Write back native variable values
    // descrA is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSetMatFillModeNative(JNIEnv *env, jclass cls, jobject descrA, jint fillMode)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSetMatFillMode");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // fillMode is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSetMatFillMode(descrA=%p, fillMode=%d)\n",
        descrA, fillMode);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;
    cusparseFillMode_t fillMode_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    fillMode_native = (cusparseFillMode_t)fillMode;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSetMatFillMode(descrA_native, fillMode_native);

    // Write back native variable values
    // descrA is a read-only native pointer
    // fillMode is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetMatFillModeNative(JNIEnv *env, jclass cls, jobject descrA)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseGetMatFillMode");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetMatFillMode(descrA=%p)\n",
        descrA);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);

    // Native function call
    cusparseFillMode_t jniResult_native = cusparseGetMatFillMode(descrA_native);

    // Write back native variable values
    // descrA is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSetMatDiagTypeNative(JNIEnv *env, jclass cls, jobject descrA, jint diagType)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSetMatDiagType");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // diagType is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSetMatDiagType(descrA=%p, diagType=%d)\n",
        descrA, diagType);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;
    cusparseDiagType_t diagType_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    diagType_native = (cusparseDiagType_t)diagType;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSetMatDiagType(descrA_native, diagType_native);

    // Write back native variable values
    // descrA is a read-only native pointer
    // diagType is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetMatDiagTypeNative(JNIEnv *env, jclass cls, jobject descrA)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseGetMatDiagType");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetMatDiagType(descrA=%p)\n",
        descrA);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);

    // Native function call
    cusparseDiagType_t jniResult_native = cusparseGetMatDiagType(descrA_native);

    // Write back native variable values
    // descrA is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSetMatIndexBaseNative(JNIEnv *env, jclass cls, jobject descrA, jint base)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSetMatIndexBase");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // base is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSetMatIndexBase(descrA=%p, base=%d)\n",
        descrA, base);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;
    cusparseIndexBase_t base_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    base_native = (cusparseIndexBase_t)base;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSetMatIndexBase(descrA_native, base_native);

    // Write back native variable values
    // descrA is a read-only native pointer
    // base is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetMatIndexBaseNative(JNIEnv *env, jclass cls, jobject descrA)
{
    // Null-checks for non-primitive arguments
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseGetMatIndexBase");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetMatIndexBase(descrA=%p)\n",
        descrA);

    // Native variable declarations
    cusparseMatDescr_t descrA_native;

    // Obtain native variable values
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);

    // Native function call
    cusparseIndexBase_t jniResult_native = cusparseGetMatIndexBase(descrA_native);

    // Write back native variable values
    // descrA is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** sparse triangular solve and incomplete-LU and Cholesky (algorithm 1) */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateSolveAnalysisInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateSolveAnalysisInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateSolveAnalysisInfo(info=%p)\n",
        info);

    // Native variable declarations
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateSolveAnalysisInfo(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroySolveAnalysisInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroySolveAnalysisInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroySolveAnalysisInfo(info=%p)\n",
        info);

    // Native variable declarations
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroySolveAnalysisInfo(info_native);

    // Write back native variable values
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseGetLevelInfoNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jintArray nlevels, jobject levelPtr, jobject levelInd)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseGetLevelInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseGetLevelInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nlevels == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nlevels' is null for cusparseGetLevelInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (levelPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'levelPtr' is null for cusparseGetLevelInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (levelInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'levelInd' is null for cusparseGetLevelInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseGetLevelInfo(handle=%p, info=%p, nlevels=%p, levelPtr=%p, levelInd=%p)\n",
        handle, info, nlevels, levelPtr, levelInd);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseSolveAnalysisInfo_t info_native;
    int nlevels_native = 0;
    int** levelPtr_native;
    int** levelInd_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    // nlevels is set here
    levelPtr_native = (int**)getPointer(env, levelPtr);
    levelInd_native = (int**)getPointer(env, levelInd);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseGetLevelInfo(handle_native, info_native, &nlevels_native, levelPtr_native, levelInd_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    if (!set(env, nlevels, 0, (jint)nlevels_native)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // levelPtr is a native pointer
    // levelInd is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** sparse triangular solve (algorithm 2) */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateCsrsv2InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateCsrsv2Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateCsrsv2Info(info=%p)\n",
        info);

    // Native variable declarations
    csrsv2Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateCsrsv2Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyCsrsv2InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyCsrsv2Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyCsrsv2Info(info=%p)\n",
        info);

    // Native variable declarations
    csrsv2Info_t info_native;

    // Obtain native variable values
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyCsrsv2Info(info_native);

    // Write back native variable values
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** incomplete Cholesky (algorithm 2)*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateCsric02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateCsric02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateCsric02Info(info=%p)\n",
        info);

    // Native variable declarations
    csric02Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateCsric02Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyCsric02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyCsric02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyCsric02Info(info=%p)\n",
        info);

    // Native variable declarations
    csric02Info_t info_native;

    // Obtain native variable values
    info_native = (csric02Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyCsric02Info(info_native);

    // Write back native variable values
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateBsric02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateBsric02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateBsric02Info(info=%p)\n",
        info);

    // Native variable declarations
    bsric02Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateBsric02Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyBsric02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyBsric02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyBsric02Info(info=%p)\n",
        info);

    // Native variable declarations
    bsric02Info_t info_native;

    // Obtain native variable values
    info_native = (bsric02Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyBsric02Info(info_native);

    // Write back native variable values
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** incomplete LU (algorithm 2) */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateCsrilu02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateCsrilu02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateCsrilu02Info(info=%p)\n",
        info);

    // Native variable declarations
    csrilu02Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateCsrilu02Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyCsrilu02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyCsrilu02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyCsrilu02Info(info=%p)\n",
        info);

    // Native variable declarations
    csrilu02Info_t info_native;

    // Obtain native variable values
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyCsrilu02Info(info_native);

    // Write back native variable values
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateBsrilu02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateBsrilu02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateBsrilu02Info(info=%p)\n",
        info);

    // Native variable declarations
    bsrilu02Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateBsrilu02Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyBsrilu02InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyBsrilu02Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyBsrilu02Info(info=%p)\n",
        info);

    // Native variable declarations
    bsrilu02Info_t info_native;

    // Obtain native variable values
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyBsrilu02Info(info_native);

    // Write back native variable values
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** block-CSR triangular solve (algorithm 2) */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateBsrsv2InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateBsrsv2Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateBsrsv2Info(info=%p)\n",
        info);

    // Native variable declarations
    bsrsv2Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateBsrsv2Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyBsrsv2InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyBsrsv2Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyBsrsv2Info(info=%p)\n",
        info);

    // Native variable declarations
    bsrsv2Info_t info_native;

    // Obtain native variable values
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyBsrsv2Info(info_native);

    // Write back native variable values
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateBsrsm2InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateBsrsm2Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateBsrsm2Info(info=%p)\n",
        info);

    // Native variable declarations
    bsrsm2Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateBsrsm2Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyBsrsm2InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyBsrsm2Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyBsrsm2Info(info=%p)\n",
        info);

    // Native variable declarations
    bsrsm2Info_t info_native;

    // Obtain native variable values
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyBsrsm2Info(info_native);

    // Write back native variable values
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** hybrid (HYB) format */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateHybMatNative(JNIEnv *env, jclass cls, jobject hybA)
{
    // Null-checks for non-primitive arguments
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseCreateHybMat");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateHybMat(hybA=%p)\n",
        hybA);

    // Native variable declarations
    cusparseHybMat_t hybA_native;

    // Obtain native variable values
    // hybA is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateHybMat(&hybA_native);

    // Write back native variable values
    setNativePointerValue(env, hybA, (jlong)hybA_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyHybMatNative(JNIEnv *env, jclass cls, jobject hybA)
{
    // Null-checks for non-primitive arguments
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseDestroyHybMat");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyHybMat(hybA=%p)\n",
        hybA);

    // Native variable declarations
    cusparseHybMat_t hybA_native;

    // Obtain native variable values
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyHybMat(hybA_native);

    // Write back native variable values
    // hybA is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** sorting information */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateCsru2csrInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateCsru2csrInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateCsru2csrInfo(info=%p)\n",
        info);

    // Native variable declarations
    csru2csrInfo_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateCsru2csrInfo(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyCsru2csrInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyCsru2csrInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyCsru2csrInfo(info=%p)\n",
        info);

    // Native variable declarations
    csru2csrInfo_t info_native;

    // Obtain native variable values
    info_native = (csru2csrInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyCsru2csrInfo(info_native);

    // Write back native variable values
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** coloring info */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateColorInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateColorInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateColorInfo(info=%p)\n",
        info);

    // Native variable declarations
    cusparseColorInfo_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateColorInfo(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyColorInfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyColorInfo");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyColorInfo(info=%p)\n",
        info);

    // Native variable declarations
    cusparseColorInfo_t info_native;

    // Obtain native variable values
    info_native = (cusparseColorInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyColorInfo(info_native);

    // Write back native variable values
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** --- Sparse Level 1 routines --- */
/** Description: Addition of a scalar multiple of a sparse vector x
and a dense vector y. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSaxpyiNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject alpha, jobject xVal, jobject xInd, jobject y, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseSaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseSaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseSaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseSaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSaxpyi(handle=%p, nnz=%d, alpha=%p, xVal=%p, xInd=%p, y=%p, idxBase=%d)\n",
        handle, nnz, alpha, xVal, xInd, y, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    float* alpha_native;
    float* xVal_native;
    int* xInd_native;
    float* y_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    xVal_native = (float*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (float*)getPointer(env, y);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSaxpyi(handle_native, nnz_native, alpha_native, xVal_native, xInd_native, y_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDaxpyiNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject alpha, jobject xVal, jobject xInd, jobject y, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseDaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseDaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDaxpyi(handle=%p, nnz=%d, alpha=%p, xVal=%p, xInd=%p, y=%p, idxBase=%d)\n",
        handle, nnz, alpha, xVal, xInd, y, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    double* alpha_native;
    double* xVal_native;
    int* xInd_native;
    double* y_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    xVal_native = (double*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (double*)getPointer(env, y);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDaxpyi(handle_native, nnz_native, alpha_native, xVal_native, xInd_native, y_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCaxpyiNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject alpha, jobject xVal, jobject xInd, jobject y, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseCaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseCaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseCaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCaxpyi(handle=%p, nnz=%d, alpha=%p, xVal=%p, xInd=%p, y=%p, idxBase=%d)\n",
        handle, nnz, alpha, xVal, xInd, y, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    cuComplex* alpha_native;
    cuComplex* xVal_native;
    int* xInd_native;
    cuComplex* y_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    xVal_native = (cuComplex*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (cuComplex*)getPointer(env, y);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCaxpyi(handle_native, nnz_native, alpha_native, xVal_native, xInd_native, y_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZaxpyiNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject alpha, jobject xVal, jobject xInd, jobject y, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseZaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseZaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZaxpyi");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZaxpyi(handle=%p, nnz=%d, alpha=%p, xVal=%p, xInd=%p, y=%p, idxBase=%d)\n",
        handle, nnz, alpha, xVal, xInd, y, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    cuDoubleComplex* alpha_native;
    cuDoubleComplex* xVal_native;
    int* xInd_native;
    cuDoubleComplex* y_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    xVal_native = (cuDoubleComplex*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (cuDoubleComplex*)getPointer(env, y);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZaxpyi(handle_native, nnz_native, alpha_native, xVal_native, xInd_native, y_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: dot product of a sparse vector x and a dense vector y. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSdotiNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject xVal, jobject xInd, jobject y, jobject resultDevHostPtr, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseSdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseSdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseSdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (resultDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'resultDevHostPtr' is null for cusparseSdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSdoti(handle=%p, nnz=%d, xVal=%p, xInd=%p, y=%p, resultDevHostPtr=%p, idxBase=%d)\n",
        handle, nnz, xVal, xInd, y, resultDevHostPtr, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    float* xVal_native;
    int* xInd_native;
    float* y_native;
    float* resultDevHostPtr_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    xVal_native = (float*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (float*)getPointer(env, y);
    PointerData *resultDevHostPtr_pointerData = initPointerData(env, resultDevHostPtr);
    if (resultDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    resultDevHostPtr_native = (float*)resultDevHostPtr_pointerData->getPointer(env);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSdoti(handle_native, nnz_native, xVal_native, xInd_native, y_native, resultDevHostPtr_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, resultDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, resultDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDdotiNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject xVal, jobject xInd, jobject y, jobject resultDevHostPtr, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseDdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseDdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (resultDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'resultDevHostPtr' is null for cusparseDdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDdoti(handle=%p, nnz=%d, xVal=%p, xInd=%p, y=%p, resultDevHostPtr=%p, idxBase=%d)\n",
        handle, nnz, xVal, xInd, y, resultDevHostPtr, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    double* xVal_native;
    int* xInd_native;
    double* y_native;
    double* resultDevHostPtr_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    xVal_native = (double*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (double*)getPointer(env, y);
    PointerData *resultDevHostPtr_pointerData = initPointerData(env, resultDevHostPtr);
    if (resultDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    resultDevHostPtr_native = (double*)resultDevHostPtr_pointerData->getPointer(env);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDdoti(handle_native, nnz_native, xVal_native, xInd_native, y_native, resultDevHostPtr_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, resultDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, resultDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCdotiNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject xVal, jobject xInd, jobject y, jobject resultDevHostPtr, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseCdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseCdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseCdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (resultDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'resultDevHostPtr' is null for cusparseCdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCdoti(handle=%p, nnz=%d, xVal=%p, xInd=%p, y=%p, resultDevHostPtr=%p, idxBase=%d)\n",
        handle, nnz, xVal, xInd, y, resultDevHostPtr, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    cuComplex* xVal_native;
    int* xInd_native;
    cuComplex* y_native;
    cuComplex* resultDevHostPtr_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    xVal_native = (cuComplex*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (cuComplex*)getPointer(env, y);
    PointerData *resultDevHostPtr_pointerData = initPointerData(env, resultDevHostPtr);
    if (resultDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    resultDevHostPtr_native = (cuComplex*)resultDevHostPtr_pointerData->getPointer(env);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCdoti(handle_native, nnz_native, xVal_native, xInd_native, y_native, resultDevHostPtr_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, resultDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, resultDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZdotiNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject xVal, jobject xInd, jobject y, jobject resultDevHostPtr, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseZdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseZdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (resultDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'resultDevHostPtr' is null for cusparseZdoti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZdoti(handle=%p, nnz=%d, xVal=%p, xInd=%p, y=%p, resultDevHostPtr=%p, idxBase=%d)\n",
        handle, nnz, xVal, xInd, y, resultDevHostPtr, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    cuDoubleComplex* xVal_native;
    int* xInd_native;
    cuDoubleComplex* y_native;
    cuDoubleComplex* resultDevHostPtr_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    xVal_native = (cuDoubleComplex*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (cuDoubleComplex*)getPointer(env, y);
    PointerData *resultDevHostPtr_pointerData = initPointerData(env, resultDevHostPtr);
    if (resultDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    resultDevHostPtr_native = (cuDoubleComplex*)resultDevHostPtr_pointerData->getPointer(env);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZdoti(handle_native, nnz_native, xVal_native, xInd_native, y_native, resultDevHostPtr_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, resultDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, resultDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: dot product of complex conjugate of a sparse vector x
and a dense vector y. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCdotciNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject xVal, jobject xInd, jobject y, jobject resultDevHostPtr, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCdotci");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseCdotci");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseCdotci");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseCdotci");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (resultDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'resultDevHostPtr' is null for cusparseCdotci");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCdotci(handle=%p, nnz=%d, xVal=%p, xInd=%p, y=%p, resultDevHostPtr=%p, idxBase=%d)\n",
        handle, nnz, xVal, xInd, y, resultDevHostPtr, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    cuComplex* xVal_native;
    int* xInd_native;
    cuComplex* y_native;
    cuComplex* resultDevHostPtr_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    xVal_native = (cuComplex*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (cuComplex*)getPointer(env, y);
    PointerData *resultDevHostPtr_pointerData = initPointerData(env, resultDevHostPtr);
    if (resultDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    resultDevHostPtr_native = (cuComplex*)resultDevHostPtr_pointerData->getPointer(env);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCdotci(handle_native, nnz_native, xVal_native, xInd_native, y_native, resultDevHostPtr_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, resultDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, resultDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZdotciNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject xVal, jobject xInd, jobject y, jobject resultDevHostPtr, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZdotci");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseZdotci");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseZdotci");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZdotci");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (resultDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'resultDevHostPtr' is null for cusparseZdotci");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZdotci(handle=%p, nnz=%d, xVal=%p, xInd=%p, y=%p, resultDevHostPtr=%p, idxBase=%d)\n",
        handle, nnz, xVal, xInd, y, resultDevHostPtr, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    cuDoubleComplex* xVal_native;
    int* xInd_native;
    cuDoubleComplex* y_native;
    cuDoubleComplex* resultDevHostPtr_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    xVal_native = (cuDoubleComplex*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (cuDoubleComplex*)getPointer(env, y);
    PointerData *resultDevHostPtr_pointerData = initPointerData(env, resultDevHostPtr);
    if (resultDevHostPtr_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    resultDevHostPtr_native = (cuDoubleComplex*)resultDevHostPtr_pointerData->getPointer(env);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZdotci(handle_native, nnz_native, xVal_native, xInd_native, y_native, resultDevHostPtr_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // If the PointerData is not backed by native memory, then this call has to block
    if (!isPointerBackedByNativeMemory(env, resultDevHostPtr))
    {
        cudaDeviceSynchronize();
    }
    if (!releasePointerData(env, resultDevHostPtr_pointerData, 0)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: Gather of non-zero elements from dense vector y into
sparse vector x. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgthrNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject y, jobject xVal, jobject xInd, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseSgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseSgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseSgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgthr(handle=%p, nnz=%d, y=%p, xVal=%p, xInd=%p, idxBase=%d)\n",
        handle, nnz, y, xVal, xInd, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    float* y_native;
    float* xVal_native;
    int* xInd_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    y_native = (float*)getPointer(env, y);
    xVal_native = (float*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgthr(handle_native, nnz_native, y_native, xVal_native, xInd_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // y is a native pointer
    // xVal is a native pointer
    // xInd is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgthrNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject y, jobject xVal, jobject xInd, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseDgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseDgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgthr(handle=%p, nnz=%d, y=%p, xVal=%p, xInd=%p, idxBase=%d)\n",
        handle, nnz, y, xVal, xInd, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    double* y_native;
    double* xVal_native;
    int* xInd_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    y_native = (double*)getPointer(env, y);
    xVal_native = (double*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgthr(handle_native, nnz_native, y_native, xVal_native, xInd_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // y is a native pointer
    // xVal is a native pointer
    // xInd is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgthrNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject y, jobject xVal, jobject xInd, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseCgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseCgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseCgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgthr(handle=%p, nnz=%d, y=%p, xVal=%p, xInd=%p, idxBase=%d)\n",
        handle, nnz, y, xVal, xInd, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    cuComplex* y_native;
    cuComplex* xVal_native;
    int* xInd_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    y_native = (cuComplex*)getPointer(env, y);
    xVal_native = (cuComplex*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgthr(handle_native, nnz_native, y_native, xVal_native, xInd_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // y is a native pointer
    // xVal is a native pointer
    // xInd is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgthrNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject y, jobject xVal, jobject xInd, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseZgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseZgthr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgthr(handle=%p, nnz=%d, y=%p, xVal=%p, xInd=%p, idxBase=%d)\n",
        handle, nnz, y, xVal, xInd, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    cuDoubleComplex* y_native;
    cuDoubleComplex* xVal_native;
    int* xInd_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    y_native = (cuDoubleComplex*)getPointer(env, y);
    xVal_native = (cuDoubleComplex*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgthr(handle_native, nnz_native, y_native, xVal_native, xInd_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // y is a native pointer
    // xVal is a native pointer
    // xInd is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: Gather of non-zero elements from desne vector y into
sparse vector x (also replacing these elements in y by zeros). */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgthrzNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject y, jobject xVal, jobject xInd, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseSgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseSgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseSgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgthrz(handle=%p, nnz=%d, y=%p, xVal=%p, xInd=%p, idxBase=%d)\n",
        handle, nnz, y, xVal, xInd, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    float* y_native;
    float* xVal_native;
    int* xInd_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    y_native = (float*)getPointer(env, y);
    xVal_native = (float*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgthrz(handle_native, nnz_native, y_native, xVal_native, xInd_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // y is a native pointer
    // xVal is a native pointer
    // xInd is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgthrzNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject y, jobject xVal, jobject xInd, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseDgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseDgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgthrz(handle=%p, nnz=%d, y=%p, xVal=%p, xInd=%p, idxBase=%d)\n",
        handle, nnz, y, xVal, xInd, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    double* y_native;
    double* xVal_native;
    int* xInd_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    y_native = (double*)getPointer(env, y);
    xVal_native = (double*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgthrz(handle_native, nnz_native, y_native, xVal_native, xInd_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // y is a native pointer
    // xVal is a native pointer
    // xInd is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgthrzNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject y, jobject xVal, jobject xInd, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseCgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseCgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseCgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgthrz(handle=%p, nnz=%d, y=%p, xVal=%p, xInd=%p, idxBase=%d)\n",
        handle, nnz, y, xVal, xInd, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    cuComplex* y_native;
    cuComplex* xVal_native;
    int* xInd_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    y_native = (cuComplex*)getPointer(env, y);
    xVal_native = (cuComplex*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgthrz(handle_native, nnz_native, y_native, xVal_native, xInd_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // y is a native pointer
    // xVal is a native pointer
    // xInd is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgthrzNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject y, jobject xVal, jobject xInd, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseZgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseZgthrz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgthrz(handle=%p, nnz=%d, y=%p, xVal=%p, xInd=%p, idxBase=%d)\n",
        handle, nnz, y, xVal, xInd, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    cuDoubleComplex* y_native;
    cuDoubleComplex* xVal_native;
    int* xInd_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    y_native = (cuDoubleComplex*)getPointer(env, y);
    xVal_native = (cuDoubleComplex*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgthrz(handle_native, nnz_native, y_native, xVal_native, xInd_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // y is a native pointer
    // xVal is a native pointer
    // xInd is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: Scatter of elements of the sparse vector x into
dense vector y. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSsctrNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject xVal, jobject xInd, jobject y, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseSsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseSsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseSsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSsctr(handle=%p, nnz=%d, xVal=%p, xInd=%p, y=%p, idxBase=%d)\n",
        handle, nnz, xVal, xInd, y, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    float* xVal_native;
    int* xInd_native;
    float* y_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    xVal_native = (float*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (float*)getPointer(env, y);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSsctr(handle_native, nnz_native, xVal_native, xInd_native, y_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDsctrNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject xVal, jobject xInd, jobject y, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseDsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseDsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDsctr(handle=%p, nnz=%d, xVal=%p, xInd=%p, y=%p, idxBase=%d)\n",
        handle, nnz, xVal, xInd, y, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    double* xVal_native;
    int* xInd_native;
    double* y_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    xVal_native = (double*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (double*)getPointer(env, y);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDsctr(handle_native, nnz_native, xVal_native, xInd_native, y_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCsctrNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject xVal, jobject xInd, jobject y, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseCsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseCsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseCsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCsctr(handle=%p, nnz=%d, xVal=%p, xInd=%p, y=%p, idxBase=%d)\n",
        handle, nnz, xVal, xInd, y, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    cuComplex* xVal_native;
    int* xInd_native;
    cuComplex* y_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    xVal_native = (cuComplex*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (cuComplex*)getPointer(env, y);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCsctr(handle_native, nnz_native, xVal_native, xInd_native, y_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZsctrNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject xVal, jobject xInd, jobject y, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseZsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseZsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZsctr");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZsctr(handle=%p, nnz=%d, xVal=%p, xInd=%p, y=%p, idxBase=%d)\n",
        handle, nnz, xVal, xInd, y, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    cuDoubleComplex* xVal_native;
    int* xInd_native;
    cuDoubleComplex* y_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    xVal_native = (cuDoubleComplex*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (cuDoubleComplex*)getPointer(env, y);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZsctr(handle_native, nnz_native, xVal_native, xInd_native, y_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: Givens rotation, where c and s are cosine and sine,
x and y are sparse and dense vectors, respectively. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSrotiNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject xVal, jobject xInd, jobject y, jobject c, jobject s, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSroti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseSroti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseSroti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseSroti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (c == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'c' is null for cusparseSroti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (s == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 's' is null for cusparseSroti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSroti(handle=%p, nnz=%d, xVal=%p, xInd=%p, y=%p, c=%p, s=%p, idxBase=%d)\n",
        handle, nnz, xVal, xInd, y, c, s, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    float* xVal_native;
    int* xInd_native;
    float* y_native;
    float* c_native;
    float* s_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    xVal_native = (float*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (float*)getPointer(env, y);
    c_native = (float*)getPointer(env, c);
    s_native = (float*)getPointer(env, s);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSroti(handle_native, nnz_native, xVal_native, xInd_native, y_native, c_native, s_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // c is a native pointer
    // s is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDrotiNative(JNIEnv *env, jclass cls, jobject handle, jint nnz, jobject xVal, jobject xInd, jobject y, jobject c, jobject s, jint idxBase)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDroti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnz is primitive
    if (xVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xVal' is null for cusparseDroti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (xInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'xInd' is null for cusparseDroti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDroti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (c == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'c' is null for cusparseDroti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (s == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 's' is null for cusparseDroti");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // idxBase is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDroti(handle=%p, nnz=%d, xVal=%p, xInd=%p, y=%p, c=%p, s=%p, idxBase=%d)\n",
        handle, nnz, xVal, xInd, y, c, s, idxBase);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int nnz_native = 0;
    double* xVal_native;
    int* xInd_native;
    double* y_native;
    double* c_native;
    double* s_native;
    cusparseIndexBase_t idxBase_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    nnz_native = (int)nnz;
    xVal_native = (double*)getPointer(env, xVal);
    xInd_native = (int*)getPointer(env, xInd);
    y_native = (double*)getPointer(env, y);
    c_native = (double*)getPointer(env, c);
    s_native = (double*)getPointer(env, s);
    idxBase_native = (cusparseIndexBase_t)idxBase;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDroti(handle_native, nnz_native, xVal_native, xInd_native, y_native, c_native, s_native, idxBase_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // nnz is primitive
    // xVal is a native pointer
    // xInd is a native pointer
    // y is a native pointer
    // c is a native pointer
    // s is a native pointer
    // idxBase is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** --- Sparse Level 2 routines --- */
/** Description: Matrix-vector multiplication  y = alpha * op(A) * x  + beta * y,
where A is a sparse matrix in CSR storage format, x and y are dense vectors. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrmvNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseScsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseScsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseScsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseScsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrmv(handle=%p, transA=%d, m=%d, n=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, x=%p, beta=%p, y=%p)\n",
        handle, transA, m, n, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    float* x_native;
    float* beta_native;
    float* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    x_native = (float*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float*)beta_pointerData->getPointer(env);
    y_native = (float*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrmv(handle_native, transA_native, m_native, n_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrmvNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrmv(handle=%p, transA=%d, m=%d, n=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, x=%p, beta=%p, y=%p)\n",
        handle, transA, m, n, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    double* x_native;
    double* beta_native;
    double* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    x_native = (double*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double*)beta_pointerData->getPointer(env);
    y_native = (double*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrmv(handle_native, transA_native, m_native, n_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrmvNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseCcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrmv(handle=%p, transA=%d, m=%d, n=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, x=%p, beta=%p, y=%p)\n",
        handle, transA, m, n, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cuComplex* x_native;
    cuComplex* beta_native;
    cuComplex* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    x_native = (cuComplex*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex*)beta_pointerData->getPointer(env);
    y_native = (cuComplex*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrmv(handle_native, transA_native, m_native, n_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrmvNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZcsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrmv(handle=%p, transA=%d, m=%d, n=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, x=%p, beta=%p, y=%p)\n",
        handle, transA, m, n, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    int nnz_native = 0;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cuDoubleComplex* x_native;
    cuDoubleComplex* beta_native;
    cuDoubleComplex* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    x_native = (cuDoubleComplex*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex*)beta_pointerData->getPointer(env);
    y_native = (cuDoubleComplex*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrmv(handle_native, transA_native, m_native, n_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // n is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: Matrix-vector multiplication  y = alpha * op(A) * x  + beta * y,
where A is a sparse matrix in HYB storage format, x and y are dense vectors. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseShybmvNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jobject alpha, jobject descrA, jobject hybA, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseShybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseShybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseShybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseShybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseShybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseShybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseShybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseShybmv(handle=%p, transA=%d, alpha=%p, descrA=%p, hybA=%p, x=%p, beta=%p, y=%p)\n",
        handle, transA, alpha, descrA, hybA, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    cusparseHybMat_t hybA_native;
    float* x_native;
    float* beta_native;
    float* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);
    x_native = (float*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float*)beta_pointerData->getPointer(env);
    y_native = (float*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseShybmv(handle_native, transA_native, alpha_native, descrA_native, hybA_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // hybA is a read-only native pointer
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDhybmvNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jobject alpha, jobject descrA, jobject hybA, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseDhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDhybmv(handle=%p, transA=%d, alpha=%p, descrA=%p, hybA=%p, x=%p, beta=%p, y=%p)\n",
        handle, transA, alpha, descrA, hybA, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    cusparseHybMat_t hybA_native;
    double* x_native;
    double* beta_native;
    double* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);
    x_native = (double*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double*)beta_pointerData->getPointer(env);
    y_native = (double*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDhybmv(handle_native, transA_native, alpha_native, descrA_native, hybA_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // hybA is a read-only native pointer
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseChybmvNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jobject alpha, jobject descrA, jobject hybA, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseChybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseChybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseChybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseChybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseChybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseChybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseChybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseChybmv(handle=%p, transA=%d, alpha=%p, descrA=%p, hybA=%p, x=%p, beta=%p, y=%p)\n",
        handle, transA, alpha, descrA, hybA, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cusparseHybMat_t hybA_native;
    cuComplex* x_native;
    cuComplex* beta_native;
    cuComplex* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);
    x_native = (cuComplex*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex*)beta_pointerData->getPointer(env);
    y_native = (cuComplex*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseChybmv(handle_native, transA_native, alpha_native, descrA_native, hybA_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // hybA is a read-only native pointer
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZhybmvNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jobject alpha, jobject descrA, jobject hybA, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseZhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZhybmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZhybmv(handle=%p, transA=%d, alpha=%p, descrA=%p, hybA=%p, x=%p, beta=%p, y=%p)\n",
        handle, transA, alpha, descrA, hybA, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cusparseHybMat_t hybA_native;
    cuDoubleComplex* x_native;
    cuDoubleComplex* beta_native;
    cuDoubleComplex* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);
    x_native = (cuDoubleComplex*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex*)beta_pointerData->getPointer(env);
    y_native = (cuDoubleComplex*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZhybmv(handle_native, transA_native, alpha_native, descrA_native, hybA_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // hybA is a read-only native pointer
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: Matrix-vector multiplication  y = alpha * op(A) * x  + beta * y,
where A is a sparse matrix in BSR storage format, x and y are dense vectors. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseSbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrmv(handle=%p, dirA=%d, transA=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    float* x_native;
    float* beta_native;
    float* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (float*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float*)beta_pointerData->getPointer(env);
    y_native = (float*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrmv(handle_native, dirA_native, transA_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrmv(handle=%p, dirA=%d, transA=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    double* x_native;
    double* beta_native;
    double* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (double*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double*)beta_pointerData->getPointer(env);
    y_native = (double*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrmv(handle_native, dirA_native, transA_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseCbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrmv(handle=%p, dirA=%d, transA=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    cuComplex* x_native;
    cuComplex* beta_native;
    cuComplex* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (cuComplex*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex*)beta_pointerData->getPointer(env);
    y_native = (cuComplex*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrmv(handle_native, dirA_native, transA_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZbsrmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrmv(handle=%p, dirA=%d, transA=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    cuDoubleComplex* x_native;
    cuDoubleComplex* beta_native;
    cuDoubleComplex* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (cuDoubleComplex*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex*)beta_pointerData->getPointer(env);
    y_native = (cuDoubleComplex*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrmv(handle_native, dirA_native, transA_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: Matrix-vector multiplication  y = alpha * op(A) * x  + beta * y,
where A is a sparse matrix in extended BSR storage format, x and y are dense
vectors. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrxmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint sizeOfMask, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedMaskPtrA, jobject bsrSortedRowPtrA, jobject bsrSortedEndPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedMaskPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedMaskPtrA' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedEndPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedEndPtrA' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseSbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrxmv(handle=%p, dirA=%d, transA=%d, sizeOfMask=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedMaskPtrA=%p, bsrSortedRowPtrA=%p, bsrSortedEndPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, sizeOfMask, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedMaskPtrA, bsrSortedRowPtrA, bsrSortedEndPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int sizeOfMask_native = 0;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedValA_native;
    int* bsrSortedMaskPtrA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedEndPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    float* x_native;
    float* beta_native;
    float* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    sizeOfMask_native = (int)sizeOfMask;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float*)getPointer(env, bsrSortedValA);
    bsrSortedMaskPtrA_native = (int*)getPointer(env, bsrSortedMaskPtrA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedEndPtrA_native = (int*)getPointer(env, bsrSortedEndPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (float*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float*)beta_pointerData->getPointer(env);
    y_native = (float*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrxmv(handle_native, dirA_native, transA_native, sizeOfMask_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedMaskPtrA_native, bsrSortedRowPtrA_native, bsrSortedEndPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedMaskPtrA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedEndPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrxmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint sizeOfMask, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedMaskPtrA, jobject bsrSortedRowPtrA, jobject bsrSortedEndPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedMaskPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedMaskPtrA' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedEndPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedEndPtrA' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseDbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrxmv(handle=%p, dirA=%d, transA=%d, sizeOfMask=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedMaskPtrA=%p, bsrSortedRowPtrA=%p, bsrSortedEndPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, sizeOfMask, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedMaskPtrA, bsrSortedRowPtrA, bsrSortedEndPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int sizeOfMask_native = 0;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedValA_native;
    int* bsrSortedMaskPtrA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedEndPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    double* x_native;
    double* beta_native;
    double* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    sizeOfMask_native = (int)sizeOfMask;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double*)getPointer(env, bsrSortedValA);
    bsrSortedMaskPtrA_native = (int*)getPointer(env, bsrSortedMaskPtrA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedEndPtrA_native = (int*)getPointer(env, bsrSortedEndPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (double*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double*)beta_pointerData->getPointer(env);
    y_native = (double*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrxmv(handle_native, dirA_native, transA_native, sizeOfMask_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedMaskPtrA_native, bsrSortedRowPtrA_native, bsrSortedEndPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedMaskPtrA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedEndPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrxmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint sizeOfMask, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedMaskPtrA, jobject bsrSortedRowPtrA, jobject bsrSortedEndPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedMaskPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedMaskPtrA' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedEndPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedEndPtrA' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseCbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrxmv(handle=%p, dirA=%d, transA=%d, sizeOfMask=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedMaskPtrA=%p, bsrSortedRowPtrA=%p, bsrSortedEndPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, sizeOfMask, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedMaskPtrA, bsrSortedRowPtrA, bsrSortedEndPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int sizeOfMask_native = 0;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedValA_native;
    int* bsrSortedMaskPtrA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedEndPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    cuComplex* x_native;
    cuComplex* beta_native;
    cuComplex* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    sizeOfMask_native = (int)sizeOfMask;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex*)getPointer(env, bsrSortedValA);
    bsrSortedMaskPtrA_native = (int*)getPointer(env, bsrSortedMaskPtrA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedEndPtrA_native = (int*)getPointer(env, bsrSortedEndPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (cuComplex*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex*)beta_pointerData->getPointer(env);
    y_native = (cuComplex*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrxmv(handle_native, dirA_native, transA_native, sizeOfMask_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedMaskPtrA_native, bsrSortedRowPtrA_native, bsrSortedEndPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedMaskPtrA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedEndPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrxmvNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint sizeOfMask, jint mb, jint nb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedMaskPtrA, jobject bsrSortedRowPtrA, jobject bsrSortedEndPtrA, jobject bsrSortedColIndA, jint blockDim, jobject x, jobject beta, jobject y)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedMaskPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedMaskPtrA' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedEndPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedEndPtrA' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (y == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'y' is null for cusparseZbsrxmv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrxmv(handle=%p, dirA=%d, transA=%d, sizeOfMask=%d, mb=%d, nb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedMaskPtrA=%p, bsrSortedRowPtrA=%p, bsrSortedEndPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, x=%p, beta=%p, y=%p)\n",
        handle, dirA, transA, sizeOfMask, mb, nb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedMaskPtrA, bsrSortedRowPtrA, bsrSortedEndPtrA, bsrSortedColIndA, blockDim, x, beta, y);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int sizeOfMask_native = 0;
    int mb_native = 0;
    int nb_native = 0;
    int nnzb_native = 0;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedValA_native;
    int* bsrSortedMaskPtrA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedEndPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    cuDoubleComplex* x_native;
    cuDoubleComplex* beta_native;
    cuDoubleComplex* y_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    sizeOfMask_native = (int)sizeOfMask;
    mb_native = (int)mb;
    nb_native = (int)nb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex*)getPointer(env, bsrSortedValA);
    bsrSortedMaskPtrA_native = (int*)getPointer(env, bsrSortedMaskPtrA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedEndPtrA_native = (int*)getPointer(env, bsrSortedEndPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    x_native = (cuDoubleComplex*)getPointer(env, x);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex*)beta_pointerData->getPointer(env);
    y_native = (cuDoubleComplex*)getPointer(env, y);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrxmv(handle_native, dirA_native, transA_native, sizeOfMask_native, mb_native, nb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedMaskPtrA_native, bsrSortedRowPtrA_native, bsrSortedEndPtrA_native, bsrSortedColIndA_native, blockDim_native, x_native, beta_native, y_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // sizeOfMask is primitive
    // mb is primitive
    // nb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedMaskPtrA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedEndPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // x is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // y is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: Solution of triangular linear system op(A) * x = alpha * f,
where A is a sparse matrix in CSR storage format, rhs f and solution x
are dense vectors. This routine implements algorithm 1 for the solve. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrsv_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrsv_analysis(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrsv_analysis(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrsv_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrsv_analysis(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrsv_analysis(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrsv_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrsv_analysis(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrsv_analysis(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrsv_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrsv_analysis(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrsv_analysis(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrsv_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject f, jobject x)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseScsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseScsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseScsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrsv_solve(handle=%p, transA=%d, m=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, f=%p, x=%p)\n",
        handle, transA, m, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, f, x);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;
    float* f_native;
    float* x_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    f_native = (float*)getPointer(env, f);
    x_native = (float*)getPointer(env, x);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrsv_solve(handle_native, transA_native, m_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, f_native, x_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrsv_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject f, jobject x)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseDcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrsv_solve(handle=%p, transA=%d, m=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, f=%p, x=%p)\n",
        handle, transA, m, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, f, x);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;
    double* f_native;
    double* x_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    f_native = (double*)getPointer(env, f);
    x_native = (double*)getPointer(env, x);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrsv_solve(handle_native, transA_native, m_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, f_native, x_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrsv_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject f, jobject x)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseCcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrsv_solve(handle=%p, transA=%d, m=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, f=%p, x=%p)\n",
        handle, transA, m, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, f, x);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;
    cuComplex* f_native;
    cuComplex* x_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    f_native = (cuComplex*)getPointer(env, f);
    x_native = (cuComplex*)getPointer(env, x);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrsv_solve(handle_native, transA_native, m_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, f_native, x_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrsv_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject f, jobject x)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseZcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZcsrsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrsv_solve(handle=%p, transA=%d, m=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, f=%p, x=%p)\n",
        handle, transA, m, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, f, x);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;
    cuDoubleComplex* f_native;
    cuDoubleComplex* x_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    f_native = (cuDoubleComplex*)getPointer(env, f);
    x_native = (cuDoubleComplex*)getPointer(env, x);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrsv_solve(handle_native, transA_native, m_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, f_native, x_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
* <pre>
* Description: Solution of triangular linear system op(A) * x = alpha * f,
where A is a sparse matrix in CSR storage format, rhs f and solution y
are dense vectors. This routine implements algorithm 1 for this problem.
* Also, it provides a utility function to query size of buffer used.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcsrsv2_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcsrsv2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXcsrsv2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXcsrsv2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcsrsv2_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csrsv2Info_t info_native;
    int* position_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    position_native = (int*)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcsrsv2_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // position is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrsv2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseScsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrsv2_bufferSize(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrsv2_bufferSize(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrsv2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrsv2_bufferSize(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrsv2_bufferSize(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrsv2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrsv2_bufferSize(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrsv2_bufferSize(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrsv2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZcsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrsv2_bufferSize(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrsv2_bufferSize(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrsv2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseScsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrsv2_bufferSizeExt(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSize=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrsv2_bufferSizeExt(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrsv2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseDcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrsv2_bufferSizeExt(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSize=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrsv2_bufferSizeExt(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrsv2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseCcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrsv2_bufferSizeExt(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSize=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrsv2_bufferSizeExt(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrsv2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseZcsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrsv2_bufferSizeExt(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSize=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrsv2_bufferSizeExt(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrsv2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseScsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrsv2_analysis(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrsv2_analysis(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrsv2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrsv2_analysis(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrsv2_analysis(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrsv2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrsv2_analysis(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrsv2_analysis(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrsv2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZcsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrsv2_analysis(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrsv2_analysis(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrsv2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject f, jobject x, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseScsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseScsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseScsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseScsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrsv2_solve(handle=%p, transA=%d, m=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, f=%p, x=%p, policy=%d, pBuffer=%p)\n",
        handle, transA, m, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, f, x, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    float* f_native;
    float* x_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    f_native = (float*)getPointer(env, f);
    x_native = (float*)getPointer(env, x);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrsv2_solve(handle_native, transA_native, m_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, f_native, x_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrsv2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject f, jobject x, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseDcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrsv2_solve(handle=%p, transA=%d, m=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, f=%p, x=%p, policy=%d, pBuffer=%p)\n",
        handle, transA, m, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, f, x, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    double* f_native;
    double* x_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    f_native = (double*)getPointer(env, f);
    x_native = (double*)getPointer(env, x);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrsv2_solve(handle_native, transA_native, m_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, f_native, x_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrsv2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject f, jobject x, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseCcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrsv2_solve(handle=%p, transA=%d, m=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, f=%p, x=%p, policy=%d, pBuffer=%p)\n",
        handle, transA, m, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, f, x, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    cuComplex* f_native;
    cuComplex* x_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    f_native = (cuComplex*)getPointer(env, f);
    x_native = (cuComplex*)getPointer(env, x);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrsv2_solve(handle_native, transA_native, m_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, f_native, x_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrsv2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject f, jobject x, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseZcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZcsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrsv2_solve(handle=%p, transA=%d, m=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, f=%p, x=%p, policy=%d, pBuffer=%p)\n",
        handle, transA, m, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, f, x, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrsv2Info_t info_native;
    cuDoubleComplex* f_native;
    cuDoubleComplex* x_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrsv2Info_t)getNativePointerValue(env, info);
    f_native = (cuDoubleComplex*)getPointer(env, f);
    x_native = (cuDoubleComplex*)getPointer(env, x);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrsv2_solve(handle_native, transA_native, m_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, f_native, x_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
* <pre>
* Description: Solution of triangular linear system op(A) * x = alpha * f,
where A is a sparse matrix in block-CSR storage format, rhs f and solution y
are dense vectors. This routine implements algorithm 2 for this problem.
* Also, it provides a utility function to query size of buffer used.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXbsrsv2_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXbsrsv2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXbsrsv2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXbsrsv2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXbsrsv2_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrsv2Info_t info_native;
    int* position_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    position_native = (int*)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXbsrsv2_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // position is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsv2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseSbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsv2_bufferSize(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsv2_bufferSize(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsv2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsv2_bufferSize(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsv2_bufferSize(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsv2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsv2_bufferSize(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsv2_bufferSize(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsv2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZbsrsv2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsv2_bufferSize(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsv2_bufferSize(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsv2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseSbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsv2_bufferSizeExt(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockSize_native = 0;
    bsrsv2Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockSize_native = (int)blockSize;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsv2_bufferSizeExt(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsv2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseDbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsv2_bufferSizeExt(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockSize_native = 0;
    bsrsv2Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockSize_native = (int)blockSize;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsv2_bufferSizeExt(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsv2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseCbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsv2_bufferSizeExt(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockSize_native = 0;
    bsrsv2Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockSize_native = (int)blockSize;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsv2_bufferSizeExt(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsv2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseZbsrsv2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsv2_bufferSizeExt(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockSize_native = 0;
    bsrsv2Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockSize_native = (int)blockSize;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsv2_bufferSizeExt(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsv2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsv2_analysis(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsv2_analysis(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsv2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsv2_analysis(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsv2_analysis(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsv2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsv2_analysis(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsv2_analysis(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsv2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsrsv2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsv2_analysis(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsv2_analysis(handle_native, dirA_native, transA_native, mb_native, nnzb_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsv2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jobject f, jobject x, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsv2_solve(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, f=%p, x=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, f, x, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    float* f_native;
    float* x_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    f_native = (float*)getPointer(env, f);
    x_native = (float*)getPointer(env, x);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsv2_solve(handle_native, dirA_native, transA_native, mb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, f_native, x_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsv2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jobject f, jobject x, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsv2_solve(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, f=%p, x=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, f, x, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    double* f_native;
    double* x_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    f_native = (double*)getPointer(env, f);
    x_native = (double*)getPointer(env, x);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsv2_solve(handle_native, dirA_native, transA_native, mb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, f_native, x_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsv2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jobject f, jobject x, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsv2_solve(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, f=%p, x=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, f, x, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    cuComplex* f_native;
    cuComplex* x_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    f_native = (cuComplex*)getPointer(env, f);
    x_native = (cuComplex*)getPointer(env, x);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsv2_solve(handle_native, dirA_native, transA_native, mb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, f_native, x_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsv2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint mb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockDim, jobject info, jobject f, jobject x, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsrsv2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsv2_solve(handle=%p, dirA=%d, transA=%d, mb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockDim=%d, info=%p, f=%p, x=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, mb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockDim, info, f, x, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockDim_native = 0;
    bsrsv2Info_t info_native;
    cuDoubleComplex* f_native;
    cuDoubleComplex* x_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockDim_native = (int)blockDim;
    info_native = (bsrsv2Info_t)getNativePointerValue(env, info);
    f_native = (cuDoubleComplex*)getPointer(env, f);
    x_native = (cuDoubleComplex*)getPointer(env, x);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsv2_solve(handle_native, dirA_native, transA_native, mb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockDim_native, info_native, f_native, x_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // mb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: Solution of triangular linear system op(A) * x = alpha * f,
where A is a sparse matrix in HYB storage format, rhs f and solution x
are dense vectors. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseShybsv_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jobject descrA, jobject hybA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseShybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseShybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseShybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseShybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseShybsv_analysis(handle=%p, transA=%d, descrA=%p, hybA=%p, info=%p)\n",
        handle, transA, descrA, hybA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseMatDescr_t descrA_native;
    cusparseHybMat_t hybA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseShybsv_analysis(handle_native, transA_native, descrA_native, hybA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // descrA is a read-only native pointer
    // hybA is a read-only native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDhybsv_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jobject descrA, jobject hybA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDhybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDhybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseDhybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDhybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDhybsv_analysis(handle=%p, transA=%d, descrA=%p, hybA=%p, info=%p)\n",
        handle, transA, descrA, hybA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseMatDescr_t descrA_native;
    cusparseHybMat_t hybA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDhybsv_analysis(handle_native, transA_native, descrA_native, hybA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // descrA is a read-only native pointer
    // hybA is a read-only native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseChybsv_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jobject descrA, jobject hybA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseChybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseChybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseChybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseChybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseChybsv_analysis(handle=%p, transA=%d, descrA=%p, hybA=%p, info=%p)\n",
        handle, transA, descrA, hybA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseMatDescr_t descrA_native;
    cusparseHybMat_t hybA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseChybsv_analysis(handle_native, transA_native, descrA_native, hybA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // descrA is a read-only native pointer
    // hybA is a read-only native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZhybsv_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jobject descrA, jobject hybA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZhybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZhybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseZhybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZhybsv_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZhybsv_analysis(handle=%p, transA=%d, descrA=%p, hybA=%p, info=%p)\n",
        handle, transA, descrA, hybA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseMatDescr_t descrA_native;
    cusparseHybMat_t hybA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZhybsv_analysis(handle_native, transA_native, descrA_native, hybA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // descrA is a read-only native pointer
    // hybA is a read-only native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseShybsv_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint trans, jobject alpha, jobject descra, jobject hybA, jobject info, jobject f, jobject x)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseShybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseShybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descra == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descra' is null for cusparseShybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseShybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseShybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseShybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseShybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseShybsv_solve(handle=%p, trans=%d, alpha=%p, descra=%p, hybA=%p, info=%p, f=%p, x=%p)\n",
        handle, trans, alpha, descra, hybA, info, f, x);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t trans_native;
    float* alpha_native;
    cusparseMatDescr_t descra_native;
    cusparseHybMat_t hybA_native;
    cusparseSolveAnalysisInfo_t info_native;
    float* f_native;
    float* x_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    trans_native = (cusparseOperation_t)trans;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descra_native = (cusparseMatDescr_t)getNativePointerValue(env, descra);
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    f_native = (float*)getPointer(env, f);
    x_native = (float*)getPointer(env, x);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseShybsv_solve(handle_native, trans_native, alpha_native, descra_native, hybA_native, info_native, f_native, x_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // trans is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descra is a read-only native pointer
    // hybA is a read-only native pointer
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseChybsv_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint trans, jobject alpha, jobject descra, jobject hybA, jobject info, jobject f, jobject x)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseChybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseChybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descra == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descra' is null for cusparseChybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseChybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseChybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseChybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseChybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseChybsv_solve(handle=%p, trans=%d, alpha=%p, descra=%p, hybA=%p, info=%p, f=%p, x=%p)\n",
        handle, trans, alpha, descra, hybA, info, f, x);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t trans_native;
    cuComplex* alpha_native;
    cusparseMatDescr_t descra_native;
    cusparseHybMat_t hybA_native;
    cusparseSolveAnalysisInfo_t info_native;
    cuComplex* f_native;
    cuComplex* x_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    trans_native = (cusparseOperation_t)trans;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descra_native = (cusparseMatDescr_t)getNativePointerValue(env, descra);
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    f_native = (cuComplex*)getPointer(env, f);
    x_native = (cuComplex*)getPointer(env, x);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseChybsv_solve(handle_native, trans_native, alpha_native, descra_native, hybA_native, info_native, f_native, x_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // trans is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descra is a read-only native pointer
    // hybA is a read-only native pointer
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDhybsv_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint trans, jobject alpha, jobject descra, jobject hybA, jobject info, jobject f, jobject x)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descra == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descra' is null for cusparseDhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseDhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseDhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDhybsv_solve(handle=%p, trans=%d, alpha=%p, descra=%p, hybA=%p, info=%p, f=%p, x=%p)\n",
        handle, trans, alpha, descra, hybA, info, f, x);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t trans_native;
    double* alpha_native;
    cusparseMatDescr_t descra_native;
    cusparseHybMat_t hybA_native;
    cusparseSolveAnalysisInfo_t info_native;
    double* f_native;
    double* x_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    trans_native = (cusparseOperation_t)trans;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descra_native = (cusparseMatDescr_t)getNativePointerValue(env, descra);
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    f_native = (double*)getPointer(env, f);
    x_native = (double*)getPointer(env, x);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDhybsv_solve(handle_native, trans_native, alpha_native, descra_native, hybA_native, info_native, f_native, x_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // trans is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descra is a read-only native pointer
    // hybA is a read-only native pointer
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZhybsv_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint trans, jobject alpha, jobject descra, jobject hybA, jobject info, jobject f, jobject x)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descra == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descra' is null for cusparseZhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (hybA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'hybA' is null for cusparseZhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (f == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'f' is null for cusparseZhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZhybsv_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZhybsv_solve(handle=%p, trans=%d, alpha=%p, descra=%p, hybA=%p, info=%p, f=%p, x=%p)\n",
        handle, trans, alpha, descra, hybA, info, f, x);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t trans_native;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descra_native;
    cusparseHybMat_t hybA_native;
    cusparseSolveAnalysisInfo_t info_native;
    cuDoubleComplex* f_native;
    cuDoubleComplex* x_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    trans_native = (cusparseOperation_t)trans;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descra_native = (cusparseMatDescr_t)getNativePointerValue(env, descra);
    hybA_native = (cusparseHybMat_t)getNativePointerValue(env, hybA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    f_native = (cuDoubleComplex*)getPointer(env, f);
    x_native = (cuDoubleComplex*)getPointer(env, x);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZhybsv_solve(handle_native, trans_native, alpha_native, descra_native, hybA_native, info_native, f_native, x_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // trans is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descra is a read-only native pointer
    // hybA is a read-only native pointer
    // info is a read-only native pointer
    // f is a native pointer
    // x is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** --- Sparse Level 3 routines --- */
/** Description: sparse - dense matrix multiplication C = alpha * op(A) * B  + beta * C,
where A is a sparse matrix in CSR format, B and C are dense tall matrices.  */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrmmNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jint k, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseScsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseScsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseScsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseScsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrmm(handle=%p, transA=%d, m=%d, n=%d, k=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, transA, m, n, k, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    int nnz_native = 0;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    float* B_native;
    int ldb_native = 0;
    float* beta_native;
    float* C_native;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    B_native = (float*)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float*)beta_pointerData->getPointer(env);
    C_native = (float*)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrmm(handle_native, transA_native, m_native, n_native, k_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrmmNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jint k, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseDcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseDcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrmm(handle=%p, transA=%d, m=%d, n=%d, k=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, transA, m, n, k, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    int nnz_native = 0;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    double* B_native;
    int ldb_native = 0;
    double* beta_native;
    double* C_native;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    B_native = (double*)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double*)beta_pointerData->getPointer(env);
    C_native = (double*)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrmm(handle_native, transA_native, m_native, n_native, k_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrmmNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jint k, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseCcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseCcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrmm(handle=%p, transA=%d, m=%d, n=%d, k=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, transA, m, n, k, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    int nnz_native = 0;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cuComplex* B_native;
    int ldb_native = 0;
    cuComplex* beta_native;
    cuComplex* C_native;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    B_native = (cuComplex*)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex*)beta_pointerData->getPointer(env);
    C_native = (cuComplex*)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrmm(handle_native, transA_native, m_native, n_native, k_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrmmNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jint k, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseZcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseZcsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrmm(handle=%p, transA=%d, m=%d, n=%d, k=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, transA, m, n, k, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    int nnz_native = 0;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cuDoubleComplex* B_native;
    int ldb_native = 0;
    cuDoubleComplex* beta_native;
    cuDoubleComplex* C_native;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    B_native = (cuDoubleComplex*)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex*)beta_pointerData->getPointer(env);
    C_native = (cuDoubleComplex*)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrmm(handle_native, transA_native, m_native, n_native, k_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: sparse - dense matrix multiplication C = alpha * op(A) * B  + beta * C,
where A is a sparse matrix in CSR format, B and C are dense tall matrices.
This routine allows transposition of matrix B, which may improve performance. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrmm2Native(JNIEnv *env, jclass cls, jobject handle, jint transA, jint transB, jint m, jint n, jint k, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseScsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseScsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseScsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseScsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrmm2(handle=%p, transA=%d, transB=%d, m=%d, n=%d, k=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, transA, transB, m, n, k, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    int nnz_native = 0;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    float* B_native;
    int ldb_native = 0;
    float* beta_native;
    float* C_native;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    B_native = (float*)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float*)beta_pointerData->getPointer(env);
    C_native = (float*)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrmm2(handle_native, transA_native, transB_native, m_native, n_native, k_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrmm2Native(JNIEnv *env, jclass cls, jobject handle, jint transA, jint transB, jint m, jint n, jint k, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseDcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseDcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrmm2(handle=%p, transA=%d, transB=%d, m=%d, n=%d, k=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, transA, transB, m, n, k, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    int nnz_native = 0;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    double* B_native;
    int ldb_native = 0;
    double* beta_native;
    double* C_native;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    B_native = (double*)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double*)beta_pointerData->getPointer(env);
    C_native = (double*)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrmm2(handle_native, transA_native, transB_native, m_native, n_native, k_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrmm2Native(JNIEnv *env, jclass cls, jobject handle, jint transA, jint transB, jint m, jint n, jint k, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseCcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseCcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrmm2(handle=%p, transA=%d, transB=%d, m=%d, n=%d, k=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, transA, transB, m, n, k, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    int nnz_native = 0;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cuComplex* B_native;
    int ldb_native = 0;
    cuComplex* beta_native;
    cuComplex* C_native;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    B_native = (cuComplex*)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex*)beta_pointerData->getPointer(env);
    C_native = (cuComplex*)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrmm2(handle_native, transA_native, transB_native, m_native, n_native, k_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrmm2Native(JNIEnv *env, jclass cls, jobject handle, jint transA, jint transB, jint m, jint n, jint k, jint nnz, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseZcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseZcsrmm2");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrmm2(handle=%p, transA=%d, transB=%d, m=%d, n=%d, k=%d, nnz=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, transA, transB, m, n, k, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    int nnz_native = 0;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cuDoubleComplex* B_native;
    int ldb_native = 0;
    cuDoubleComplex* beta_native;
    cuDoubleComplex* C_native;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    nnz_native = (int)nnz;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    B_native = (cuDoubleComplex*)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex*)beta_pointerData->getPointer(env);
    C_native = (cuDoubleComplex*)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrmm2(handle_native, transA_native, transB_native, m_native, n_native, k_native, nnz_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // nnz is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: sparse - dense matrix multiplication C = alpha * op(A) * B  + beta * C,
where A is a sparse matrix in block-CSR format, B and C are dense tall matrices.
This routine allows transposition of matrix B, which may improve performance. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrmmNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transB, jint mb, jint n, jint kb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockSize, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseSbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrmm(handle=%p, dirA=%d, transA=%d, transB=%d, mb=%d, n=%d, kb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockSize=%d, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockSize, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int mb_native = 0;
    int n_native = 0;
    int kb_native = 0;
    int nnzb_native = 0;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockSize_native = 0;
    float* B_native;
    int ldb_native = 0;
    float* beta_native;
    float* C_native;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    mb_native = (int)mb;
    n_native = (int)n;
    kb_native = (int)kb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (float*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockSize_native = (int)blockSize;
    B_native = (float*)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float*)beta_pointerData->getPointer(env);
    C_native = (float*)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrmm(handle_native, dirA_native, transA_native, transB_native, mb_native, n_native, kb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockSize_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockSize is primitive
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrmmNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transB, jint mb, jint n, jint kb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockSize, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseDbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrmm(handle=%p, dirA=%d, transA=%d, transB=%d, mb=%d, n=%d, kb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockSize=%d, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockSize, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int mb_native = 0;
    int n_native = 0;
    int kb_native = 0;
    int nnzb_native = 0;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockSize_native = 0;
    double* B_native;
    int ldb_native = 0;
    double* beta_native;
    double* C_native;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    mb_native = (int)mb;
    n_native = (int)n;
    kb_native = (int)kb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (double*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockSize_native = (int)blockSize;
    B_native = (double*)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double*)beta_pointerData->getPointer(env);
    C_native = (double*)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrmm(handle_native, dirA_native, transA_native, transB_native, mb_native, n_native, kb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockSize_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockSize is primitive
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrmmNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transB, jint mb, jint n, jint kb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockSize, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseCbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrmm(handle=%p, dirA=%d, transA=%d, transB=%d, mb=%d, n=%d, kb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockSize=%d, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockSize, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int mb_native = 0;
    int n_native = 0;
    int kb_native = 0;
    int nnzb_native = 0;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockSize_native = 0;
    cuComplex* B_native;
    int ldb_native = 0;
    cuComplex* beta_native;
    cuComplex* C_native;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    mb_native = (int)mb;
    n_native = (int)n;
    kb_native = (int)kb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuComplex*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockSize_native = (int)blockSize;
    B_native = (cuComplex*)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuComplex*)beta_pointerData->getPointer(env);
    C_native = (cuComplex*)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrmm(handle_native, dirA_native, transA_native, transB_native, mb_native, n_native, kb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockSize_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockSize is primitive
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrmmNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transB, jint mb, jint n, jint kb, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedValA, jobject bsrSortedRowPtrA, jobject bsrSortedColIndA, jint blockSize, jobject B, jint ldb, jobject beta, jobject C, jint ldc)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedValA' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtrA' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColIndA' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (C == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'C' is null for cusparseZbsrmm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldc is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrmm(handle=%p, dirA=%d, transA=%d, transB=%d, mb=%d, n=%d, kb=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedValA=%p, bsrSortedRowPtrA=%p, bsrSortedColIndA=%p, blockSize=%d, B=%p, ldb=%d, beta=%p, C=%p, ldc=%d)\n",
        handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, descrA, bsrSortedValA, bsrSortedRowPtrA, bsrSortedColIndA, blockSize, B, ldb, beta, C, ldc);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int mb_native = 0;
    int n_native = 0;
    int kb_native = 0;
    int nnzb_native = 0;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedValA_native;
    int* bsrSortedRowPtrA_native;
    int* bsrSortedColIndA_native;
    int blockSize_native = 0;
    cuDoubleComplex* B_native;
    int ldb_native = 0;
    cuDoubleComplex* beta_native;
    cuDoubleComplex* C_native;
    int ldc_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    mb_native = (int)mb;
    n_native = (int)n;
    kb_native = (int)kb;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedValA_native = (cuDoubleComplex*)getPointer(env, bsrSortedValA);
    bsrSortedRowPtrA_native = (int*)getPointer(env, bsrSortedRowPtrA);
    bsrSortedColIndA_native = (int*)getPointer(env, bsrSortedColIndA);
    blockSize_native = (int)blockSize;
    B_native = (cuDoubleComplex*)getPointer(env, B);
    ldb_native = (int)ldb;
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (cuDoubleComplex*)beta_pointerData->getPointer(env);
    C_native = (cuDoubleComplex*)getPointer(env, C);
    ldc_native = (int)ldc;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrmm(handle_native, dirA_native, transA_native, transB_native, mb_native, n_native, kb_native, nnzb_native, alpha_native, descrA_native, bsrSortedValA_native, bsrSortedRowPtrA_native, bsrSortedColIndA_native, blockSize_native, B_native, ldb_native, beta_native, C_native, ldc_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // kb is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedValA is a native pointer
    // bsrSortedRowPtrA is a native pointer
    // bsrSortedColIndA is a native pointer
    // blockSize is primitive
    // B is a native pointer
    // ldb is primitive
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // C is a native pointer
    // ldc is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
* <pre>
* Description: Solution of triangular linear system op(A) * X = alpha * F,
with multiple right-hand-sides, where A is a sparse matrix in CSR storage
format, rhs F and solution X are dense tall matrices.
* This routine implements algorithm 1 for this problem.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrsm_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrsm_analysis(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrsm_analysis(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrsm_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrsm_analysis(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrsm_analysis(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrsm_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrsm_analysis(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrsm_analysis(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrsm_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrsm_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrsm_analysis(handle=%p, transA=%d, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, transA, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrsm_analysis(handle_native, transA_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrsm_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject F, jint ldf, jobject X, jint ldx)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseScsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (F == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'F' is null for cusparseScsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldf is primitive
    if (X == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'X' is null for cusparseScsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldx is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrsm_solve(handle=%p, transA=%d, m=%d, n=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, F=%p, ldf=%d, X=%p, ldx=%d)\n",
        handle, transA, m, n, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, F, ldf, X, ldx);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;
    float* F_native;
    int ldf_native = 0;
    float* X_native;
    int ldx_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    F_native = (float*)getPointer(env, F);
    ldf_native = (int)ldf;
    X_native = (float*)getPointer(env, X);
    ldx_native = (int)ldx;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrsm_solve(handle_native, transA_native, m_native, n_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, F_native, ldf_native, X_native, ldx_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // F is a native pointer
    // ldf is primitive
    // X is a native pointer
    // ldx is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrsm_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject F, jint ldf, jobject X, jint ldx)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (F == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'F' is null for cusparseDcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldf is primitive
    if (X == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'X' is null for cusparseDcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldx is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrsm_solve(handle=%p, transA=%d, m=%d, n=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, F=%p, ldf=%d, X=%p, ldx=%d)\n",
        handle, transA, m, n, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, F, ldf, X, ldx);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;
    double* F_native;
    int ldf_native = 0;
    double* X_native;
    int ldx_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    F_native = (double*)getPointer(env, F);
    ldf_native = (int)ldf;
    X_native = (double*)getPointer(env, X);
    ldx_native = (int)ldx;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrsm_solve(handle_native, transA_native, m_native, n_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, F_native, ldf_native, X_native, ldx_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // F is a native pointer
    // ldf is primitive
    // X is a native pointer
    // ldx is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrsm_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject F, jint ldf, jobject X, jint ldx)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (F == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'F' is null for cusparseCcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldf is primitive
    if (X == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'X' is null for cusparseCcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldx is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrsm_solve(handle=%p, transA=%d, m=%d, n=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, F=%p, ldf=%d, X=%p, ldx=%d)\n",
        handle, transA, m, n, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, F, ldf, X, ldx);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;
    cuComplex* F_native;
    int ldf_native = 0;
    cuComplex* X_native;
    int ldx_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    F_native = (cuComplex*)getPointer(env, F);
    ldf_native = (int)ldf;
    X_native = (cuComplex*)getPointer(env, X);
    ldx_native = (int)ldx;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrsm_solve(handle_native, transA_native, m_native, n_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, F_native, ldf_native, X_native, ldx_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // F is a native pointer
    // ldf is primitive
    // X is a native pointer
    // ldx is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrsm_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint m, jint n, jobject alpha, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject F, jint ldf, jobject X, jint ldx)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // m is primitive
    // n is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (F == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'F' is null for cusparseZcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldf is primitive
    if (X == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'X' is null for cusparseZcsrsm_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldx is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrsm_solve(handle=%p, transA=%d, m=%d, n=%d, alpha=%p, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, F=%p, ldf=%d, X=%p, ldx=%d)\n",
        handle, transA, m, n, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, F, ldf, X, ldx);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;
    cuDoubleComplex* F_native;
    int ldf_native = 0;
    cuDoubleComplex* X_native;
    int ldx_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    m_native = (int)m;
    n_native = (int)n;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);
    F_native = (cuDoubleComplex*)getPointer(env, F);
    ldf_native = (int)ldf;
    X_native = (cuDoubleComplex*)getPointer(env, X);
    ldx_native = (int)ldx;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrsm_solve(handle_native, transA_native, m_native, n_native, alpha_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, F_native, ldf_native, X_native, ldx_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // m is primitive
    // n is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // F is a native pointer
    // ldf is primitive
    // X is a native pointer
    // ldx is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
* <pre>
* Description: Solution of triangular linear system op(A) * X = alpha * F,
with multiple right-hand-sides, where A is a sparse matrix in CSR storage
format, rhs F and solution X are dense tall matrices.
* This routine implements algorithm 2 for this problem.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXbsrsm2_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXbsrsm2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXbsrsm2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXbsrsm2_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXbsrsm2_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrsm2Info_t info_native;
    int* position_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    position_native = (int*)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXbsrsm2_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // position is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsm2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseSbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsm2_bufferSize(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsm2_bufferSize(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsm2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsm2_bufferSize(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsm2_bufferSize(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsm2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsm2_bufferSize(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsm2_bufferSize(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsm2_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZbsrsm2_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsm2_bufferSize(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsm2_bufferSize(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsm2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transB, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseSbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsm2_bufferSizeExt(handle=%p, dirA=%d, transA=%d, transB=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, transA, transB, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsm2_bufferSizeExt(handle_native, dirA_native, transA_native, transB_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsm2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transB, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseDbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsm2_bufferSizeExt(handle=%p, dirA=%d, transA=%d, transB=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, transA, transB, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsm2_bufferSizeExt(handle_native, dirA_native, transA_native, transB_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsm2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transB, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseCbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsm2_bufferSizeExt(handle=%p, dirA=%d, transA=%d, transB=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, transA, transB, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsm2_bufferSizeExt(handle_native, dirA_native, transA_native, transB_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsm2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transB, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseZbsrsm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsm2_bufferSizeExt(handle=%p, dirA=%d, transA=%d, transB=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, transA, transB, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsm2_bufferSizeExt(handle_native, dirA_native, transA_native, transB_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transB is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsm2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsm2_analysis(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsm2_analysis(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsm2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsm2_analysis(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsm2_analysis(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsm2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsm2_analysis(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsm2_analysis(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsm2_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsrsm2_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsm2_analysis(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsm2_analysis(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrsm2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jobject F, jint ldf, jobject X, jint ldx, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (F == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'F' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldf is primitive
    if (X == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'X' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldx is primitive
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrsm2_solve(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, F=%p, ldf=%d, X=%p, ldx=%d, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, alpha, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, F, ldf, X, ldx, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    float* F_native;
    int ldf_native = 0;
    float* X_native;
    int ldx_native = 0;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    F_native = (float*)getPointer(env, F);
    ldf_native = (int)ldf;
    X_native = (float*)getPointer(env, X);
    ldx_native = (int)ldx;
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrsm2_solve(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, alpha_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, F_native, ldf_native, X_native, ldx_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // F is a native pointer
    // ldf is primitive
    // X is a native pointer
    // ldx is primitive
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrsm2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jobject F, jint ldf, jobject X, jint ldx, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (F == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'F' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldf is primitive
    if (X == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'X' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldx is primitive
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrsm2_solve(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, F=%p, ldf=%d, X=%p, ldx=%d, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, alpha, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, F, ldf, X, ldx, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    double* F_native;
    int ldf_native = 0;
    double* X_native;
    int ldx_native = 0;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    F_native = (double*)getPointer(env, F);
    ldf_native = (int)ldf;
    X_native = (double*)getPointer(env, X);
    ldx_native = (int)ldx;
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrsm2_solve(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, alpha_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, F_native, ldf_native, X_native, ldx_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // F is a native pointer
    // ldf is primitive
    // X is a native pointer
    // ldx is primitive
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrsm2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jobject F, jint ldf, jobject X, jint ldx, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (F == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'F' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldf is primitive
    if (X == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'X' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldx is primitive
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrsm2_solve(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, F=%p, ldf=%d, X=%p, ldx=%d, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, alpha, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, F, ldf, X, ldx, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    cuComplex* F_native;
    int ldf_native = 0;
    cuComplex* X_native;
    int ldx_native = 0;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    F_native = (cuComplex*)getPointer(env, F);
    ldf_native = (int)ldf;
    X_native = (cuComplex*)getPointer(env, X);
    ldx_native = (int)ldx;
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrsm2_solve(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, alpha_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, F_native, ldf_native, X_native, ldx_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // F is a native pointer
    // ldf is primitive
    // X is a native pointer
    // ldx is primitive
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrsm2_1solveNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint transA, jint transXY, jint mb, jint n, jint nnzb, jobject alpha, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jobject F, jint ldf, jobject X, jint ldx, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (F == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'F' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldf is primitive
    if (X == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'X' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldx is primitive
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsrsm2_solve");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrsm2_solve(handle=%p, dirA=%d, transA=%d, transXY=%d, mb=%d, n=%d, nnzb=%d, alpha=%p, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, F=%p, ldf=%d, X=%p, ldx=%d, policy=%d, pBuffer=%p)\n",
        handle, dirA, transA, transXY, mb, n, nnzb, alpha, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, F, ldf, X, ldx, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transXY_native;
    int mb_native = 0;
    int n_native = 0;
    int nnzb_native = 0;
    cuDoubleComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrsm2Info_t info_native;
    cuDoubleComplex* F_native;
    int ldf_native = 0;
    cuDoubleComplex* X_native;
    int ldx_native = 0;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    transA_native = (cusparseOperation_t)transA;
    transXY_native = (cusparseOperation_t)transXY;
    mb_native = (int)mb;
    n_native = (int)n;
    nnzb_native = (int)nnzb;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuDoubleComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrsm2Info_t)getNativePointerValue(env, info);
    F_native = (cuDoubleComplex*)getPointer(env, F);
    ldf_native = (int)ldf;
    X_native = (cuDoubleComplex*)getPointer(env, X);
    ldx_native = (int)ldx;
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrsm2_solve(handle_native, dirA_native, transA_native, transXY_native, mb_native, n_native, nnzb_native, alpha_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, F_native, ldf_native, X_native, ldx_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // transA is primitive
    // transXY is primitive
    // mb is primitive
    // n is primitive
    // nnzb is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // F is a native pointer
    // ldf is primitive
    // X is a native pointer
    // ldx is primitive
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** --- Preconditioners --- */
/**
* <pre>
* Description: Compute the incomplete-LU factorization with 0 fill-in (ILU0)
of the matrix A stored in CSR format based on the information in the opaque
structure info that was obtained from the analysis phase (csrsv_analysis).
* This routine implements algorithm 1 for this problem.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrilu0Native(JNIEnv *env, jclass cls, jobject handle, jint trans, jint m, jobject descrA, jobject csrSortedValA_ValM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    // m is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_ValM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_ValM' is null for cusparseScsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrilu0(handle=%p, trans=%d, m=%d, descrA=%p, csrSortedValA_ValM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, trans, m, descrA, csrSortedValA_ValM, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t trans_native;
    int m_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_ValM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    trans_native = (cusparseOperation_t)trans;
    m_native = (int)m;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_ValM_native = (float*)getPointer(env, csrSortedValA_ValM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrilu0(handle_native, trans_native, m_native, descrA_native, csrSortedValA_ValM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // trans is primitive
    // m is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_ValM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrilu0Native(JNIEnv *env, jclass cls, jobject handle, jint trans, jint m, jobject descrA, jobject csrSortedValA_ValM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    // m is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_ValM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_ValM' is null for cusparseDcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrilu0(handle=%p, trans=%d, m=%d, descrA=%p, csrSortedValA_ValM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, trans, m, descrA, csrSortedValA_ValM, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t trans_native;
    int m_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_ValM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    trans_native = (cusparseOperation_t)trans;
    m_native = (int)m;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_ValM_native = (double*)getPointer(env, csrSortedValA_ValM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrilu0(handle_native, trans_native, m_native, descrA_native, csrSortedValA_ValM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // trans is primitive
    // m is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_ValM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrilu0Native(JNIEnv *env, jclass cls, jobject handle, jint trans, jint m, jobject descrA, jobject csrSortedValA_ValM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    // m is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_ValM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_ValM' is null for cusparseCcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrilu0(handle=%p, trans=%d, m=%d, descrA=%p, csrSortedValA_ValM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, trans, m, descrA, csrSortedValA_ValM, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t trans_native;
    int m_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_ValM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    trans_native = (cusparseOperation_t)trans;
    m_native = (int)m;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_ValM_native = (cuComplex*)getPointer(env, csrSortedValA_ValM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrilu0(handle_native, trans_native, m_native, descrA_native, csrSortedValA_ValM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // trans is primitive
    // m is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_ValM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrilu0Native(JNIEnv *env, jclass cls, jobject handle, jint trans, jint m, jobject descrA, jobject csrSortedValA_ValM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    // m is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_ValM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_ValM' is null for cusparseZcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrilu0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrilu0(handle=%p, trans=%d, m=%d, descrA=%p, csrSortedValA_ValM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, trans, m, descrA, csrSortedValA_ValM, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t trans_native;
    int m_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_ValM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    trans_native = (cusparseOperation_t)trans;
    m_native = (int)m;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_ValM_native = (cuDoubleComplex*)getPointer(env, csrSortedValA_ValM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrilu0(handle_native, trans_native, m_native, descrA_native, csrSortedValA_ValM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // trans is primitive
    // m is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_ValM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
* <pre>
* Description: Compute the incomplete-LU factorization with 0 fill-in (ILU0)
of the matrix A stored in CSR format based on the information in the opaque
structure info that was obtained from the analysis phase (csrsv2_analysis).
* This routine implements algorithm 2 for this problem.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseScsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseScsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csrilu02Info_t info_native;
    int enable_boost_native = 0;
    double* tol_native;
    float* boost_val_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    tol_native = (double*)getPointer(env, tol);
    boost_val_native = (float*)getPointer(env, boost_val);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // enable_boost is primitive
    // tol is a native pointer
    // boost_val is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseDcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseDcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csrilu02Info_t info_native;
    int enable_boost_native = 0;
    double* tol_native;
    double* boost_val_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    tol_native = (double*)getPointer(env, tol);
    boost_val_native = (double*)getPointer(env, boost_val);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // enable_boost is primitive
    // tol is a native pointer
    // boost_val is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseCcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseCcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csrilu02Info_t info_native;
    int enable_boost_native = 0;
    double* tol_native;
    cuComplex* boost_val_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    tol_native = (double*)getPointer(env, tol);
    boost_val_native = (cuComplex*)getPointer(env, boost_val);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // enable_boost is primitive
    // tol is a native pointer
    // boost_val is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseZcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseZcsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csrilu02Info_t info_native;
    int enable_boost_native = 0;
    double* tol_native;
    cuDoubleComplex* boost_val_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    tol_native = (double*)getPointer(env, tol);
    boost_val_native = (cuDoubleComplex*)getPointer(env, boost_val);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // enable_boost is primitive
    // tol is a native pointer
    // boost_val is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcsrilu02_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcsrilu02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXcsrilu02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXcsrilu02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcsrilu02_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csrilu02Info_t info_native;
    int* position_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    position_native = (int*)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcsrilu02_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // position is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseScsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrilu02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrilu02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrilu02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrilu02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrilu02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrilu02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrilu02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrilu02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrilu02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZcsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrilu02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrilu02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrilu02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrilu02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedVal, jobject csrSortedRowPtr, jobject csrSortedColInd, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedVal' is null for cusparseScsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtr' is null for cusparseScsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColInd' is null for cusparseScsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseScsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrilu02_bufferSizeExt(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedVal=%p, csrSortedRowPtr=%p, csrSortedColInd=%p, info=%p, pBufferSize=%p)\n",
        handle, m, nnz, descrA, csrSortedVal, csrSortedRowPtr, csrSortedColInd, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedVal_native;
    int* csrSortedRowPtr_native;
    int* csrSortedColInd_native;
    csrilu02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedVal_native = (float*)getPointer(env, csrSortedVal);
    csrSortedRowPtr_native = (int*)getPointer(env, csrSortedRowPtr);
    csrSortedColInd_native = (int*)getPointer(env, csrSortedColInd);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrilu02_bufferSizeExt(handle_native, m_native, nnz_native, descrA_native, csrSortedVal_native, csrSortedRowPtr_native, csrSortedColInd_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedVal is a native pointer
    // csrSortedRowPtr is a native pointer
    // csrSortedColInd is a native pointer
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrilu02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedVal, jobject csrSortedRowPtr, jobject csrSortedColInd, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedVal' is null for cusparseDcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtr' is null for cusparseDcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColInd' is null for cusparseDcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseDcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrilu02_bufferSizeExt(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedVal=%p, csrSortedRowPtr=%p, csrSortedColInd=%p, info=%p, pBufferSize=%p)\n",
        handle, m, nnz, descrA, csrSortedVal, csrSortedRowPtr, csrSortedColInd, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedVal_native;
    int* csrSortedRowPtr_native;
    int* csrSortedColInd_native;
    csrilu02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedVal_native = (double*)getPointer(env, csrSortedVal);
    csrSortedRowPtr_native = (int*)getPointer(env, csrSortedRowPtr);
    csrSortedColInd_native = (int*)getPointer(env, csrSortedColInd);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrilu02_bufferSizeExt(handle_native, m_native, nnz_native, descrA_native, csrSortedVal_native, csrSortedRowPtr_native, csrSortedColInd_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedVal is a native pointer
    // csrSortedRowPtr is a native pointer
    // csrSortedColInd is a native pointer
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrilu02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedVal, jobject csrSortedRowPtr, jobject csrSortedColInd, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedVal' is null for cusparseCcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtr' is null for cusparseCcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColInd' is null for cusparseCcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseCcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrilu02_bufferSizeExt(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedVal=%p, csrSortedRowPtr=%p, csrSortedColInd=%p, info=%p, pBufferSize=%p)\n",
        handle, m, nnz, descrA, csrSortedVal, csrSortedRowPtr, csrSortedColInd, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedVal_native;
    int* csrSortedRowPtr_native;
    int* csrSortedColInd_native;
    csrilu02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedVal_native = (cuComplex*)getPointer(env, csrSortedVal);
    csrSortedRowPtr_native = (int*)getPointer(env, csrSortedRowPtr);
    csrSortedColInd_native = (int*)getPointer(env, csrSortedColInd);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrilu02_bufferSizeExt(handle_native, m_native, nnz_native, descrA_native, csrSortedVal_native, csrSortedRowPtr_native, csrSortedColInd_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedVal is a native pointer
    // csrSortedRowPtr is a native pointer
    // csrSortedColInd is a native pointer
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrilu02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedVal, jobject csrSortedRowPtr, jobject csrSortedColInd, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedVal' is null for cusparseZcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtr' is null for cusparseZcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColInd' is null for cusparseZcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseZcsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrilu02_bufferSizeExt(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedVal=%p, csrSortedRowPtr=%p, csrSortedColInd=%p, info=%p, pBufferSize=%p)\n",
        handle, m, nnz, descrA, csrSortedVal, csrSortedRowPtr, csrSortedColInd, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedVal_native;
    int* csrSortedRowPtr_native;
    int* csrSortedColInd_native;
    csrilu02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedVal_native = (cuDoubleComplex*)getPointer(env, csrSortedVal);
    csrSortedRowPtr_native = (int*)getPointer(env, csrSortedRowPtr);
    csrSortedColInd_native = (int*)getPointer(env, csrSortedColInd);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrilu02_bufferSizeExt(handle_native, m_native, nnz_native, descrA_native, csrSortedVal_native, csrSortedRowPtr_native, csrSortedColInd_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedVal is a native pointer
    // csrSortedRowPtr is a native pointer
    // csrSortedColInd is a native pointer
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseScsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrilu02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrilu02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrilu02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrilu02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrilu02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrilu02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZcsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrilu02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrilu02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseScsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrilu02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_valM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (float*)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrilu02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrilu02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_valM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (double*)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrilu02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrilu02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_valM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (cuComplex*)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrilu02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZcsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrilu02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_valM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (cuDoubleComplex*)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrilu02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
* <pre>
* Description: Compute the incomplete-LU factorization with 0 fill-in (ILU0)
of the matrix A stored in block-CSR format based on the information in the opaque
structure info that was obtained from the analysis phase (bsrsv2_analysis).
* This routine implements algorithm 2 for this problem.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseSbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseSbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrilu02Info_t info_native;
    int enable_boost_native = 0;
    double* tol_native;
    float* boost_val_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    tol_native = (double*)getPointer(env, tol);
    boost_val_native = (float*)getPointer(env, boost_val);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // enable_boost is primitive
    // tol is a native pointer
    // boost_val is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseDbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseDbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrilu02Info_t info_native;
    int enable_boost_native = 0;
    double* tol_native;
    double* boost_val_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    tol_native = (double*)getPointer(env, tol);
    boost_val_native = (double*)getPointer(env, boost_val);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // enable_boost is primitive
    // tol is a native pointer
    // boost_val is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseCbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseCbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrilu02Info_t info_native;
    int enable_boost_native = 0;
    double* tol_native;
    cuComplex* boost_val_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    tol_native = (double*)getPointer(env, tol);
    boost_val_native = (cuComplex*)getPointer(env, boost_val);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // enable_boost is primitive
    // tol is a native pointer
    // boost_val is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrilu02_1numericBoostNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jint enable_boost, jobject tol, jobject boost_val)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // enable_boost is primitive
    if (tol == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'tol' is null for cusparseZbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (boost_val == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'boost_val' is null for cusparseZbsrilu02_numericBoost");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrilu02_numericBoost(handle=%p, info=%p, enable_boost=%d, tol=%p, boost_val=%p)\n",
        handle, info, enable_boost, tol, boost_val);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrilu02Info_t info_native;
    int enable_boost_native = 0;
    double* tol_native;
    cuDoubleComplex* boost_val_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    enable_boost_native = (int)enable_boost;
    tol_native = (double*)getPointer(env, tol);
    boost_val_native = (cuDoubleComplex*)getPointer(env, boost_val);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrilu02_numericBoost(handle_native, info_native, enable_boost_native, tol_native, boost_val_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // enable_boost is primitive
    // tol is a native pointer
    // boost_val is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXbsrilu02_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXbsrilu02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXbsrilu02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXbsrilu02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXbsrilu02_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsrilu02Info_t info_native;
    int* position_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    position_native = (int*)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXbsrilu02_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // position is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseSbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrilu02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrilu02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrilu02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrilu02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrilu02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrilu02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrilu02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZbsrilu02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrilu02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrilu02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrilu02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseSbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrilu02_bufferSizeExt(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrilu02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrilu02_bufferSizeExt(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrilu02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseDbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrilu02_bufferSizeExt(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrilu02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrilu02_bufferSizeExt(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrilu02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseCbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrilu02_bufferSizeExt(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrilu02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrilu02_bufferSizeExt(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrilu02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseZbsrilu02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrilu02_bufferSizeExt(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsrilu02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrilu02_bufferSizeExt(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrilu02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrilu02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrilu02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrilu02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrilu02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrilu02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrilu02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsrilu02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrilu02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrilu02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descra, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descra == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descra' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsrilu02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descra=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descra, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descra_native;
    float* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descra_native = (cusparseMatDescr_t)getNativePointerValue(env, descra);
    bsrSortedVal_native = (float*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsrilu02(handle_native, dirA_native, mb_native, nnzb_native, descra_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descra is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descra, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descra == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descra' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsrilu02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descra=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descra, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descra_native;
    double* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descra_native = (cusparseMatDescr_t)getNativePointerValue(env, descra);
    bsrSortedVal_native = (double*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsrilu02(handle_native, dirA_native, mb_native, nnzb_native, descra_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descra is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descra, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descra == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descra' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsrilu02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descra=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descra, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descra_native;
    cuComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descra_native = (cusparseMatDescr_t)getNativePointerValue(env, descra);
    bsrSortedVal_native = (cuComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsrilu02(handle_native, dirA_native, mb_native, nnzb_native, descra_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descra is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsrilu02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descra, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descra == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descra' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsrilu02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsrilu02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descra=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descra, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descra_native;
    cuDoubleComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsrilu02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descra_native = (cusparseMatDescr_t)getNativePointerValue(env, descra);
    bsrSortedVal_native = (cuDoubleComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsrilu02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsrilu02(handle_native, dirA_native, mb_native, nnzb_native, descra_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descra is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
* <pre>
* Description: Compute the incomplete-Cholesky factorization with 0 fill-in (IC0)
of the matrix A stored in CSR format based on the information in the opaque
structure info that was obtained from the analysis phase (csrsv_analysis).
* This routine implements algorithm 1 for this problem.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsric0Native(JNIEnv *env, jclass cls, jobject handle, jint trans, jint m, jobject descrA, jobject csrSortedValA_ValM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    // m is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_ValM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_ValM' is null for cusparseScsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsric0(handle=%p, trans=%d, m=%d, descrA=%p, csrSortedValA_ValM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, trans, m, descrA, csrSortedValA_ValM, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t trans_native;
    int m_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_ValM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    trans_native = (cusparseOperation_t)trans;
    m_native = (int)m;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_ValM_native = (float*)getPointer(env, csrSortedValA_ValM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsric0(handle_native, trans_native, m_native, descrA_native, csrSortedValA_ValM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // trans is primitive
    // m is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_ValM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsric0Native(JNIEnv *env, jclass cls, jobject handle, jint trans, jint m, jobject descrA, jobject csrSortedValA_ValM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    // m is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_ValM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_ValM' is null for cusparseDcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsric0(handle=%p, trans=%d, m=%d, descrA=%p, csrSortedValA_ValM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, trans, m, descrA, csrSortedValA_ValM, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t trans_native;
    int m_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_ValM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    trans_native = (cusparseOperation_t)trans;
    m_native = (int)m;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_ValM_native = (double*)getPointer(env, csrSortedValA_ValM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsric0(handle_native, trans_native, m_native, descrA_native, csrSortedValA_ValM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // trans is primitive
    // m is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_ValM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsric0Native(JNIEnv *env, jclass cls, jobject handle, jint trans, jint m, jobject descrA, jobject csrSortedValA_ValM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    // m is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_ValM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_ValM' is null for cusparseCcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsric0(handle=%p, trans=%d, m=%d, descrA=%p, csrSortedValA_ValM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, trans, m, descrA, csrSortedValA_ValM, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t trans_native;
    int m_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_ValM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    trans_native = (cusparseOperation_t)trans;
    m_native = (int)m;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_ValM_native = (cuComplex*)getPointer(env, csrSortedValA_ValM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsric0(handle_native, trans_native, m_native, descrA_native, csrSortedValA_ValM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // trans is primitive
    // m is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_ValM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsric0Native(JNIEnv *env, jclass cls, jobject handle, jint trans, jint m, jobject descrA, jobject csrSortedValA_ValM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // trans is primitive
    // m is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_ValM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_ValM' is null for cusparseZcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsric0");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsric0(handle=%p, trans=%d, m=%d, descrA=%p, csrSortedValA_ValM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p)\n",
        handle, trans, m, descrA, csrSortedValA_ValM, csrSortedRowPtrA, csrSortedColIndA, info);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t trans_native;
    int m_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_ValM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseSolveAnalysisInfo_t info_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    trans_native = (cusparseOperation_t)trans;
    m_native = (int)m;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_ValM_native = (cuDoubleComplex*)getPointer(env, csrSortedValA_ValM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (cusparseSolveAnalysisInfo_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsric0(handle_native, trans_native, m_native, descrA_native, csrSortedValA_ValM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // trans is primitive
    // m is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_ValM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
* <pre>
* Description: Compute the incomplete-Cholesky factorization with 0 fill-in (IC0)
of the matrix A stored in CSR format based on the information in the opaque
structure info that was obtained from the analysis phase (csrsv2_analysis).
* This routine implements algorithm 2 for this problem.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcsric02_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcsric02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXcsric02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXcsric02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcsric02_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    csric02Info_t info_native;
    int* position_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    position_native = (int*)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcsric02_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // position is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseScsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsric02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csric02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsric02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsric02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csric02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsric02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsric02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csric02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsric02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZcsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsric02_bufferSize(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csric02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsric02_bufferSize(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsric02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedVal, jobject csrSortedRowPtr, jobject csrSortedColInd, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedVal' is null for cusparseScsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtr' is null for cusparseScsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColInd' is null for cusparseScsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseScsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsric02_bufferSizeExt(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedVal=%p, csrSortedRowPtr=%p, csrSortedColInd=%p, info=%p, pBufferSize=%p)\n",
        handle, m, nnz, descrA, csrSortedVal, csrSortedRowPtr, csrSortedColInd, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedVal_native;
    int* csrSortedRowPtr_native;
    int* csrSortedColInd_native;
    csric02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedVal_native = (float*)getPointer(env, csrSortedVal);
    csrSortedRowPtr_native = (int*)getPointer(env, csrSortedRowPtr);
    csrSortedColInd_native = (int*)getPointer(env, csrSortedColInd);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsric02_bufferSizeExt(handle_native, m_native, nnz_native, descrA_native, csrSortedVal_native, csrSortedRowPtr_native, csrSortedColInd_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedVal is a native pointer
    // csrSortedRowPtr is a native pointer
    // csrSortedColInd is a native pointer
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsric02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedVal, jobject csrSortedRowPtr, jobject csrSortedColInd, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedVal' is null for cusparseDcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtr' is null for cusparseDcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColInd' is null for cusparseDcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseDcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsric02_bufferSizeExt(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedVal=%p, csrSortedRowPtr=%p, csrSortedColInd=%p, info=%p, pBufferSize=%p)\n",
        handle, m, nnz, descrA, csrSortedVal, csrSortedRowPtr, csrSortedColInd, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedVal_native;
    int* csrSortedRowPtr_native;
    int* csrSortedColInd_native;
    csric02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedVal_native = (double*)getPointer(env, csrSortedVal);
    csrSortedRowPtr_native = (int*)getPointer(env, csrSortedRowPtr);
    csrSortedColInd_native = (int*)getPointer(env, csrSortedColInd);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsric02_bufferSizeExt(handle_native, m_native, nnz_native, descrA_native, csrSortedVal_native, csrSortedRowPtr_native, csrSortedColInd_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedVal is a native pointer
    // csrSortedRowPtr is a native pointer
    // csrSortedColInd is a native pointer
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsric02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedVal, jobject csrSortedRowPtr, jobject csrSortedColInd, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedVal' is null for cusparseCcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtr' is null for cusparseCcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColInd' is null for cusparseCcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseCcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsric02_bufferSizeExt(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedVal=%p, csrSortedRowPtr=%p, csrSortedColInd=%p, info=%p, pBufferSize=%p)\n",
        handle, m, nnz, descrA, csrSortedVal, csrSortedRowPtr, csrSortedColInd, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedVal_native;
    int* csrSortedRowPtr_native;
    int* csrSortedColInd_native;
    csric02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedVal_native = (cuComplex*)getPointer(env, csrSortedVal);
    csrSortedRowPtr_native = (int*)getPointer(env, csrSortedRowPtr);
    csrSortedColInd_native = (int*)getPointer(env, csrSortedColInd);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsric02_bufferSizeExt(handle_native, m_native, nnz_native, descrA_native, csrSortedVal_native, csrSortedRowPtr_native, csrSortedColInd_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedVal is a native pointer
    // csrSortedRowPtr is a native pointer
    // csrSortedColInd is a native pointer
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsric02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedVal, jobject csrSortedRowPtr, jobject csrSortedColInd, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedVal' is null for cusparseZcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtr' is null for cusparseZcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColInd' is null for cusparseZcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseZcsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsric02_bufferSizeExt(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedVal=%p, csrSortedRowPtr=%p, csrSortedColInd=%p, info=%p, pBufferSize=%p)\n",
        handle, m, nnz, descrA, csrSortedVal, csrSortedRowPtr, csrSortedColInd, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedVal_native;
    int* csrSortedRowPtr_native;
    int* csrSortedColInd_native;
    csric02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedVal_native = (cuDoubleComplex*)getPointer(env, csrSortedVal);
    csrSortedRowPtr_native = (int*)getPointer(env, csrSortedRowPtr);
    csrSortedColInd_native = (int*)getPointer(env, csrSortedColInd);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsric02_bufferSizeExt(handle_native, m_native, nnz_native, descrA_native, csrSortedVal_native, csrSortedRowPtr_native, csrSortedColInd_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedVal is a native pointer
    // csrSortedRowPtr is a native pointer
    // csrSortedColInd is a native pointer
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseScsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsric02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsric02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsric02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsric02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsric02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsric02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZcsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsric02_analysis(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsric02_analysis(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsric02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseScsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsric02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    float* csrSortedValA_valM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (float*)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsric02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsric02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsric02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    double* csrSortedValA_valM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (double*)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsric02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsric02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsric02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* csrSortedValA_valM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (cuComplex*)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsric02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsric02Native(JNIEnv *env, jclass cls, jobject handle, jint m, jint nnz, jobject descrA, jobject csrSortedValA_valM, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // nnz is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValA_valM == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA_valM' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZcsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsric02(handle=%p, m=%d, nnz=%d, descrA=%p, csrSortedValA_valM=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, info=%p, policy=%d, pBuffer=%p)\n",
        handle, m, nnz, descrA, csrSortedValA_valM, csrSortedRowPtrA, csrSortedColIndA, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int nnz_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* csrSortedValA_valM_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    csric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    nnz_native = (int)nnz;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    csrSortedValA_valM_native = (cuDoubleComplex*)getPointer(env, csrSortedValA_valM);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    info_native = (csric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsric02(handle_native, m_native, nnz_native, descrA_native, csrSortedValA_valM_native, csrSortedRowPtrA_native, csrSortedColIndA_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // nnz is primitive
    // descrA is a read-only native pointer
    // csrSortedValA_valM is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
* <pre>
* Description: Compute the incomplete-Cholesky factorization with 0 fill-in (IC0)
of the matrix A stored in block-CSR format based on the information in the opaque
structure info that was obtained from the analysis phase (bsrsv2_analysis).
* This routine implements algorithm 1 for this problem.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXbsric02_1zeroPivotNative(JNIEnv *env, jclass cls, jobject handle, jobject info, jobject position)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXbsric02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseXbsric02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (position == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'position' is null for cusparseXbsric02_zeroPivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXbsric02_zeroPivot(handle=%p, info=%p, position=%p)\n",
        handle, info, position);

    // Native variable declarations
    cusparseHandle_t handle_native;
    bsric02Info_t info_native;
    int* position_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    position_native = (int*)getPointer(env, position);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXbsric02_zeroPivot(handle_native, info_native, position_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // info is a read-only native pointer
    // position is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseSbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsric02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsric02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsric02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsric02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsric02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsric02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsric02_1bufferSizeNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jobject pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseZbsric02_bufferSize");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsric02_bufferSize(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, pBufferSizeInBytes=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    int* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (int*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsric02_bufferSize(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsric02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseSbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsric02_bufferSizeExt(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsric02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsric02_bufferSizeExt(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsric02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseDbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsric02_bufferSizeExt(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsric02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsric02_bufferSizeExt(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsric02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseCbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsric02_bufferSizeExt(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsric02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsric02_bufferSizeExt(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsric02_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockSize, jobject info, jlongArray pBufferSize)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockSize is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSize == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSize' is null for cusparseZbsric02_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsric02_bufferSizeExt(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockSize=%d, info=%p, pBufferSize=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockSize, info, pBufferSize);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockSize_native = 0;
    bsric02Info_t info_native;
    size_t* pBufferSize_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockSize_native = (int)blockSize;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    pBufferSize_native = (size_t*)getPointer(env, pBufferSize);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsric02_bufferSizeExt(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockSize_native, info_native, pBufferSize_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockSize is primitive
    // info is a read-only native pointer
    // pBufferSize is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pInputBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pInputBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pInputBuffer' is null for cusparseSbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsric02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pInputBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pInputBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pInputBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pInputBuffer_native = (void*)getPointer(env, pInputBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsric02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pInputBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pInputBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pInputBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pInputBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pInputBuffer' is null for cusparseDbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsric02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pInputBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pInputBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pInputBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pInputBuffer_native = (void*)getPointer(env, pInputBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsric02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pInputBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pInputBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pInputBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pInputBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pInputBuffer' is null for cusparseCbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsric02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pInputBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pInputBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pInputBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pInputBuffer_native = (void*)getPointer(env, pInputBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsric02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pInputBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pInputBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsric02_1analysisNative(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pInputBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pInputBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pInputBuffer' is null for cusparseZbsric02_analysis");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsric02_analysis(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pInputBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pInputBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pInputBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pInputBuffer_native = (void*)getPointer(env, pInputBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsric02_analysis(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pInputBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pInputBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSbsric02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseSbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSbsric02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    float* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (float*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSbsric02(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDbsric02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseDbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDbsric02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    double* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (double*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDbsric02(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCbsric02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseCbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCbsric02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCbsric02(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZbsric02Native(JNIEnv *env, jclass cls, jobject handle, jint dirA, jint mb, jint nnzb, jobject descrA, jobject bsrSortedVal, jobject bsrSortedRowPtr, jobject bsrSortedColInd, jint blockDim, jobject info, jint policy, jobject pBuffer)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedVal == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedVal' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedRowPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedRowPtr' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (bsrSortedColInd == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'bsrSortedColInd' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // blockDim is primitive
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // policy is primitive
    if (pBuffer == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBuffer' is null for cusparseZbsric02");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZbsric02(handle=%p, dirA=%d, mb=%d, nnzb=%d, descrA=%p, bsrSortedVal=%p, bsrSortedRowPtr=%p, bsrSortedColInd=%p, blockDim=%d, info=%p, policy=%d, pBuffer=%p)\n",
        handle, dirA, mb, nnzb, descrA, bsrSortedVal, bsrSortedRowPtr, bsrSortedColInd, blockDim, info, policy, pBuffer);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseDirection_t dirA_native;
    int mb_native = 0;
    int nnzb_native = 0;
    cusparseMatDescr_t descrA_native;
    cuDoubleComplex* bsrSortedVal_native;
    int* bsrSortedRowPtr_native;
    int* bsrSortedColInd_native;
    int blockDim_native = 0;
    bsric02Info_t info_native;
    cusparseSolvePolicy_t policy_native;
    void* pBuffer_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    dirA_native = (cusparseDirection_t)dirA;
    mb_native = (int)mb;
    nnzb_native = (int)nnzb;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    bsrSortedVal_native = (cuDoubleComplex*)getPointer(env, bsrSortedVal);
    bsrSortedRowPtr_native = (int*)getPointer(env, bsrSortedRowPtr);
    bsrSortedColInd_native = (int*)getPointer(env, bsrSortedColInd);
    blockDim_native = (int)blockDim;
    info_native = (bsric02Info_t)getNativePointerValue(env, info);
    policy_native = (cusparseSolvePolicy_t)policy;
    pBuffer_native = (void*)getPointer(env, pBuffer);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZbsric02(handle_native, dirA_native, mb_native, nnzb_native, descrA_native, bsrSortedVal_native, bsrSortedRowPtr_native, bsrSortedColInd_native, blockDim_native, info_native, policy_native, pBuffer_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // dirA is primitive
    // mb is primitive
    // nnzb is primitive
    // descrA is a read-only native pointer
    // bsrSortedVal is a native pointer
    // bsrSortedRowPtr is a native pointer
    // bsrSortedColInd is a native pointer
    // blockDim is primitive
    // info is a read-only native pointer
    // policy is primitive
    // pBuffer is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
* <pre>
* Description: Solution of tridiagonal linear system A * X = F,
with multiple right-hand-sides. The coefficient matrix A is
composed of lower (dl), main (d) and upper (du) diagonals, and
the right-hand-sides F are overwritten with the solution X.
* These routine use pivoting.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgtsvNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseSgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgtsv(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d)\n",
        handle, m, n, dl, d, du, B, ldb);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float* dl_native;
    float* d_native;
    float* du_native;
    float* B_native;
    int ldb_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (float*)getPointer(env, dl);
    d_native = (float*)getPointer(env, d);
    du_native = (float*)getPointer(env, du);
    B_native = (float*)getPointer(env, B);
    ldb_native = (int)ldb;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgtsv(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgtsvNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseDgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgtsv(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d)\n",
        handle, m, n, dl, d, du, B, ldb);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double* dl_native;
    double* d_native;
    double* du_native;
    double* B_native;
    int ldb_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (double*)getPointer(env, dl);
    d_native = (double*)getPointer(env, d);
    du_native = (double*)getPointer(env, du);
    B_native = (double*)getPointer(env, B);
    ldb_native = (int)ldb;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgtsv(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgtsvNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseCgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgtsv(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d)\n",
        handle, m, n, dl, d, du, B, ldb);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex* dl_native;
    cuComplex* d_native;
    cuComplex* du_native;
    cuComplex* B_native;
    int ldb_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (cuComplex*)getPointer(env, dl);
    d_native = (cuComplex*)getPointer(env, d);
    du_native = (cuComplex*)getPointer(env, du);
    B_native = (cuComplex*)getPointer(env, B);
    ldb_native = (int)ldb;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgtsv(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgtsvNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseZgtsv");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgtsv(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d)\n",
        handle, m, n, dl, d, du, B, ldb);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex* dl_native;
    cuDoubleComplex* d_native;
    cuDoubleComplex* du_native;
    cuDoubleComplex* B_native;
    int ldb_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (cuDoubleComplex*)getPointer(env, dl);
    d_native = (cuDoubleComplex*)getPointer(env, d);
    du_native = (cuDoubleComplex*)getPointer(env, du);
    B_native = (cuDoubleComplex*)getPointer(env, B);
    ldb_native = (int)ldb;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgtsv(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
* <pre>
* Description: Solution of tridiagonal linear system A * X = F,
with multiple right-hand-sides. The coefficient matrix A is
composed of lower (dl), main (d) and upper (du) diagonals, and
the right-hand-sides F are overwritten with the solution X.
* These routine does not use pivoting.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgtsv_1nopivotNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseSgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgtsv_nopivot(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d)\n",
        handle, m, n, dl, d, du, B, ldb);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    float* dl_native;
    float* d_native;
    float* du_native;
    float* B_native;
    int ldb_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (float*)getPointer(env, dl);
    d_native = (float*)getPointer(env, d);
    du_native = (float*)getPointer(env, du);
    B_native = (float*)getPointer(env, B);
    ldb_native = (int)ldb;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgtsv_nopivot(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgtsv_1nopivotNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseDgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgtsv_nopivot(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d)\n",
        handle, m, n, dl, d, du, B, ldb);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    double* dl_native;
    double* d_native;
    double* du_native;
    double* B_native;
    int ldb_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (double*)getPointer(env, dl);
    d_native = (double*)getPointer(env, d);
    du_native = (double*)getPointer(env, du);
    B_native = (double*)getPointer(env, B);
    ldb_native = (int)ldb;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgtsv_nopivot(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgtsv_1nopivotNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseCgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgtsv_nopivot(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d)\n",
        handle, m, n, dl, d, du, B, ldb);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuComplex* dl_native;
    cuComplex* d_native;
    cuComplex* du_native;
    cuComplex* B_native;
    int ldb_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (cuComplex*)getPointer(env, dl);
    d_native = (cuComplex*)getPointer(env, d);
    du_native = (cuComplex*)getPointer(env, du);
    B_native = (cuComplex*)getPointer(env, B);
    ldb_native = (int)ldb;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgtsv_nopivot(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgtsv_1nopivotNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jobject dl, jobject d, jobject du, jobject B, jint ldb)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (B == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'B' is null for cusparseZgtsv_nopivot");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // ldb is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgtsv_nopivot(handle=%p, m=%d, n=%d, dl=%p, d=%p, du=%p, B=%p, ldb=%d)\n",
        handle, m, n, dl, d, du, B, ldb);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    cuDoubleComplex* dl_native;
    cuDoubleComplex* d_native;
    cuDoubleComplex* du_native;
    cuDoubleComplex* B_native;
    int ldb_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    dl_native = (cuDoubleComplex*)getPointer(env, dl);
    d_native = (cuDoubleComplex*)getPointer(env, d);
    du_native = (cuDoubleComplex*)getPointer(env, du);
    B_native = (cuDoubleComplex*)getPointer(env, B);
    ldb_native = (int)ldb;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgtsv_nopivot(handle_native, m_native, n_native, dl_native, d_native, du_native, B_native, ldb_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // n is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // B is a native pointer
    // ldb is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
* <pre>
* Description: Solution of a set of tridiagonal linear systems
A_{i} * x_{i} = f_{i} for i=1,...,batchCount. The coefficient
matrices A_{i} are composed of lower (dl), main (d) and upper (du)
diagonals and stored separated by a batchStride. Also, the
* right-hand-sides/solutions f_{i}/x_{i} are separated by a batchStride.
* </pre>
*/
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseSgtsvStridedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jint batchStride)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseSgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseSgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseSgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseSgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseSgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    // batchStride is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseSgtsvStridedBatch(handle=%p, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, batchStride=%d)\n",
        handle, m, dl, d, du, x, batchCount, batchStride);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    float* dl_native;
    float* d_native;
    float* du_native;
    float* x_native;
    int batchCount_native = 0;
    int batchStride_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    dl_native = (float*)getPointer(env, dl);
    d_native = (float*)getPointer(env, d);
    du_native = (float*)getPointer(env, du);
    x_native = (float*)getPointer(env, x);
    batchCount_native = (int)batchCount;
    batchStride_native = (int)batchStride;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseSgtsvStridedBatch(handle_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, batchStride_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // batchStride is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDgtsvStridedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jint batchStride)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseDgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseDgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseDgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseDgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    // batchStride is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDgtsvStridedBatch(handle=%p, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, batchStride=%d)\n",
        handle, m, dl, d, du, x, batchCount, batchStride);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    double* dl_native;
    double* d_native;
    double* du_native;
    double* x_native;
    int batchCount_native = 0;
    int batchStride_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    dl_native = (double*)getPointer(env, dl);
    d_native = (double*)getPointer(env, d);
    du_native = (double*)getPointer(env, du);
    x_native = (double*)getPointer(env, x);
    batchCount_native = (int)batchCount;
    batchStride_native = (int)batchStride;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDgtsvStridedBatch(handle_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, batchStride_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // batchStride is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCgtsvStridedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jint batchStride)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseCgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseCgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseCgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseCgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    // batchStride is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCgtsvStridedBatch(handle=%p, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, batchStride=%d)\n",
        handle, m, dl, d, du, x, batchCount, batchStride);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    cuComplex* dl_native;
    cuComplex* d_native;
    cuComplex* du_native;
    cuComplex* x_native;
    int batchCount_native = 0;
    int batchStride_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    dl_native = (cuComplex*)getPointer(env, dl);
    d_native = (cuComplex*)getPointer(env, d);
    du_native = (cuComplex*)getPointer(env, du);
    x_native = (cuComplex*)getPointer(env, x);
    batchCount_native = (int)batchCount;
    batchStride_native = (int)batchStride;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCgtsvStridedBatch(handle_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, batchStride_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // batchStride is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZgtsvStridedBatchNative(JNIEnv *env, jclass cls, jobject handle, jint m, jobject dl, jobject d, jobject du, jobject x, jint batchCount, jint batchStride)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    if (dl == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'dl' is null for cusparseZgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (d == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'd' is null for cusparseZgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (du == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'du' is null for cusparseZgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (x == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'x' is null for cusparseZgtsvStridedBatch");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // batchCount is primitive
    // batchStride is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZgtsvStridedBatch(handle=%p, m=%d, dl=%p, d=%p, du=%p, x=%p, batchCount=%d, batchStride=%d)\n",
        handle, m, dl, d, du, x, batchCount, batchStride);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    cuDoubleComplex* dl_native;
    cuDoubleComplex* d_native;
    cuDoubleComplex* du_native;
    cuDoubleComplex* x_native;
    int batchCount_native = 0;
    int batchStride_native = 0;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    dl_native = (cuDoubleComplex*)getPointer(env, dl);
    d_native = (cuDoubleComplex*)getPointer(env, d);
    du_native = (cuDoubleComplex*)getPointer(env, du);
    x_native = (cuDoubleComplex*)getPointer(env, x);
    batchCount_native = (int)batchCount;
    batchStride_native = (int)batchStride;

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZgtsvStridedBatch(handle_native, m_native, dl_native, d_native, du_native, x_native, batchCount_native, batchStride_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // dl is a native pointer
    // d is a native pointer
    // du is a native pointer
    // x is a native pointer
    // batchCount is primitive
    // batchStride is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** --- Sparse Level 4 routines --- */
/** Description: Compute sparse - sparse matrix multiplication for matrices
stored in CSR format. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseXcsrgemmNnzNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint transB, jint m, jint n, jint k, jobject descrA, jint nnzA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrB, jint nnzB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedRowPtrC, jobject nnzTotalDevHostPtr)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseXcsrgemmNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseXcsrgemmNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseXcsrgemmNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseXcsrgemmNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseXcsrgemmNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseXcsrgemmNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseXcsrgemmNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseXcsrgemmNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseXcsrgemmNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (nnzTotalDevHostPtr == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nnzTotalDevHostPtr' is null for cusparseXcsrgemmNnz");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseXcsrgemmNnz(handle=%p, transA=%d, transB=%d, m=%d, n=%d, k=%d, descrA=%p, nnzA=%d, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrB=%p, nnzB=%d, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedRowPtrC=%p, nnzTotalDevHostPtr=%p)\n",
        handle, transA, transB, m, n, k, descrA, nnzA, csrSortedRowPtrA, csrSortedColIndA, descrB, nnzB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedRowPtrC, nnzTotalDevHostPtr);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    int* csrSortedRowPtrB_native;
    int* csrSortedColIndB_native;
    cusparseMatDescr_t descrC_native;
    int* csrSortedRowPtrC_native;
    int* nnzTotalDevHostPtr_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedRowPtrB_native = (int*)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int*)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedRowPtrC_native = (int*)getPointer(env, csrSortedRowPtrC);
    nnzTotalDevHostPtr_native = (int*)getPointer(env, nnzTotalDevHostPtr);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseXcsrgemmNnz(handle_native, transA_native, transB_native, m_native, n_native, k_native, descrA_native, nnzA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrB_native, nnzB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedRowPtrC_native, nnzTotalDevHostPtr_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // descrA is a read-only native pointer
    // nnzA is primitive
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrB is a read-only native pointer
    // nnzB is primitive
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is a read-only native pointer
    // csrSortedRowPtrC is a native pointer
    // nnzTotalDevHostPtr is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrgemmNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint transB, jint m, jint n, jint k, jobject descrA, jint nnzA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrB, jint nnzB, jobject csrSortedValB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedValB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValB' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseScsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrgemm(handle=%p, transA=%d, transB=%d, m=%d, n=%d, k=%d, descrA=%p, nnzA=%d, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrB=%p, nnzB=%d, csrSortedValB=%p, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p)\n",
        handle, transA, transB, m, n, k, descrA, nnzA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, descrB, nnzB, csrSortedValB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    float* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    float* csrSortedValB_native;
    int* csrSortedRowPtrB_native;
    int* csrSortedColIndB_native;
    cusparseMatDescr_t descrC_native;
    float* csrSortedValC_native;
    int* csrSortedRowPtrC_native;
    int* csrSortedColIndC_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedValA_native = (float*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedValB_native = (float*)getPointer(env, csrSortedValB);
    csrSortedRowPtrB_native = (int*)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int*)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (float*)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int*)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int*)getPointer(env, csrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrgemm(handle_native, transA_native, transB_native, m_native, n_native, k_native, descrA_native, nnzA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrB_native, nnzB_native, csrSortedValB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // descrA is a read-only native pointer
    // nnzA is primitive
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrB is a read-only native pointer
    // nnzB is primitive
    // csrSortedValB is a native pointer
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is a read-only native pointer
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrgemmNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint transB, jint m, jint n, jint k, jobject descrA, jint nnzA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrB, jint nnzB, jobject csrSortedValB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedValB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValB' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseDcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrgemm(handle=%p, transA=%d, transB=%d, m=%d, n=%d, k=%d, descrA=%p, nnzA=%d, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrB=%p, nnzB=%d, csrSortedValB=%p, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p)\n",
        handle, transA, transB, m, n, k, descrA, nnzA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, descrB, nnzB, csrSortedValB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    double* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    double* csrSortedValB_native;
    int* csrSortedRowPtrB_native;
    int* csrSortedColIndB_native;
    cusparseMatDescr_t descrC_native;
    double* csrSortedValC_native;
    int* csrSortedRowPtrC_native;
    int* csrSortedColIndC_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedValA_native = (double*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedValB_native = (double*)getPointer(env, csrSortedValB);
    csrSortedRowPtrB_native = (int*)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int*)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (double*)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int*)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int*)getPointer(env, csrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrgemm(handle_native, transA_native, transB_native, m_native, n_native, k_native, descrA_native, nnzA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrB_native, nnzB_native, csrSortedValB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // descrA is a read-only native pointer
    // nnzA is primitive
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrB is a read-only native pointer
    // nnzB is primitive
    // csrSortedValB is a native pointer
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is a read-only native pointer
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrgemmNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint transB, jint m, jint n, jint k, jobject descrA, jint nnzA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrB, jint nnzB, jobject csrSortedValB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedValB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValB' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseCcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrgemm(handle=%p, transA=%d, transB=%d, m=%d, n=%d, k=%d, descrA=%p, nnzA=%d, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrB=%p, nnzB=%d, csrSortedValB=%p, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p)\n",
        handle, transA, transB, m, n, k, descrA, nnzA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, descrB, nnzB, csrSortedValB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    cuComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    cuComplex* csrSortedValB_native;
    int* csrSortedRowPtrB_native;
    int* csrSortedColIndB_native;
    cusparseMatDescr_t descrC_native;
    cuComplex* csrSortedValC_native;
    int* csrSortedRowPtrC_native;
    int* csrSortedColIndC_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedValA_native = (cuComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedValB_native = (cuComplex*)getPointer(env, csrSortedValB);
    csrSortedRowPtrB_native = (int*)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int*)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (cuComplex*)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int*)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int*)getPointer(env, csrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCcsrgemm(handle_native, transA_native, transB_native, m_native, n_native, k_native, descrA_native, nnzA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrB_native, nnzB_native, csrSortedValB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // descrA is a read-only native pointer
    // nnzA is primitive
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrB is a read-only native pointer
    // nnzB is primitive
    // csrSortedValB is a native pointer
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is a read-only native pointer
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseZcsrgemmNative(JNIEnv *env, jclass cls, jobject handle, jint transA, jint transB, jint m, jint n, jint k, jobject descrA, jint nnzA, jobject csrSortedValA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrB, jint nnzB, jobject csrSortedValB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject descrC, jobject csrSortedValC, jobject csrSortedRowPtrC, jobject csrSortedColIndC)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedValA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValA' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedValB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValB' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrC' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedValC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedValC' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedRowPtrC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrC' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndC == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndC' is null for cusparseZcsrgemm");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseZcsrgemm(handle=%p, transA=%d, transB=%d, m=%d, n=%d, k=%d, descrA=%p, nnzA=%d, csrSortedValA=%p, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrB=%p, nnzB=%d, csrSortedValB=%p, csrSortedRowPtrB=%p, csrSortedColIndB=%p, descrC=%p, csrSortedValC=%p, csrSortedRowPtrC=%p, csrSortedColIndC=%p)\n",
        handle, transA, transB, m, n, k, descrA, nnzA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, descrB, nnzB, csrSortedValB, csrSortedRowPtrB, csrSortedColIndB, descrC, csrSortedValC, csrSortedRowPtrC, csrSortedColIndC);

    // Native variable declarations
    cusparseHandle_t handle_native;
    cusparseOperation_t transA_native;
    cusparseOperation_t transB_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    cuDoubleComplex* csrSortedValA_native;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    cuDoubleComplex* csrSortedValB_native;
    int* csrSortedRowPtrB_native;
    int* csrSortedColIndB_native;
    cusparseMatDescr_t descrC_native;
    cuDoubleComplex* csrSortedValC_native;
    int* csrSortedRowPtrC_native;
    int* csrSortedColIndC_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    transA_native = (cusparseOperation_t)transA;
    transB_native = (cusparseOperation_t)transB;
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedValA_native = (cuDoubleComplex*)getPointer(env, csrSortedValA);
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedValB_native = (cuDoubleComplex*)getPointer(env, csrSortedValB);
    csrSortedRowPtrB_native = (int*)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int*)getPointer(env, csrSortedColIndB);
    descrC_native = (cusparseMatDescr_t)getNativePointerValue(env, descrC);
    csrSortedValC_native = (cuDoubleComplex*)getPointer(env, csrSortedValC);
    csrSortedRowPtrC_native = (int*)getPointer(env, csrSortedRowPtrC);
    csrSortedColIndC_native = (int*)getPointer(env, csrSortedColIndC);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseZcsrgemm(handle_native, transA_native, transB_native, m_native, n_native, k_native, descrA_native, nnzA_native, csrSortedValA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrB_native, nnzB_native, csrSortedValB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, descrC_native, csrSortedValC_native, csrSortedRowPtrC_native, csrSortedColIndC_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // transA is primitive
    // transB is primitive
    // m is primitive
    // n is primitive
    // k is primitive
    // descrA is a read-only native pointer
    // nnzA is primitive
    // csrSortedValA is a native pointer
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrB is a read-only native pointer
    // nnzB is primitive
    // csrSortedValB is a native pointer
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    // descrC is a read-only native pointer
    // csrSortedValC is a native pointer
    // csrSortedRowPtrC is a native pointer
    // csrSortedColIndC is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/** Description: Compute sparse - sparse matrix multiplication for matrices
stored in CSR format. */
JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCreateCsrgemm2InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCreateCsrgemm2Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCreateCsrgemm2Info(info=%p)\n",
        info);

    // Native variable declarations
    csrgemm2Info_t info_native;

    // Obtain native variable values
    // info is write-only

    // Native function call
    cusparseStatus_t jniResult_native = cusparseCreateCsrgemm2Info(&info_native);

    // Write back native variable values
    setNativePointerValue(env, info, (jlong)info_native);

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDestroyCsrgemm2InfoNative(JNIEnv *env, jclass cls, jobject info)
{
    // Null-checks for non-primitive arguments
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDestroyCsrgemm2Info");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDestroyCsrgemm2Info(info=%p)\n",
        info);

    // Native variable declarations
    csrgemm2Info_t info_native;

    // Obtain native variable values
    info_native = (csrgemm2Info_t)getNativePointerValue(env, info);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDestroyCsrgemm2Info(info_native);

    // Write back native variable values
    // info is a read-only native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseScsrgemm2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint k, jobject alpha, jobject descrA, jint nnzA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrB, jint nnzB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject beta, jobject descrD, jint nnzD, jobject csrSortedRowPtrD, jobject csrSortedColIndD, jobject info, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // k is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrD == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrD' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzD is primitive
    if (csrSortedRowPtrD == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrD' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndD == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndD' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseScsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseScsrgemm2_bufferSizeExt(handle=%p, m=%d, n=%d, k=%d, alpha=%p, descrA=%p, nnzA=%d, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrB=%p, nnzB=%d, csrSortedRowPtrB=%p, csrSortedColIndB=%p, beta=%p, descrD=%p, nnzD=%d, csrSortedRowPtrD=%p, csrSortedColIndD=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, n, k, alpha, descrA, nnzA, csrSortedRowPtrA, csrSortedColIndA, descrB, nnzB, csrSortedRowPtrB, csrSortedColIndB, beta, descrD, nnzD, csrSortedRowPtrD, csrSortedColIndD, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    float* alpha_native;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    int* csrSortedRowPtrB_native;
    int* csrSortedColIndB_native;
    float* beta_native;
    cusparseMatDescr_t descrD_native;
    int nnzD_native = 0;
    int* csrSortedRowPtrD_native;
    int* csrSortedColIndD_native;
    csrgemm2Info_t info_native;
    size_t* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (float*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedRowPtrB_native = (int*)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int*)getPointer(env, csrSortedColIndB);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (float*)beta_pointerData->getPointer(env);
    descrD_native = (cusparseMatDescr_t)getNativePointerValue(env, descrD);
    nnzD_native = (int)nnzD;
    csrSortedRowPtrD_native = (int*)getPointer(env, csrSortedRowPtrD);
    csrSortedColIndD_native = (int*)getPointer(env, csrSortedColIndD);
    info_native = (csrgemm2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (size_t*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseScsrgemm2_bufferSizeExt(handle_native, m_native, n_native, k_native, alpha_native, descrA_native, nnzA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrB_native, nnzB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, beta_native, descrD_native, nnzD_native, csrSortedRowPtrD_native, csrSortedColIndD_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // n is primitive
    // k is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // nnzA is primitive
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrB is a read-only native pointer
    // nnzB is primitive
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrD is a read-only native pointer
    // nnzD is primitive
    // csrSortedRowPtrD is a native pointer
    // csrSortedColIndD is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseDcsrgemm2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint k, jobject alpha, jobject descrA, jint nnzA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrB, jint nnzB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject beta, jobject descrD, jint nnzD, jobject csrSortedRowPtrD, jobject csrSortedColIndD, jobject info, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // k is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrD == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrD' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzD is primitive
    if (csrSortedRowPtrD == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrD' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndD == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndD' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseDcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseDcsrgemm2_bufferSizeExt(handle=%p, m=%d, n=%d, k=%d, alpha=%p, descrA=%p, nnzA=%d, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrB=%p, nnzB=%d, csrSortedRowPtrB=%p, csrSortedColIndB=%p, beta=%p, descrD=%p, nnzD=%d, csrSortedRowPtrD=%p, csrSortedColIndD=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, n, k, alpha, descrA, nnzA, csrSortedRowPtrA, csrSortedColIndA, descrB, nnzB, csrSortedRowPtrB, csrSortedColIndB, beta, descrD, nnzD, csrSortedRowPtrD, csrSortedColIndD, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    double* alpha_native;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    int* csrSortedRowPtrB_native;
    int* csrSortedColIndB_native;
    double* beta_native;
    cusparseMatDescr_t descrD_native;
    int nnzD_native = 0;
    int* csrSortedRowPtrD_native;
    int* csrSortedColIndD_native;
    csrgemm2Info_t info_native;
    size_t* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (double*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedRowPtrB_native = (int*)getPointer(env, csrSortedRowPtrB);
    csrSortedColIndB_native = (int*)getPointer(env, csrSortedColIndB);
    PointerData *beta_pointerData = initPointerData(env, beta);
    if (beta_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    beta_native = (double*)beta_pointerData->getPointer(env);
    descrD_native = (cusparseMatDescr_t)getNativePointerValue(env, descrD);
    nnzD_native = (int)nnzD;
    csrSortedRowPtrD_native = (int*)getPointer(env, csrSortedRowPtrD);
    csrSortedColIndD_native = (int*)getPointer(env, csrSortedColIndD);
    info_native = (csrgemm2Info_t)getNativePointerValue(env, info);
    pBufferSizeInBytes_native = (size_t*)getPointer(env, pBufferSizeInBytes);

    // Native function call
    cusparseStatus_t jniResult_native = cusparseDcsrgemm2_bufferSizeExt(handle_native, m_native, n_native, k_native, alpha_native, descrA_native, nnzA_native, csrSortedRowPtrA_native, csrSortedColIndA_native, descrB_native, nnzB_native, csrSortedRowPtrB_native, csrSortedColIndB_native, beta_native, descrD_native, nnzD_native, csrSortedRowPtrD_native, csrSortedColIndD_native, info_native, pBufferSizeInBytes_native);

    // Write back native variable values
    // handle is a read-only native pointer
    // m is primitive
    // n is primitive
    // k is primitive
    if (!releasePointerData(env, alpha_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrA is a read-only native pointer
    // nnzA is primitive
    // csrSortedRowPtrA is a native pointer
    // csrSortedColIndA is a native pointer
    // descrB is a read-only native pointer
    // nnzB is primitive
    // csrSortedRowPtrB is a native pointer
    // csrSortedColIndB is a native pointer
    if (!releasePointerData(env, beta_pointerData, JNI_ABORT)) return JCUSPARSE_STATUS_INTERNAL_ERROR;
    // descrD is a read-only native pointer
    // nnzD is primitive
    // csrSortedRowPtrD is a native pointer
    // csrSortedColIndD is a native pointer
    // info is a read-only native pointer
    // pBufferSizeInBytes is a native pointer

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

JNIEXPORT jint JNICALL Java_jcuda_jcusparse_JCusparse_cusparseCcsrgemm2_1bufferSizeExtNative(JNIEnv *env, jclass cls, jobject handle, jint m, jint n, jint k, jobject alpha, jobject descrA, jint nnzA, jobject csrSortedRowPtrA, jobject csrSortedColIndA, jobject descrB, jint nnzB, jobject csrSortedRowPtrB, jobject csrSortedColIndB, jobject beta, jobject descrD, jint nnzD, jobject csrSortedRowPtrD, jobject csrSortedColIndD, jobject info, jlongArray pBufferSizeInBytes)
{
    // Null-checks for non-primitive arguments
    if (handle == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'handle' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // m is primitive
    // n is primitive
    // k is primitive
    if (alpha == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'alpha' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrA' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzA is primitive
    if (csrSortedRowPtrA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrA' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndA == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndA' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrB' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzB is primitive
    if (csrSortedRowPtrB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrB' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndB == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndB' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (beta == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'beta' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (descrD == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'descrD' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    // nnzD is primitive
    if (csrSortedRowPtrD == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedRowPtrD' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (csrSortedColIndD == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'csrSortedColIndD' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (info == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'info' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    if (pBufferSizeInBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pBufferSizeInBytes' is null for cusparseCcsrgemm2_bufferSizeExt");
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing cusparseCcsrgemm2_bufferSizeExt(handle=%p, m=%d, n=%d, k=%d, alpha=%p, descrA=%p, nnzA=%d, csrSortedRowPtrA=%p, csrSortedColIndA=%p, descrB=%p, nnzB=%d, csrSortedRowPtrB=%p, csrSortedColIndB=%p, beta=%p, descrD=%p, nnzD=%d, csrSortedRowPtrD=%p, csrSortedColIndD=%p, info=%p, pBufferSizeInBytes=%p)\n",
        handle, m, n, k, alpha, descrA, nnzA, csrSortedRowPtrA, csrSortedColIndA, descrB, nnzB, csrSortedRowPtrB, csrSortedColIndB, beta, descrD, nnzD, csrSortedRowPtrD, csrSortedColIndD, info, pBufferSizeInBytes);

    // Native variable declarations
    cusparseHandle_t handle_native;
    int m_native = 0;
    int n_native = 0;
    int k_native = 0;
    cuComplex* alpha_native;
    cusparseMatDescr_t descrA_native;
    int nnzA_native = 0;
    int* csrSortedRowPtrA_native;
    int* csrSortedColIndA_native;
    cusparseMatDescr_t descrB_native;
    int nnzB_native = 0;
    int* csrSortedRowPtrB_native;
    int* csrSortedColIndB_native;
    cuComplex* beta_native;
    cusparseMatDescr_t descrD_native;
    int nnzD_native = 0;
    int* csrSortedRowPtrD_native;
    int* csrSortedColIndD_native;
    csrgemm2Info_t info_native;
    size_t* pBufferSizeInBytes_native;

    // Obtain native variable values
    handle_native = (cusparseHandle_t)getNativePointerValue(env, handle);
    m_native = (int)m;
    n_native = (int)n;
    k_native = (int)k;
    PointerData *alpha_pointerData = initPointerData(env, alpha);
    if (alpha_pointerData == NULL)
    {
        return JCUSPARSE_STATUS_INTERNAL_ERROR;
    }
    alpha_native = (cuComplex*)alpha_pointerData->getPointer(env);
    descrA_native = (cusparseMatDescr_t)getNativePointerValue(env, descrA);
    nnzA_native = (int)nnzA;
    csrSortedRowPtrA_native = (int*)getPointer(env, csrSortedRowPtrA);
    csrSortedColIndA_native = (int*)getPointer(env, csrSortedColIndA);
    descrB_native = (cusparseMatDescr_t)getNativePointerValue(env, descrB);
    nnzB_native = (int)nnzB;
    csrSortedRowPtrB_native = (int*)getPointer(env, csrSortedRowPtrB);
    csrSort