/*
 * JNpp - Java bindings for NPP, to be used with JCuda
 *
 * Copyright (c) 2010-2012 Marco Hutter - http://www.jcuda.org
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */


#include "JNppi.hpp"
#include "JNppi_common.hpp"
#include <iostream>
#include <string>
#include <map>


jfieldID cuComplex_x; // float
jfieldID cuComplex_y; // float

jfieldID cuDoubleComplex_x; // double
jfieldID cuDoubleComplex_y; // double

jfieldID NppiPoint_x; // int
jfieldID NppiPoint_y; // int

jfieldID NppiSize_width; // int
jfieldID NppiSize_height; // int

jfieldID NppiRect_x; // int
jfieldID NppiRect_y; // int
jfieldID NppiRect_width; // int
jfieldID NppiRect_height; // int


jfieldID Npp16sc_re; // short
jfieldID Npp16sc_im; // short

jfieldID Npp32sc_re; // int
jfieldID Npp32sc_im; // int

jfieldID Npp64sc_re; // long
jfieldID Npp64sc_im; // long

jfieldID Npp32fc_re; // float
jfieldID Npp32fc_im; // float

jfieldID Npp64fc_re; // double
jfieldID Npp64fc_im; // double


// Class and method ID for TypedPointer and its constructor
jclass TypedPointer_Class;
jmethodID TypedPointer_Constructor;


/**
 * Creates a global reference to the class with the given name and
 * stores it in the given jclass argument, and stores the no-args
 * constructor ID for this class in the given jmethodID.
 * Returns whether this initialization succeeded.
 */
bool init(JNIEnv *env, const char *className, jclass &globalCls, jmethodID &constructor)
{
    jclass cls = NULL;
    if (!init(env, cls, className)) return false;
    if (!init(env, cls, constructor, "<init>", "()V")) return false;

    globalCls = (jclass)env->NewGlobalRef(cls);
    if (globalCls == NULL)
    {
        Logger::log(LOG_ERROR, "Failed to create reference to class %s\n", className);
        return false;
    }
    return true;
}

/**
 * Called when the library is loaded. Will initialize all
 * required field and method IDs
 */
JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM *jvm, void *reserved)
{
    JNIEnv *env = NULL;
    if (jvm->GetEnv((void **)&env, JNI_VERSION_1_4))
    {
        return JNI_ERR;
    }

    Logger::log(LOG_TRACE, "Initializing JNppi\n");

    jclass cls = NULL;

    // Initialize the JNIUtils and PointerUtils
    if (initJNIUtils(env) == JNI_ERR) return JNI_ERR;
    if (initPointerUtils(env) == JNI_ERR) return JNI_ERR;

    // Obtain the constructor for the TypedPointer class
    if (!init(env, "jcuda/jnpp/TypedPointer", TypedPointer_Class, TypedPointer_Constructor)) return JNI_ERR;

    // Obtain the fieldIDs for cuComplex#x and cuComplex#y
    if (!init(env, cls, "jcuda/cuComplex")) return JNI_ERR;
    if (!init(env, cls, cuComplex_x, "x", "F")) return JNI_ERR;
    if (!init(env, cls, cuComplex_y, "y", "F")) return JNI_ERR;

    // Obtain the fieldIDs for cuDoubleComplex#x and cuDoubleComplex#y
    if (!init(env, cls, "jcuda/cuDoubleComplex")) return JNI_ERR;
    if (!init(env, cls, cuDoubleComplex_x, "x", "D")) return JNI_ERR;
    if (!init(env, cls, cuDoubleComplex_y, "y", "D")) return JNI_ERR;

    // Obtain the fieldIDs for NppiPoint
    if (!init(env, cls, "jcuda/jnpp/NppiPoint")) return JNI_ERR;
    if (!init(env, cls, NppiPoint_x, "x", "I")) return JNI_ERR;
    if (!init(env, cls, NppiPoint_y, "y", "I")) return JNI_ERR;

    // Obtain the fieldIDs for NppiSize
    if (!init(env, cls, "jcuda/jnpp/NppiSize")) return JNI_ERR;
    if (!init(env, cls, NppiSize_width,  "width",  "I")) return JNI_ERR;
    if (!init(env, cls, NppiSize_height, "height", "I")) return JNI_ERR;

    // Obtain the fieldIDs for NppiRect
    if (!init(env, cls, "jcuda/jnpp/NppiRect")) return JNI_ERR;
    if (!init(env, cls, NppiRect_x,      "x",      "I")) return JNI_ERR;
    if (!init(env, cls, NppiRect_y,      "y",      "I")) return JNI_ERR;
    if (!init(env, cls, NppiRect_width,  "width",  "I")) return JNI_ERR;
    if (!init(env, cls, NppiRect_height, "height", "I")) return JNI_ERR;

    // Obtain the fieldIDs for Npp16sc
    if (!init(env, cls, "jcuda/jnpp/types/Npp16sc")) return JNI_ERR;
    if (!init(env, cls, Npp16sc_re, "re", "S")) return JNI_ERR;
    if (!init(env, cls, Npp16sc_im, "im", "S")) return JNI_ERR;

    // Obtain the fieldIDs for Npp32sc
    if (!init(env, cls, "jcuda/jnpp/types/Npp32sc")) return JNI_ERR;
    if (!init(env, cls, Npp32sc_re, "re", "I")) return JNI_ERR;
    if (!init(env, cls, Npp32sc_im, "im", "I")) return JNI_ERR;

    // Obtain the fieldIDs for Npp64sc
    if (!init(env, cls, "jcuda/jnpp/types/Npp64sc")) return JNI_ERR;
    if (!init(env, cls, Npp64sc_re, "re", "J")) return JNI_ERR;
    if (!init(env, cls, Npp64sc_im, "im", "J")) return JNI_ERR;

    // Obtain the fieldIDs for Npp32fc
    if (!init(env, cls, "jcuda/jnpp/types/Npp32fc")) return JNI_ERR;
    if (!init(env, cls, Npp32fc_re, "re", "F")) return JNI_ERR;
    if (!init(env, cls, Npp32fc_im, "im", "F")) return JNI_ERR;

    // Obtain the fieldIDs for Npp64fc
    if (!init(env, cls, "jcuda/jnpp/types/Npp64fc")) return JNI_ERR;
    if (!init(env, cls, Npp64fc_re, "re", "D")) return JNI_ERR;
    if (!init(env, cls, Npp64fc_im, "im", "D")) return JNI_ERR;

    return JNI_VERSION_1_4;

}


/**
 * Create a new TypedPointer object with the given native pointer value
 */
jobject createPointer(JNIEnv *env, jlong pointerValue)
{
    jobject result = env->NewObject(TypedPointer_Class, TypedPointer_Constructor);
    if (env->ExceptionCheck())
    {
        return NULL;
    }
    setPointer(env, result, pointerValue);
    return result;
}


/**
 * Initializes the given native output object with the values from the
 * given java input object.
 */
bool initNative(JNIEnv *env, jobject input, NppiPoint &output)
{
    output.x = (int)env->GetIntField(input, NppiPoint_x);
    output.y = (int)env->GetIntField(input, NppiPoint_y);
    return true;
}

/**
 * Initializes the given native output object with the values from the
 * given java input object.
 */
bool initNative(JNIEnv *env, jobject input, NppiSize &output)
{
    output.width  = (int)env->GetIntField(input, NppiSize_width);
    output.height = (int)env->GetIntField(input, NppiSize_height);
    return true;
}

/**
 * Initializes the given native output object with the values from the
 * given java input object.
 */
bool initNative(JNIEnv *env, jobject input, NppiRect &output)
{
    output.x      = (int)env->GetIntField(input, NppiRect_x);
    output.y      = (int)env->GetIntField(input, NppiRect_y);
    output.width  = (int)env->GetIntField(input, NppiRect_width);
    output.height = (int)env->GetIntField(input, NppiRect_height);
    return true;
}


//=== Complex number struct init =============================================

/**
 * Initializes the given native output object with the values from the
 * given java input object.
 */
bool initNative(JNIEnv *env, jobject input, Npp16sc &output)
{
    output.re = (Npp16s)env->GetShortField(input, Npp16sc_re);
    output.im = (Npp16s)env->GetShortField(input, Npp16sc_im);
    return true;
}

/**
 * Initializes the given native output object with the values from the
 * given java input object.
 */
bool initNative(JNIEnv *env, jobject input, Npp32sc &output)
{
    output.re = (Npp32s)env->GetIntField(input, Npp32sc_re);
    output.im = (Npp32s)env->GetIntField(input, Npp32sc_im);
    return true;
}

/**
 * Initializes the given native output object with the values from the
 * given java input object.
 */
bool initNative(JNIEnv *env, jobject input, Npp64sc &output)
{
    output.re = (Npp64s)env->GetLongField(input, Npp64sc_re);
    output.im = (Npp64s)env->GetLongField(input, Npp64sc_im);
    return true;
}

/**
 * Initializes the given native output object with the values from the
 * given java input object.
 */
bool initNative(JNIEnv *env, jobject input, Npp32fc &output)
{
    output.re = (Npp32f)env->GetFloatField(input, Npp32fc_re);
    output.im = (Npp32f)env->GetFloatField(input, Npp32fc_im);
    return true;
}

/**
 * Initializes the given native output object with the values from the
 * given java input object.
 */
bool initNative(JNIEnv *env, jobject input, Npp64fc &output)
{
    output.re = (Npp64f)env->GetDoubleField(input, Npp64fc_re);
    output.im = (Npp64f)env->GetDoubleField(input, Npp64fc_im);
    return true;
}



//=== Complex number struct release ==========================================

/**
 * Write the values from the given native output object back to
 * the java input object
 */
bool releaseNative(JNIEnv *env, Npp16sc &output, jobject input)
{
    env->SetShortField(input, Npp16sc_re, (jshort)output.re);
    env->SetShortField(input, Npp16sc_im, (jshort)output.im);
    return true;
}

/**
 * Write the values from the given native output object back to
 * the java input object
 */
bool releaseNative(JNIEnv *env, Npp32sc &output, jobject input)
{
    env->SetIntField(input, Npp32sc_re, (jint)output.re);
    env->SetIntField(input, Npp32sc_im, (jint)output.im);
    return true;
}

/**
 * Write the values from the given native output object back to
 * the java input object
 */
bool releaseNative(JNIEnv *env, Npp64sc &output, jobject input)
{
    env->SetLongField(input, Npp64sc_re, (jlong)output.re);
    env->SetLongField(input, Npp64sc_im, (jlong)output.im);
    return true;
}

/**
 * Write the values from the given native output object back to
 * the java input object
 */
bool releaseNative(JNIEnv *env, Npp32fc &output, jobject input)
{
    env->SetFloatField(input, Npp32fc_re, (jfloat)output.re);
    env->SetFloatField(input, Npp32fc_im, (jfloat)output.im);
    return true;
}

/**
 * Write the values from the given native output object back to
 * the java input object
 */
bool releaseNative(JNIEnv *env, Npp64fc &output, jobject input)
{
    env->SetDoubleField(input, Npp64fc_re, (jdouble)output.re);
    env->SetDoubleField(input, Npp64fc_im, (jdouble)output.im);
    return true;
}





//=== 2D prmitive arrays =====================================================


// TODO: Many of these methods could be replaced by more generic
// ones that should be defined in CommonJNI

/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, double output[4][2])
{
    int outerSize = 4;
    int innerSize = 2;
    jsize size = env->GetArrayLength(input);
    if (size < outerSize)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException", "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<outerSize; i++)
    {
        jdoubleArray inputElement = (jdoubleArray)env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException", "Array contains 'null' elements");
            return false;
        }
        jsize elementSize = env->GetArrayLength(inputElement);
        if (elementSize < innerSize)
        {
            ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException", "Array parameter has insufficient size");
            return false;
        }
        jdouble *arrayContents = (jdouble*)env->GetPrimitiveArrayCritical(inputElement, NULL);
        for (int j=0; j<innerSize; j++)
        {
            output[i][j] = (double)arrayContents[j];
        }
        env->ReleasePrimitiveArrayCritical(inputElement, arrayContents, JNI_ABORT);
    }
    return true;
}

/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, double output[4][2], jobjectArray input)
{
    int outerSize = 4;
    int innerSize = 2;
    jsize size = env->GetArrayLength(input);
    if (size < outerSize)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException", "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<outerSize; i++)
    {
        jdoubleArray inputElement = (jdoubleArray)env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException", "Array contains 'null' elements");
            return false;
        }
        jsize elementSize = env->GetArrayLength(inputElement);
        if (elementSize < innerSize)
        {
            ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException", "Array parameter has insufficient size");
            return false;
        }
        jdouble *arrayContents = (jdouble*)env->GetPrimitiveArrayCritical(inputElement, NULL);
        for (int j=0; j<innerSize; j++)
        {
            arrayContents[j] = (jdouble)output[i][j];
        }
        env->ReleasePrimitiveArrayCritical(inputElement, arrayContents, 0);
    }
    return true;
}





/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, double output[2][3])
{
    int outerSize = 2;
    int innerSize = 3;
    jsize size = env->GetArrayLength(input);
    if (size < outerSize)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException", "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<outerSize; i++)
    {
        jdoubleArray inputElement = (jdoubleArray)env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException", "Array contains 'null' elements");
            return false;
        }
        jsize elementSize = env->GetArrayLength(inputElement);
        if (elementSize < innerSize)
        {
            ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException", "Array parameter has insufficient size");
            return false;
        }
        jdouble *arrayContents = (jdouble*)env->GetPrimitiveArrayCritical(inputElement, NULL);
        for (int j=0; j<innerSize; j++)
        {
            output[i][j] = (double)arrayContents[j];
        }
        env->ReleasePrimitiveArrayCritical(inputElement, arrayContents, JNI_ABORT);
    }
    return true;
}

/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, double output[2][3], jobjectArray input)
{
    int outerSize = 2;
    int innerSize = 3;
    jsize size = env->GetArrayLength(input);
    if (size < outerSize)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException", "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<outerSize; i++)
    {
        jdoubleArray inputElement = (jdoubleArray)env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException", "Array contains 'null' elements");
            return false;
        }
        jsize elementSize = env->GetArrayLength(inputElement);
        if (elementSize < innerSize)
        {
            ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException", "Array parameter has insufficient size");
            return false;
        }
        jdouble *arrayContents = (jdouble*)env->GetPrimitiveArrayCritical(inputElement, NULL);
        for (int j=0; j<innerSize; j++)
        {
            arrayContents[j] = (jdouble)output[i][j];
        }
        env->ReleasePrimitiveArrayCritical(inputElement, arrayContents, 0);
    }
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, Npp32f output[3][4])
{
    int outerSize = 3;
    int innerSize = 4;
    jsize size = env->GetArrayLength(input);
    if (size < outerSize)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException", "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<outerSize; i++)
    {
        jfloatArray inputElement = (jfloatArray)env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException", "Array contains 'null' elements");
            return false;
        }
        jsize elementSize = env->GetArrayLength(inputElement);
        if (elementSize < innerSize)
        {
            ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException", "Array parameter has insufficient size");
            return false;
        }
        jfloat *arrayContents = (jfloat*)env->GetPrimitiveArrayCritical(inputElement, NULL);
        for (int j=0; j<innerSize; j++)
        {
            output[i][j] = (Npp32f)arrayContents[j];
        }
        env->ReleasePrimitiveArrayCritical(inputElement, arrayContents, JNI_ABORT);
    }
    return true;
}

/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp32f output[3][4], jobjectArray input)
{
    int outerSize = 3;
    int innerSize = 4;
    jsize size = env->GetArrayLength(input);
    if (size < outerSize)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException", "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<outerSize; i++)
    {
        jfloatArray inputElement = (jfloatArray)env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException", "Array contains 'null' elements");
            return false;
        }
        jsize elementSize = env->GetArrayLength(inputElement);
        if (elementSize < innerSize)
        {
            ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException", "Array parameter has insufficient size");
            return false;
        }
        jfloat *arrayContents = (jfloat*)env->GetPrimitiveArrayCritical(inputElement, NULL);
        for (int j=0; j<innerSize; j++)
        {
            arrayContents[j] = (jfloat)output[i][j];
        }
        env->ReleasePrimitiveArrayCritical(inputElement, arrayContents, 0);
    }
    return true;
}






//=== Primitive arrays =======================================================

/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jbyteArray input, Npp8s output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jbyte* arrayContents = (jbyte*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        output[i] = (Npp8s)arrayContents[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, JNI_ABORT);
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp8s output[], jbyteArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jbyte* arrayContents = (jbyte*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        arrayContents[i] = (jbyte)output[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, 0);
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jbyteArray input, Npp8u output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jbyte* arrayContents = (jbyte*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        output[i] = (Npp8u)arrayContents[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, JNI_ABORT);
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp8u output[], jbyteArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jbyte* arrayContents = (jbyte*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        arrayContents[i] = (jbyte)output[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, 0);
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jshortArray input, Npp16s output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jshort* arrayContents = (jshort*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        output[i] = (Npp16s)arrayContents[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, JNI_ABORT);
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp16s output[], jshortArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jshort* arrayContents = (jshort*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        arrayContents[i] = (jshort)output[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, 0);
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jshortArray input, Npp16u output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jshort* arrayContents = (jshort*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        output[i] = (Npp16u)arrayContents[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, JNI_ABORT);
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp16u output[], jshortArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jshort* arrayContents = (jshort*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        arrayContents[i] = (jshort)output[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, 0);
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jintArray input, Npp32s output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jint* arrayContents = (jint*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        output[i] = (Npp32s)arrayContents[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, JNI_ABORT);
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp32s output[], jintArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jint* arrayContents = (jint*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        arrayContents[i] = (jint)output[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, 0);
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jintArray input, Npp32u output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jint* arrayContents = (jint*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        output[i] = (Npp32u)arrayContents[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, JNI_ABORT);
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp32u output[], jintArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jint* arrayContents = (jint*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        arrayContents[i] = (jint)output[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, 0);
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jlongArray input, Npp64s output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jlong* arrayContents = (jlong*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        output[i] = (Npp64s)arrayContents[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, JNI_ABORT);
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp64s output[], jlongArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jlong* arrayContents = (jlong*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        arrayContents[i] = (jlong)output[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, 0);
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jlongArray input, Npp64u output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jlong* arrayContents = (jlong*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        output[i] = (Npp64u)arrayContents[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, JNI_ABORT);
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp64u output[], jlongArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jlong* arrayContents = (jlong*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        arrayContents[i] = (jlong)output[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, 0);
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jfloatArray input, Npp32f output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jfloat* arrayContents = (jfloat*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        output[i] = (Npp32f)arrayContents[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, JNI_ABORT);
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp32f output[], jfloatArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jfloat* arrayContents = (jfloat*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        arrayContents[i] = (jfloat)output[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, 0);
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jdoubleArray input, Npp64f output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jdouble* arrayContents = (jdouble*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        output[i] = (Npp64f)arrayContents[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, JNI_ABORT);
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp64f output[], jdoubleArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    jdouble* arrayContents = (jdouble*)env->GetPrimitiveArrayCritical(input, NULL);
    if (arrayContents == NULL)
    {
        return false;
    }
    for (int i=0; i<size; i++)
    {
        arrayContents[i] = (jdouble)output[i];
    }
    env->ReleasePrimitiveArrayCritical(input, arrayContents, 0);
    return true;
}





//=== Complex number arrays ==================================================

/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, Npp16sc output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        if (!initNative(env, inputElement, output[i]))
        {
            return false;
        }
    }
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp16sc output[], jobjectArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        if (!releaseNative(env, output[i], inputElement))
        {
            return false;
        }
    }
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, Npp32sc output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        if (!initNative(env, inputElement, output[i]))
        {
            return false;
        }
    }
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp32sc output[], jobjectArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        if (!releaseNative(env, output[i], inputElement))
        {
            return false;
        }
    }
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, Npp32fc output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        if (!initNative(env, inputElement, output[i]))
        {
            return false;
        }
    }
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp32fc output[], jobjectArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        if (!releaseNative(env, output[i], inputElement))
        {
            return false;
        }
    }
    return true;
}



/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, Npp64fc output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        if (!initNative(env, inputElement, output[i]))
        {
            return false;
        }
    }
    return true;
}


/**
 * Release the given native output array, writing its contents back to
 * the given java input array
 */
bool releaseNative(JNIEnv *env, Npp64fc output[], jobjectArray input, int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        if (!releaseNative(env, output[i], inputElement))
        {
            return false;
        }
    }
    return true;
}





//=== Primitive pointer arrays ===============================================

/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, Npp8u* output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        output[i] = (Npp8u*)getPointer(env, inputElement);
    }
    return true;
}




/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, const Npp8u* output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        output[i] = (const Npp8u*)getPointer(env, inputElement);
    }
    return true;
}




/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, Npp16s* output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        output[i] = (Npp16s*)getPointer(env, inputElement);
    }
    return true;
}




/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, const Npp16s* output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        output[i] = (const Npp16s*)getPointer(env, inputElement);
    }
    return true;
}




/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, Npp16u* output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        output[i] = (Npp16u*)getPointer(env, inputElement);
    }
    return true;
}




/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, const Npp16u* output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        output[i] = (const Npp16u*)getPointer(env, inputElement);
    }
    return true;
}




/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, Npp32s* output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        output[i] = (Npp32s*)getPointer(env, inputElement);
    }
    return true;
}




/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, const Npp32s* output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        output[i] = (const Npp32s*)getPointer(env, inputElement);
    }
    return true;
}




/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, Npp32f* output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        output[i] = (Npp32f*)getPointer(env, inputElement);
    }
    return true;
}




/**
 * Initialize the given native output array by filling it with the data from
 * the given java input array
 */
bool initNative(JNIEnv *env, jobjectArray input, const Npp32f* output[], int size)
{
    jsize arraySize = env->GetArrayLength(input);
    if (arraySize < size)
    {
        ThrowByName(env, "java/lang/ArrayIndexOutOfBoundsException",
            "Array parameter has insufficient size");
        return false;
    }
    for (int i=0; i<size; i++)
    {
        jobject inputElement = env->GetObjectArrayElement(input, i);
        if (inputElement == NULL)
        {
            ThrowByName(env, "java/lang/NullPointerException",
                "Input array contains 'null' elements");
            return false;
        }
        output[i] = (const Npp32f*)getPointer(env, inputElement);
    }
    return true;
}























/*
 * Class:     jcuda_jnpp_JNppi
 * Method:    setLogLevelNative
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_jcuda_jnpp_JNppi_setLogLevelNative
  (JNIEnv *env, jclass cla, jint logLevel)
{
    Logger::setLogLevel((LogLevel)logLevel);
}









//=== Auto-generated part: ===================================================

/**
 * <pre>
 * 8-bit unsigned image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_18u_1C1Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_8u_C1");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_8u_C1(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp8u* jniResult_native = nppiMalloc_8u_C1(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 2 channel 8-bit unsigned image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_18u_1C2Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_8u_C2");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_8u_C2(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp8u* jniResult_native = nppiMalloc_8u_C2(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 3 channel 8-bit unsigned image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_18u_1C3Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_8u_C3");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_8u_C3(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp8u* jniResult_native = nppiMalloc_8u_C3(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 4 channel 8-bit unsigned image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_18u_1C4Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_8u_C4");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_8u_C4(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp8u* jniResult_native = nppiMalloc_8u_C4(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 16-bit unsigned image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_116u_1C1Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_16u_C1");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_16u_C1(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp16u* jniResult_native = nppiMalloc_16u_C1(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 2 channel 16-bit unsigned image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_116u_1C2Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_16u_C2");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_16u_C2(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp16u* jniResult_native = nppiMalloc_16u_C2(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 3 channel 16-bit unsigned image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_116u_1C3Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_16u_C3");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_16u_C3(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp16u* jniResult_native = nppiMalloc_16u_C3(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 4 channel 16-bit unsigned image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_116u_1C4Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_16u_C4");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_16u_C4(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp16u* jniResult_native = nppiMalloc_16u_C4(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 16-bit signed image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_116s_1C1Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_16s_C1");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_16s_C1(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp16s* jniResult_native = nppiMalloc_16s_C1(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_116s_1C2Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_16s_C2");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_16s_C2(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp16s* jniResult_native = nppiMalloc_16s_C2(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 4 channel 16-bit signed image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_116s_1C4Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_16s_C4");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_16s_C4(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp16s* jniResult_native = nppiMalloc_16s_C4(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 1 channel 16-bit signed complex image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_116sc_1C1Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_16sc_C1");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_16sc_C1(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp16sc* jniResult_native = nppiMalloc_16sc_C1(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 2 channel 16-bit signed complex image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_116sc_1C2Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_16sc_C2");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_16sc_C2(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp16sc* jniResult_native = nppiMalloc_16sc_C2(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 3 channel 16-bit signed complex image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_116sc_1C3Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_16sc_C3");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_16sc_C3(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp16sc* jniResult_native = nppiMalloc_16sc_C3(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 4 channel 16-bit signed complex image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_116sc_1C4Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_16sc_C4");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_16sc_C4(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp16sc* jniResult_native = nppiMalloc_16sc_C4(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 32-bit signed image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132s_1C1Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32s_C1");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32s_C1(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32s* jniResult_native = nppiMalloc_32s_C1(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 3 channel 32-bit signed image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132s_1C3Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32s_C3");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32s_C3(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32s* jniResult_native = nppiMalloc_32s_C3(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 4 channel 32-bit signed image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132s_1C4Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32s_C4");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32s_C4(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32s* jniResult_native = nppiMalloc_32s_C4(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 32-bit integer complex image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132sc_1C1Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32sc_C1");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32sc_C1(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32sc* jniResult_native = nppiMalloc_32sc_C1(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 2 channel 32-bit integer complex image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132sc_1C2Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32sc_C2");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32sc_C2(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32sc* jniResult_native = nppiMalloc_32sc_C2(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 3 channel 32-bit integer complex image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132sc_1C3Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32sc_C3");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32sc_C3(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32sc* jniResult_native = nppiMalloc_32sc_C3(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 4 channel 32-bit integer complex image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132sc_1C4Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32sc_C4");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32sc_C4(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32sc* jniResult_native = nppiMalloc_32sc_C4(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 32-bit floating point image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132f_1C1Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32f_C1");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32f_C1(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32f* jniResult_native = nppiMalloc_32f_C1(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 2 channel 32-bit floating point image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132f_1C2Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32f_C2");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32f_C2(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32f* jniResult_native = nppiMalloc_32f_C2(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 3 channel 32-bit floating point image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132f_1C3Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32f_C3");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32f_C3(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32f* jniResult_native = nppiMalloc_32f_C3(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 4 channel 32-bit floating point image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132f_1C4Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32f_C4");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32f_C4(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32f* jniResult_native = nppiMalloc_32f_C4(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132fc_1C1Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32fc_C1");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32fc_C1(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32fc* jniResult_native = nppiMalloc_32fc_C1(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 2 channel 32-bit float complex image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132fc_1C2Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32fc_C2");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32fc_C2(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32fc* jniResult_native = nppiMalloc_32fc_C2(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 3 channel 32-bit float complex image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132fc_1C3Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32fc_C3");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32fc_C3(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32fc* jniResult_native = nppiMalloc_32fc_C3(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

/**
 * <pre>
 * 4 channel 32-bit float complex image memory allocator.
 * @param nWidthPixels Image width.
 * @param nHeightPixels Image height.
 * @param pStepBytes \ref line_step.
 * @return
 *  Pointer to new image data.
 * </pre>
 */
JNIEXPORT jobject JNICALL Java_jcuda_jnpp_JNppi_nppiMalloc_132fc_1C4Native(JNIEnv *env, jclass cls, jint nWidthPixels, jint nHeightPixels, jintArray pStepBytes)
{
    // Null-checks for non-primitive arguments
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (pStepBytes == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pStepBytes' is null for nppiMalloc_32fc_C4");
        return NULL;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMalloc_32fc_C4(nWidthPixels=%d, nHeightPixels=%d, pStepBytes=%p)\n",
        nWidthPixels, nHeightPixels, pStepBytes);

    // Native variable declarations
    int nWidthPixels_native = 0;
    int nHeightPixels_native = 0;
    int pStepBytes_native = 0;

    // Obtain native variable values
    nWidthPixels_native = (int)nWidthPixels;
    nHeightPixels_native = (int)nHeightPixels;
    // pStepBytes is set here

    // Native function call
    Npp32fc* jniResult_native = nppiMalloc_32fc_C4(nWidthPixels_native, nHeightPixels_native, &pStepBytes_native);

    // Write back native variable values
    // nWidthPixels is primitive
    // nHeightPixels is primitive
    if (!set(env, pStepBytes, 0, (jint)pStepBytes_native)) return NULL;

    // Return the result
    return createPointer(env, (jlong)jniResult_native);
}

///@} Malloc
/**
 * <pre>
 * Free method for any 2D allocated memory.
 * This method should be used to free memory allocated with any of the nppiMalloc_<modifier> methods.
 * @param pData A pointer to memory allocated using nppiMalloc_<modifier>.
 * </pre>
 */
JNIEXPORT void JNICALL Java_jcuda_jnpp_JNppi_nppiFreeNative(JNIEnv *env, jclass cls, jobject pData)
{
    // Null-checks for non-primitive arguments
    if (pData == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pData' is null for nppiFree");
        return;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiFree(pData=%p)\n",
        pData);

    // Native variable declarations
    void* pData_native;

    // Obtain native variable values
    PointerData *pData_pointerData = initPointerData(env, pData);
    if (pData_pointerData == NULL)
    {
        return;
    }
    pData_native = (void*)pData_pointerData->getPointer(env);

    // Native function call
    nppiFree(pData_native);

    // Write back native variable values
    if (!releasePointerData(env, pData_pointerData, 0)) return;

}

///@} Image-Memory Management
/**
 * <pre>
 * @defgroup image_data_exchange_and_initialization Data-Exchange and Initialization
 *
 * Primitives for initializtion, copying and converting image data.
 *
 * </pre>
 */
///@{
/**
 * <pre>
 * @name Image-Memory Set
 * Set methods for images of various types. Images are passed to these primitives via a pointer
 * to the image data (first pixel in the ROI) and a step-width, i.e. the number of bytes between
 * successive lines. The size of the area to be set (region-of-interest, ROI) is specified via
 * a NppiSize struct. 
 * In addition to the image data and ROI, all methods have a parameter to specify the value being
 * set. In case of single channel images this is a single value, in case of multi-channel, an 
 * array of values is passed. 
 * </pre>
 */
///@{
/**
 * <pre>
 * 8-bit image set.
 * @param nValue The pixel value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_18s_1C1RNative(JNIEnv *env, jclass cls, jbyte nValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_8s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_8s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_8s_C1R(nValue=%c, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        nValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8s nValue_native;
    Npp8s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nValue_native = (Npp8s)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_8s_C1R(nValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 8-bit two-channel image set.
 * @param aValue The pixel value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_18s_1C2RNative(JNIEnv *env, jclass cls, jbyteArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_8s_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_8s_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_8s_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_8s_C2R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8s aValue_native[2];
    Npp8s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 2)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_8s_C2R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 2)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 8-bit three-channel image set.
 * @param aValue The pixel value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_18s_1C3RNative(JNIEnv *env, jclass cls, jbyteArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_8s_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_8s_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_8s_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_8s_C3R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8s aValue_native[3];
    Npp8s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_8s_C3R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 8-bit four-channel image set.
 * @param aValue The pixel value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_18s_1C4RNative(JNIEnv *env, jclass cls, jbyteArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_8s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_8s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_8s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_8s_C4R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8s aValue_native[4];
    Npp8s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_8s_C4R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 8-bit four-channel image set ignoring alpha channel.
 * @param aValue The pixel value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_18s_1AC4RNative(JNIEnv *env, jclass cls, jbyteArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_8s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_8s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_8s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_8s_AC4R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8s aValue_native[3];
    Npp8s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_8s_AC4R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 8-bit unsigned image set.
 * @param nValue The pixel value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_18u_1C1RNative(JNIEnv *env, jclass cls, jbyte nValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_8u_C1R(nValue=%c, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        nValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u nValue_native;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nValue_native = (Npp8u)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_8u_C1R(nValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 8-bit unsigned image set. 
 * The 8-bit mask image affects setting of the respective pixels in the destination image.
 * If the mask value is zero (0) the pixel is not set, if the mask is non-zero, the corresponding
 * destination pixel is set to specified value.
 * @param nValue The pixel value to be set.
 * @param pDst Pointer \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_18u_1C1MRNative(JNIEnv *env, jclass cls, jbyte nValue, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_8u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_8u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_8u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_8u_C1MR(nValue=%c, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        nValue, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp8u nValue_native;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    nValue_native = (Npp8u)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_8u_C1MR(nValue_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 8-bit unsigned image set.
 * @param aValues Four-channel array containing the pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_18u_1C4RNative(JNIEnv *env, jclass cls, jbyteArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_8u_C4R(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValues, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u aValues_native[4];
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_8u_C4R(aValues_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 4 channel 8-bit unsigned image set.
 * @param aValues Four-channel array containing the pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_18u_1C4MRNative(JNIEnv *env, jclass cls, jbyteArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_8u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_8u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_8u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_8u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_8u_C4MR(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp8u aValues_native[4];
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_8u_C4MR(aValues_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 8-bit unsigned image set method, not affecting Alpha channel.
 * For RGBA images, this method allows setting of the RGB values without changing the contents
 * of the alpha-channel (fourth channel).
 * @param aValues Three-channel array containing the pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_18u_1AC4RNative(JNIEnv *env, jclass cls, jbyteArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_8u_AC4R(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValues, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u aValues_native[3];
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_8u_AC4R(aValues_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 4 channel 8-bit unsigned image set method, not affecting Alpha channel.
 * For RGBA images, this method allows setting of the RGB values without changing the contents
 * of the alpha-channel (fourth channel).
 * @param aValues Three-channel array containing the pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_18u_1AC4MRNative(JNIEnv *env, jclass cls, jbyteArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_8u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_8u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_8u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_8u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_8u_AC4MR(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp8u aValues_native[3];
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_8u_AC4MR(aValues_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 8-bit unsigned image set affecting only single channel.
 * For RGBA images, this method allows setting of a single of the four (RGBA) values 
 * without changing the contents of the other three channels. The channel is selected
 * via the pDst pointer. The pointer needs to point to the actual first value to be set,
 * e.g. in order to set the R-channel (first channel), one would pass pDst unmodified, since
 * its value actually points to the r channel. If one wanted to modify the B channel (second
 * channel), one would pass pDst + 2 to the function.
 * @param nValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_18u_1C4CRNative(JNIEnv *env, jclass cls, jbyte nValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_8u_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_8u_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_8u_C4CR(nValue=%c, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        nValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u nValue_native;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nValue_native = (Npp8u)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_8u_C4CR(nValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit unsigned image set.
 * @param nValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116u_1C1RNative(JNIEnv *env, jclass cls, jshort nValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16u_C1R(nValue=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        nValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u nValue_native;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nValue_native = (Npp16u)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16u_C1R(nValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 16-bit unsigned image set.
 * @param nValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116u_1C1MRNative(JNIEnv *env, jclass cls, jshort nValue, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_16u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16u_C1MR(nValue=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        nValue, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16u nValue_native;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    nValue_native = (Npp16u)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_16u_C1MR(nValue_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 2 channel 16-bit unsigned image set.
 * @param aValues New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116u_1C2RNative(JNIEnv *env, jclass cls, jshortArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_16u_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16u_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16u_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16u_C2R(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValues, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u aValues_native[2];
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 2)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16u_C2R(aValues_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 2)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit unsigned image set.
 * @param aValues New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116u_1C4RNative(JNIEnv *env, jclass cls, jshortArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_16u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16u_C4R(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValues, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u aValues_native[4];
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16u_C4R(aValues_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 4 channel 16-bit unsigned image set.
 * @param aValues New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116u_1C4MRNative(JNIEnv *env, jclass cls, jshortArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_16u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_16u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16u_C4MR(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16u aValues_native[4];
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_16u_C4MR(aValues_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit unsigned image set method, not affecting Alpha channel.
 * For RGBA images, this method allows setting of the RGB values without changing the contents
 * of the alpha-channel (fourth channel).
 * @param aValues Three-channel array containing the pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116u_1AC4RNative(JNIEnv *env, jclass cls, jshortArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_16u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16u_AC4R(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValues, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u aValues_native[3];
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16u_AC4R(aValues_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 4 channel 16-bit unsigned image set method, not affecting Alpha channel.
 * For RGBA images, this method allows setting of the RGB values without changing the contents
 * of the alpha-channel (fourth channel).
 * @param aValues Three-channel array containing the pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116u_1AC4MRNative(JNIEnv *env, jclass cls, jshortArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_16u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_16u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16u_AC4MR(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16u aValues_native[3];
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_16u_AC4MR(aValues_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit unsigned image set affecting only single channel.
 * For RGBA images, this method allows setting of a single of the four (RGBA) values 
 * without changing the contents of the other three channels. The channel is selected
 * via the pDst pointer. The pointer needs to point to the actual first value to be set,
 * e.g. in order to set the R-channel (first channel), one would pass pDst unmodified, since
 * its value actually points to the r channel. If one wanted to modify the B channel (second
 * channel), one would pass pDst + 2 to the function.
 * @param nValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116u_1C4CRNative(JNIEnv *env, jclass cls, jshort nValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16u_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16u_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16u_C4CR(nValue=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        nValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u nValue_native;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nValue_native = (Npp16u)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16u_C4CR(nValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit image set.
 * @param nValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116s_1C1RNative(JNIEnv *env, jclass cls, jshort nValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16s_C1R(nValue=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        nValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s nValue_native;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nValue_native = (Npp16s)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16s_C1R(nValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 16-bit image set.
 * @param nValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116s_1C1MRNative(JNIEnv *env, jclass cls, jshort nValue, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_16s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16s_C1MR(nValue=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        nValue, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16s nValue_native;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    nValue_native = (Npp16s)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_16s_C1MR(nValue_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 2 channel 16-bit image set.
 * @param aValues New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116s_1C2RNative(JNIEnv *env, jclass cls, jshortArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_16s_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16s_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16s_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16s_C2R(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValues, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s aValues_native[2];
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 2)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16s_C2R(aValues_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 2)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit image set.
 * @param aValues New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116s_1C4RNative(JNIEnv *env, jclass cls, jshortArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_16s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16s_C4R(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValues, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s aValues_native[4];
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16s_C4R(aValues_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 4 channel 16-bit image set.
 * @param aValues New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116s_1C4MRNative(JNIEnv *env, jclass cls, jshortArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_16s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_16s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16s_C4MR(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16s aValues_native[4];
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_16s_C4MR(aValues_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit image set method, not affecting Alpha channel.
 * For RGBA images, this method allows setting of the RGB values without changing the contents
 * of the alpha-channel (fourth channel).
 * @param aValues Three-channel array containing the pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116s_1AC4RNative(JNIEnv *env, jclass cls, jshortArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_16s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16s_AC4R(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValues, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s aValues_native[3];
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16s_AC4R(aValues_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 4 channel 16-bit image set method, not affecting Alpha channel.
 * For RGBA images, this method allows setting of the RGB values without changing the contents
 * of the alpha-channel (fourth channel).
 * @param aValues Three-channel array containing the pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116s_1AC4MRNative(JNIEnv *env, jclass cls, jshortArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_16s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_16s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16s_AC4MR(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16s aValues_native[3];
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_16s_AC4MR(aValues_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit unsigned image set affecting only single channel.
 * For RGBA images, this method allows setting of a single of the four (RGBA) values 
 * without changing the contents of the other three channels. The channel is selected
 * via the pDst pointer. The pointer needs to point to the actual first value to be set,
 * e.g. in order to set the R-channel (first channel), one would pass pDst unmodified, since
 * its value actually points to the r channel. If one wanted to modify the B channel (second
 * channel), one would pass pDst + 2 to the function.
 * @param nValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116s_1C4CRNative(JNIEnv *env, jclass cls, jshort nValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16s_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16s_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16s_C4CR(nValue=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        nValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s nValue_native;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nValue_native = (Npp16s)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16s_C4CR(nValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit complex integer image set.
 * @param oValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116sc_1C1RNative(JNIEnv *env, jclass cls, jobject oValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (oValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oValue' is null for nppiSet_16sc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16sc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16sc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16sc_C1R(oValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        oValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16sc oValue_native;
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, oValue, oValue_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16sc_C1R(oValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, oValue_native, oValue)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit complex integer two-channel image set.
 * @param aValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116sc_1C2RNative(JNIEnv *env, jclass cls, jobjectArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_16sc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16sc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16sc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16sc_C2R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16sc aValue_native[2];
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 2)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16sc_C2R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 2)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit complex integer three-channel image set.
 * @param aValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116sc_1C3RNative(JNIEnv *env, jclass cls, jobjectArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_16sc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16sc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16sc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16sc_C3R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16sc aValue_native[3];
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16sc_C3R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit complex integer four-channel image set ignoring alpha.
 * @param aValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116sc_1AC4RNative(JNIEnv *env, jclass cls, jobjectArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_16sc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16sc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16sc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16sc_AC4R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16sc aValue_native[3];
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16sc_AC4R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit complex integer four-channel image set.
 * @param aValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_116sc_1C4RNative(JNIEnv *env, jclass cls, jobjectArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_16sc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_16sc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_16sc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_16sc_C4R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16sc aValue_native[4];
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_16sc_C4R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 32-bit image set.
 * @param nValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132s_1C1RNative(JNIEnv *env, jclass cls, jint nValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32s_C1R(nValue=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        nValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s nValue_native;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nValue_native = (Npp32s)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32s_C1R(nValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 32-bit image set.
 * @param nValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132s_1C1MRNative(JNIEnv *env, jclass cls, jint nValue, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_32s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32s_C1MR(nValue=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        nValue, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32s nValue_native;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    nValue_native = (Npp32s)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_32s_C1MR(nValue_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 32-bit image set.
 * @param aValues New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132s_1C4RNative(JNIEnv *env, jclass cls, jintArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_32s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32s_C4R(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValues, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s aValues_native[4];
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32s_C4R(aValues_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 4 channel 32-bit image set.
 * @param aValues New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132s_1C4MRNative(JNIEnv *env, jclass cls, jintArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_32s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_32s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32s_C4MR(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32s aValues_native[4];
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_32s_C4MR(aValues_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit image set method, not affecting Alpha channel.
 * For RGBA images, this method allows setting of the RGB values without changing the contents
 * of the alpha-channel (fourth channel).
 * @param aValues Three-channel array containing the pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132s_1AC4RNative(JNIEnv *env, jclass cls, jintArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_32s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32s_AC4R(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValues, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s aValues_native[3];
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32s_AC4R(aValues_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 4 channel 16-bit image set method, not affecting Alpha channel.
 * For RGBA images, this method allows setting of the RGB values without changing the contents
 * of the alpha-channel (fourth channel).
 * @param aValues Three-channel array containing the pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132s_1AC4MRNative(JNIEnv *env, jclass cls, jintArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_32s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_32s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32s_AC4MR(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32s aValues_native[3];
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_32s_AC4MR(aValues_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 32-bit unsigned image set affecting only single channel.
 * For RGBA images, this method allows setting of a single of the four (RGBA) values 
 * without changing the contents of the other three channels. The channel is selected
 * via the pDst pointer. The pointer needs to point to the actual first value to be set,
 * e.g. in order to set the R-channel (first channel), one would pass pDst unmodified, since
 * its value actually points to the r channel. If one wanted to modify the B channel (second
 * channel), one would pass pDst + 2 to the function.
 * @param nValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132s_1C4CRNative(JNIEnv *env, jclass cls, jint nValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32s_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32s_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32s_C4CR(nValue=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        nValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s nValue_native;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nValue_native = (Npp32s)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32s_C4CR(nValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Single channel 32-bit complex integer image set.
 * @param oValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132sc_1C1RNative(JNIEnv *env, jclass cls, jobject oValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (oValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oValue' is null for nppiSet_32sc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32sc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32sc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32sc_C1R(oValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        oValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32sc oValue_native;
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, oValue, oValue_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32sc_C1R(oValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, oValue_native, oValue)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Two channel 32-bit complex integer image set.
 * @param aValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132sc_1C2RNative(JNIEnv *env, jclass cls, jobjectArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_32sc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32sc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32sc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32sc_C2R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32sc aValue_native[2];
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 2)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32sc_C2R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 2)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three channel 32-bit complex integer image set.
 * @param aValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132sc_1C3RNative(JNIEnv *env, jclass cls, jobjectArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_32sc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32sc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32sc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32sc_C3R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32sc aValue_native[3];
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32sc_C3R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four channel 32-bit complex integer image set.
 * @param aValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132sc_1C4RNative(JNIEnv *env, jclass cls, jobjectArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_32sc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32sc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32sc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32sc_C4R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32sc aValue_native[4];
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32sc_C4R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 32-bit complex integer four-channel image set ignoring alpha.
 * @param aValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132sc_1AC4RNative(JNIEnv *env, jclass cls, jobjectArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_32sc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32sc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32sc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32sc_AC4R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32sc aValue_native[3];
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32sc_AC4R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 32-bit floating point image set.
 * @param nValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132f_1C1RNative(JNIEnv *env, jclass cls, jfloat nValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32f_C1R(nValue=%f, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        nValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f nValue_native;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nValue_native = (Npp32f)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32f_C1R(nValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 32-bit floating point image set.
 * @param nValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132f_1C1MRNative(JNIEnv *env, jclass cls, jfloat nValue, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32f_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32f_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_32f_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32f_C1MR(nValue=%f, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        nValue, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32f nValue_native;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    nValue_native = (Npp32f)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_32f_C1MR(nValue_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 32-bit floating point image set.
 * @param aValues New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132f_1C4RNative(JNIEnv *env, jclass cls, jfloatArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32f_C4R(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValues, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f aValues_native[4];
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32f_C4R(aValues_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 4 channel 32-bit floating point image set.
 * @param aValues New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132f_1C4MRNative(JNIEnv *env, jclass cls, jfloatArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_32f_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32f_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32f_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_32f_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32f_C4MR(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32f aValues_native[4];
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_32f_C4MR(aValues_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 32-bit floating point image set method, not affecting Alpha channel.
 * For RGBA images, this method allows setting of the RGB values without changing the contents
 * of the alpha-channel (fourth channel).
 * @param aValues Three-channel array containing the pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132f_1AC4RNative(JNIEnv *env, jclass cls, jfloatArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32f_AC4R(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValues, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f aValues_native[3];
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32f_AC4R(aValues_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Masked 4 channel 32-bit floating point image set method, not affecting Alpha channel.
 * For RGBA images, this method allows setting of the RGB values without changing the contents
 * of the alpha-channel (fourth channel).
 * @param aValues Three-channel array containing the pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
 * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132f_1AC4MRNative(JNIEnv *env, jclass cls, jfloatArray aValues, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (aValues == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValues' is null for nppiSet_32f_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32f_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32f_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiSet_32f_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32f_AC4MR(aValues=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32f aValues_native[3];
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    if (!initNative(env, aValues, aValues_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiSet_32f_AC4MR(aValues_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releaseNative(env, aValues_native, aValues, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 32-bit floating point image set affecting only single channel.
 * For RGBA images, this method allows setting of a single of the four (RGBA) values 
 * without changing the contents of the other three channels. The channel is selected
 * via the pDst pointer. The pointer needs to point to the actual first value to be set,
 * e.g. in order to set the R-channel (first channel), one would pass pDst unmodified, since
 * its value actually points to the r channel. If one wanted to modify the B channel (second
 * channel), one would pass pDst + 2 to the function.
 * @param nValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132f_1C4CRNative(JNIEnv *env, jclass cls, jfloat nValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nValue is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32f_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32f_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32f_C4CR(nValue=%f, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        nValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f nValue_native;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nValue_native = (Npp32f)nValue;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32f_C4CR(nValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nValue is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Single channel 32-bit complex image set.
 * @param oValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132fc_1C1RNative(JNIEnv *env, jclass cls, jobject oValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (oValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oValue' is null for nppiSet_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32fc_C1R(oValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        oValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc oValue_native;
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, oValue, oValue_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32fc_C1R(oValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, oValue_native, oValue)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Two channel 32-bit complex image set.
 * @param aValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132fc_1C2RNative(JNIEnv *env, jclass cls, jobjectArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_32fc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32fc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32fc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32fc_C2R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc aValue_native[2];
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 2)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32fc_C2R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 2)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three channel 32-bit complex image set.
 * @param aValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132fc_1C3RNative(JNIEnv *env, jclass cls, jobjectArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_32fc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32fc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32fc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32fc_C3R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc aValue_native[3];
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32fc_C3R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four channel 32-bit complex image set.
 * @param aValue The pixel-value to be set.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132fc_1C4RNative(JNIEnv *env, jclass cls, jobjectArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_32fc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32fc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32fc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32fc_C4R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc aValue_native[4];
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32fc_C4R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 32-bit complex four-channel image set ignoring alpha.
 * @param aValue New pixel value.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSet_132fc_1AC4RNative(JNIEnv *env, jclass cls, jobjectArray aValue, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiSet_32fc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiSet_32fc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSet_32fc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSet_32fc_AC4R(aValue=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aValue, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc aValue_native[3];
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aValue, aValue_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSet_32fc_AC4R(aValue_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, aValue_native, aValue, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

///@}
/**
 * <pre>
 * @name Image-Memory Copy
 * Copy methods for images of various types. In addition to routines for
 * copying pixels of identical layout from one image to another, there are
 * copy routines for select channels as well as packed-planar conversions:
 * - Select channel to multi-channel copy. E.g. given a three-channel source
 *      and destination image one may copy the second channel of the source
 *      to the third channel of the destination.
 * - Single channel to multi-channel copy. E.g. given a single-channel source
 *      and a four-channel destination, one may copy the contents of the single-
 *      channel source to the second channel of the destination.
 * - Select channel to single-channel copy. E.g. given a three-channel source
 *      and a single-channel destination one may copy the third channel of the 
 *      source to the destination.
 * - Multi-channel to planar copy. These copy operations split a multi-channel
 *      image into a set of single-channel images.
 * - Planar image to multi-channel copy. These copy routines combine separate
 *      color-planes (single channel images) into a single multi-channel image.
 * 
 * </pre>
 */
///@{
/**
 * <pre>
 * 8-bit image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18s_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8s_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8s_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Two-channel 8-bit image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18s_1C2RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8s_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8s_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8s_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8s_C2R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8s_C2R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 8-bit image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18s_1C3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8s_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8s_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8s_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8s_C3R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8s_C3R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 8-bit image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18s_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8s_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8s_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 8-bit image copy, ignoring alpha channel.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18s_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8s_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8s_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 8-bit unsigned image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 8-bit unsigned image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 8-bit unsigned image copy, not affecting Alpha channel.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit unsigned image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit unsigned image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit unsigned image copy, not affecting Alpha channel.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit image copy, not affecting Alpha.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit complex image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116sc_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16sc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16sc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16sc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16sc_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16sc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16sc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16sc_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Two-channel 16-bit complex image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116sc_1C2RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16sc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16sc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16sc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16sc_C2R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16sc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16sc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16sc_C2R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 16-bit complex image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116sc_1C3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16sc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16sc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16sc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16sc_C3R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16sc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16sc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16sc_C3R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 16-bit complex image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116sc_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16sc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16sc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16sc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16sc_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16sc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16sc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16sc_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 16-bit complex image copy, ignoring alpha.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116sc_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16sc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16sc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16sc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16sc_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16sc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16sc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16sc_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 32-bit image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 32-bit image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 32-bit image copy, not affecting Alpha.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 32-bit complex image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132sc_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32sc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32sc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32sc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32sc_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32sc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32sc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32sc_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Two-channel 32-bit complex image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132sc_1C2RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32sc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32sc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32sc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32sc_C2R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32sc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32sc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32sc_C2R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 32-bit complex image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132sc_1C3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32sc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32sc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32sc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32sc_C3R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32sc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32sc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32sc_C3R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 32-bit complex image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132sc_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32sc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32sc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32sc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32sc_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32sc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32sc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32sc_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 32-bit complex image copy, ignoring alpha.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132sc_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32sc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32sc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32sc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32sc_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32sc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32sc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32sc_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 32-bit floating point image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 32-bit floating point image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 32-bit floating point image copy, not affecting Alpha.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 32-bit floating-point complex image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132fc_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32fc_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32fc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32fc_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Two-channel 32-bit floating-point complex image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132fc_1C2RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32fc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32fc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32fc_C2R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32fc_C2R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32fc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32fc_C2R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 32-bit floating-point complex image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132fc_1C3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32fc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32fc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32fc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32fc_C3R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32fc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32fc_C3R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 32-bit floating-point complex image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132fc_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32fc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32fc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32fc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32fc_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32fc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32fc_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 32-bit floating-point complex image copy, ignoring alpha.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132fc_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32fc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32fc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32fc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32fc_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32fc*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32fc_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation 8-bit unsigned image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C1MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_8u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C1MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C1MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation three channel 8-bit unsigned image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C3MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_8u_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C3MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C3MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation four channel 8-bit unsigned image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C4MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_8u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C4MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C4MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation four channel 8-bit unsigned image copy, ignoring alpha.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1AC4MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_8u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_AC4MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_AC4MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation 16-bit unsigned image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C1MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_16u_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C1MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C1MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation three channel 16-bit unsigned image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C3MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_16u_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C3MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C3MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation four channel 16-bit unsigned image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C4MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_16u_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C4MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C4MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation four channel 16-bit unsigned image copy, ignoring alpha.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1AC4MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_16u_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_AC4MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_AC4MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation 16-bit signed image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C1MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_16s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C1MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C1MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation three channel 16-bit signed image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C3MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_16s_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C3MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C3MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation four channel 16-bit signed image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C4MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_16s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C4MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C4MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation four channel 16-bit signed image copy, ignoring alpha.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1AC4MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_16s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_AC4MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_AC4MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation 32-bit signed image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C1MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_32s_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C1MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C1MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation three channel 32-bit signed image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C3MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_32s_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C3MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C3MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation four channel 32-bit signed image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C4MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_32s_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C4MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C4MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation four channel 32-bit signed image copy, ignoring alpha.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1AC4MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_32s_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_AC4MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_AC4MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation 32-bit float image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C1MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_32f_C1MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C1MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C1MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation three channel 32-bit float image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C3MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_32f_C3MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C3MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C3MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation four channel 32-bit float image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C4MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_32f_C4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C4MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C4MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * \ref masked_operation four channel 32-bit float image copy, ignoring alpha.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param pMask \ref mask_image_pointer.
 * @param nMaskStep \ref mask_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1AC4MRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jobject pMask, jint nMaskStep)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pMask == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pMask' is null for nppiCopy_32f_AC4MR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nMaskStep is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_AC4MR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, pMask=%p, nMaskStep=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    Npp8u* pMask_native = NULL;
    int nMaskStep_native = 0;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pMask_pointerData = initPointerData(env, pMask);
    if (pMask_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pMask_native = (Npp8u*)pMask_pointerData->getPointer(env);
    nMaskStep_native = (int)nMaskStep;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_AC4MR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, pMask_native, nMaskStep_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releasePointerData(env, pMask_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nMaskStep is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Select-channel 8-bit unsigned image copy for three-channel images.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref select_destination_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C3CRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C3CR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C3CR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Select-channel 8-bit unsigned image copy for four-channel images.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref select_destination_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C4CRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C4CR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C4CR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Select-channel 16-bit signed image copy for three-channel images.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref select_destination_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C3CRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C3CR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C3CR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Select-channel 16-bit signed image copy for four-channel images.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref select_destination_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C4CRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C4CR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C4CR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Select-channel 16-bit unsigned image copy for three-channel images.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref select_destination_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C3CRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C3CR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C3CR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Select-channel 16-bit unsigned image copy for four-channel images.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref select_destination_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C4CRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C4CR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C4CR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Select-channel 32-bit signed image copy for three-channel images.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref select_destination_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C3CRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C3CR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C3CR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Select-channel 32-bit signed image copy for four-channel images.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref select_destination_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C4CRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C4CR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C4CR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Select-channel 32-bit float image copy for three-channel images.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref select_destination_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C3CRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C3CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C3CR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C3CR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Select-channel 32-bit float image copy for four-channel images.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref select_destination_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C4CRNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C4CR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C4CR(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C4CR(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel to single-channel 8-bit unsigned image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C3C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C3C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C3C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel to single-channel 8-bit unsigned image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C4C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C4C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C4C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel to single-channel 16-bit signed image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C3C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C3C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C3C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel to single-channel 16-bit signed image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C4C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C4C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C4C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel to single-channel 16-bit unsigned image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C3C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C3C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C3C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel to single-channel 16-bit unsigned image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C4C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C4C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C4C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel to single-channel 32-bit signed image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C3C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C3C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C3C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel to single-channel 32-bit signed image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C4C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C4C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C4C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel to single-channel 32-bit float image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C3C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C3C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C3C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C3C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel to single-channel 32-bit float image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C4C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C4C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C4C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C4C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Single-channel to three-channel 8-bit unsigned image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C1C3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C1C3R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C1C3R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Single-channel to four-channel 8-bit unsigned image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C1C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C1C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Single-channel to three-channel 16-bit signed image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C1C3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C1C3R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C1C3R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Single-channel to four-channel 16-bit signed image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C1C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C1C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Single-channel to three-channel 16-bit unsigned image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C1C3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C1C3R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C1C3R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Single-channel to four-channel 16-bit unsigned image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C1C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C1C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Single-channel to three-channel 32-bit signed image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C1C3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C1C3R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C1C3R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Single-channel to four-channel 32-bit signed image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C1C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C1C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Single-channel to three-channel 32-bit float image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C1C3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C1C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C1C3R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C1C3R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Single-channel to four-channel 32-bit float image copy.
 * @param pSrc \ref select_source_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C1C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C1C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C1C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 8-bit unsigned packed to planar image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param aDst Planar \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C3P3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobjectArray aDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (aDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aDst' is null for nppiCopy_8u_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C3P3R(pSrc=%p, nSrcStep=%d, aDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, aDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* aDst_native[3];
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, aDst, aDst_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C3P3R(pSrc_native, nSrcStep_native, aDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // aDst is an array of native pointers
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 8-bit unsigned packed to planar image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param aDst Planar \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1C4P4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobjectArray aDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_8u_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (aDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aDst' is null for nppiCopy_8u_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_C4P4R(pSrc=%p, nSrcStep=%d, aDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, aDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* aDst_native[4];
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, aDst, aDst_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_C4P4R(pSrc_native, nSrcStep_native, aDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // aDst is an array of native pointers
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 16-bit signed packed to planar image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param aDst Planar \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C3P3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobjectArray aDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (aDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aDst' is null for nppiCopy_16s_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C3P3R(pSrc=%p, nSrcStep=%d, aDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, aDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* aDst_native[3];
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, aDst, aDst_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C3P3R(pSrc_native, nSrcStep_native, aDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // aDst is an array of native pointers
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 16-bit signed packed to planar image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param aDst Planar \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1C4P4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobjectArray aDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16s_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (aDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aDst' is null for nppiCopy_16s_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_C4P4R(pSrc=%p, nSrcStep=%d, aDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, aDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* aDst_native[4];
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, aDst, aDst_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_C4P4R(pSrc_native, nSrcStep_native, aDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // aDst is an array of native pointers
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 16-bit unsigned packed to planar image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param aDst Planar \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C3P3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobjectArray aDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (aDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aDst' is null for nppiCopy_16u_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C3P3R(pSrc=%p, nSrcStep=%d, aDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, aDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* aDst_native[3];
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, aDst, aDst_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C3P3R(pSrc_native, nSrcStep_native, aDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // aDst is an array of native pointers
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 16-bit unsigned packed to planar image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param aDst Planar \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1C4P4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobjectArray aDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_16u_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (aDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aDst' is null for nppiCopy_16u_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_C4P4R(pSrc=%p, nSrcStep=%d, aDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, aDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* aDst_native[4];
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, aDst, aDst_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_C4P4R(pSrc_native, nSrcStep_native, aDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // aDst is an array of native pointers
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 32-bit signed packed to planar image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param aDst Planar \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C3P3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobjectArray aDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (aDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aDst' is null for nppiCopy_32s_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C3P3R(pSrc=%p, nSrcStep=%d, aDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, aDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* aDst_native[3];
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, aDst, aDst_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C3P3R(pSrc_native, nSrcStep_native, aDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // aDst is an array of native pointers
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 32-bit signed packed to planar image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param aDst Planar \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1C4P4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobjectArray aDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32s_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (aDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aDst' is null for nppiCopy_32s_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_C4P4R(pSrc=%p, nSrcStep=%d, aDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, aDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* aDst_native[4];
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, aDst, aDst_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_C4P4R(pSrc_native, nSrcStep_native, aDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // aDst is an array of native pointers
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 32-bit float packed to planar image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param aDst Planar \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C3P3RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobjectArray aDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (aDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aDst' is null for nppiCopy_32f_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C3P3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C3P3R(pSrc=%p, nSrcStep=%d, aDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, aDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* aDst_native[3];
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, aDst, aDst_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C3P3R(pSrc_native, nSrcStep_native, aDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // aDst is an array of native pointers
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 32-bit float packed to planar image copy.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param aDst Planar \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1C4P4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobjectArray aDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopy_32f_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (aDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aDst' is null for nppiCopy_32f_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_C4P4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_C4P4R(pSrc=%p, nSrcStep=%d, aDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, aDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* aDst_native[4];
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, aDst, aDst_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_C4P4R(pSrc_native, nSrcStep_native, aDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // aDst is an array of native pointers
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 8-bit unsigned planar to packed image copy.
 * @param aSrc Planar \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1P3C3RNative(JNIEnv *env, jclass cls, jobjectArray aSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aSrc' is null for nppiCopy_8u_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_P3C3R(aSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    const Npp8u* aSrc_native[3];
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aSrc, aSrc_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_P3C3R(aSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // aSrc is an array of native pointers
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 8-bit unsigned planar to packed image copy.
 * @param aSrc Planar \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_18u_1P4C4RNative(JNIEnv *env, jclass cls, jobjectArray aSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aSrc' is null for nppiCopy_8u_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_8u_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_8u_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_8u_P4C4R(aSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    const Npp8u* aSrc_native[4];
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aSrc, aSrc_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_8u_P4C4R(aSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // aSrc is an array of native pointers
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 16-bit unsigned planar to packed image copy.
 * @param aSrc Planar \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1P3C3RNative(JNIEnv *env, jclass cls, jobjectArray aSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aSrc' is null for nppiCopy_16u_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_P3C3R(aSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    const Npp16u* aSrc_native[3];
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aSrc, aSrc_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_P3C3R(aSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // aSrc is an array of native pointers
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 16-bit unsigned planar to packed image copy.
 * @param aSrc Planar \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116u_1P4C4RNative(JNIEnv *env, jclass cls, jobjectArray aSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aSrc' is null for nppiCopy_16u_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16u_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16u_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16u_P4C4R(aSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    const Npp16u* aSrc_native[4];
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aSrc, aSrc_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16u_P4C4R(aSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // aSrc is an array of native pointers
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 16-bit signed planar to packed image copy.
 * @param aSrc Planar \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1P3C3RNative(JNIEnv *env, jclass cls, jobjectArray aSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aSrc' is null for nppiCopy_16s_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_P3C3R(aSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    const Npp16s* aSrc_native[3];
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aSrc, aSrc_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_P3C3R(aSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // aSrc is an array of native pointers
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 16-bit signed planar to packed image copy.
 * @param aSrc Planar \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_116s_1P4C4RNative(JNIEnv *env, jclass cls, jobjectArray aSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aSrc' is null for nppiCopy_16s_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_16s_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_16s_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_16s_P4C4R(aSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    const Npp16s* aSrc_native[4];
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aSrc, aSrc_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_16s_P4C4R(aSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // aSrc is an array of native pointers
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 32-bit signed planar to packed image copy.
 * @param aSrc Planar \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1P3C3RNative(JNIEnv *env, jclass cls, jobjectArray aSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aSrc' is null for nppiCopy_32s_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_P3C3R(aSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    const Npp32s* aSrc_native[3];
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aSrc, aSrc_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_P3C3R(aSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // aSrc is an array of native pointers
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 32-bit signed planar to packed image copy.
 * @param aSrc Planar \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132s_1P4C4RNative(JNIEnv *env, jclass cls, jobjectArray aSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aSrc' is null for nppiCopy_32s_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32s_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32s_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32s_P4C4R(aSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    const Npp32s* aSrc_native[4];
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aSrc, aSrc_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32s_P4C4R(aSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // aSrc is an array of native pointers
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three-channel 32-bit float planar to packed image copy.
 * @param aSrc Planar \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1P3C3RNative(JNIEnv *env, jclass cls, jobjectArray aSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aSrc' is null for nppiCopy_32f_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_P3C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_P3C3R(aSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    const Npp32f* aSrc_native[3];
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aSrc, aSrc_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_P3C3R(aSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // aSrc is an array of native pointers
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four-channel 32-bit float planar to packed image copy.
 * @param aSrc Planar \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopy_132f_1P4C4RNative(JNIEnv *env, jclass cls, jobjectArray aSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (aSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aSrc' is null for nppiCopy_32f_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopy_32f_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiCopy_32f_P4C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopy_32f_P4C4R(aSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        aSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    const Npp32f* aSrc_native[4];
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, aSrc, aSrc_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopy_32f_P4C4R(aSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    // aSrc is an array of native pointers
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

///@}
/**
 * <pre>
 * @name Bit-Depth Conversion
 * Convert bit-depth up and down.
 *
 * The integer conversion methods do not involve any scaling. Conversions that reduce bit-depth saturate
 * values exceeding the reduced range to the range's maximum/minimum value.
 * When converting from floating-point values to integer values, a rounding mode can be specified. After rounding
 * to integer values the values get saturated to the destination data type's range.
 * </pre>
 */
///@{
/**
 * <pre>
 * 8-bit unsigned to 16-bit unsigned conversion.
 *
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_18u16u_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_8u16u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_8u16u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_8u16u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_8u16u_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_8u16u_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit unsigned to 8-bit unsigned conversion.
 * For detailed documentation see nppiConvert_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_116u8u_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_16u8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_16u8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_16u8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_16u8u_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_16u8u_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 8-bit unsigned to 16-bit unsigned  conversion.
 * For detailed documentation see nppiConvert_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_18u16u_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_8u16u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_8u16u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_8u16u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_8u16u_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_8u16u_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit unsigned to 8-bit unsigned conversion.
 * For detailed documentation see nppiConvert_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_116u8u_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_16u8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_16u8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_16u8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_16u8u_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_16u8u_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 8-bit unsigned to 16-bit unsigned conversion, not affecting Alpha.
 * For detailed documentation see nppiConvert_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_18u16u_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_8u16u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_8u16u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_8u16u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_8u16u_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_8u16u_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit unsigned to 8-bit unsigned conversion, not affecting Alpha.
 * For detailed documentation see nppiConvert_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_116u8u_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_16u8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_16u8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_16u8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_16u8u_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_16u8u_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 8-bit unsigned to 16-bit signed conversion.
 * For detailed documentation see nppiConvert_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_18u16s_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_8u16s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_8u16s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_8u16s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_8u16s_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_8u16s_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit signed to 8-bit unsigned conversion.
 * For detailed documentation see nppiConvert_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_116s8u_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_16s8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_16s8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_16s8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_16s8u_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_16s8u_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 8-bit unsigned to 16-bit signed conversion.
 * For detailed documentation see nppiConvert_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_18u16s_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_8u16s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_8u16s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_8u16s_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_8u16s_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_8u16s_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit signed to 8-bit unsignedconversion, not affecting Alpha.
 * For detailed documentation see nppiConvert_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_116s8u_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_16s8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_16s8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_16s8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_16s8u_C4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_16s8u_C4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 8-bit unsigned to 16-bit signed conversion, not affecting Alpha.
 * For detailed documentation see nppiConverte_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_18u16s_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_8u16s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_8u16s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_8u16s_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_8u16s_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_8u16s_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 16-bit signed to 8-bit unsigned conversion, not affecting Alpha.
 * For detailed documentation see nppiConverte_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_116s8u_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_16s8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_16s8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_16s8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_16s8u_AC4R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_16s8u_AC4R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit singedto 32-bit floating point conversion.
 * For detailed documentation see nppiConverte_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_116s32f_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_16s32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_16s32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_16s32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_16s32f_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_16s32f_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 32-bit floating point to 16-bit conversion.
 * For detailed documentation see nppiConverte_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param eRoundMode Flag specifying how fractional float values are rounded to integer values.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_132f16s_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jint eRoundMode)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_32f16s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_32f16s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_32f16s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // eRoundMode is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_32f16s_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, eRoundMode=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, eRoundMode);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    NppRoundMode eRoundMode_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    eRoundMode_native = (NppRoundMode)eRoundMode;

    // Native function call
    NppStatus jniResult_native = nppiConvert_32f16s_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, eRoundMode_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // eRoundMode is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 8-bit unsigned to 32-bit floating point conversion.
 * For detailed documentation see nppiConverte_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_18u32f_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_8u32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_8u32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_8u32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_8u32f_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_8u32f_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit unsigned to 32-bit floating point conversion.
 * For detailed documentation see nppiConverte_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_116u32f_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_16u32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_16u32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_16u32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_16u32f_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_16u32f_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 32-bit floating point to 16-bit unsigned conversion.
 * For detailed documentation see nppiConverte_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param eRoundMode Flag specifying how fractional float values are rounded to integer values.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_132f16u_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jint eRoundMode)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_32f16u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_32f16u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_32f16u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // eRoundMode is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_32f16u_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, eRoundMode=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, eRoundMode);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    NppRoundMode eRoundMode_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    eRoundMode_native = (NppRoundMode)eRoundMode;

    // Native function call
    NppStatus jniResult_native = nppiConvert_32f16u_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, eRoundMode_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // eRoundMode is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 32-bit floating point to 8-bit unsigned conversion.
 * For detailed documentation see nppiConverte_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param eRoundMode Flag specifying how fractional float values are rounded to integer values.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_132f8u_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI, jint eRoundMode)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_32f8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_32f8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_32f8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // eRoundMode is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_32f8u_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, eRoundMode=%d)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI, eRoundMode);

    // Native variable declarations
    Npp32f* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    NppRoundMode eRoundMode_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32f*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    eRoundMode_native = (NppRoundMode)eRoundMode;

    // Native function call
    NppStatus jniResult_native = nppiConvert_32f8u_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native, eRoundMode_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // eRoundMode is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit unsigned to 32-bit signed conversion.
 * For detailed documentation see nppiConverte_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_116u32s_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_16u32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_16u32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_16u32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_16u32s_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_16u32s_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 16-bit to 32-bit  conversion.
 * For detailed documentation see nppiConvert_8u16u_C1R().
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiConvert_116s32s_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiConvert_16s32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiConvert_16s32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiConvert_16s32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiConvert_16s32s_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp16s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp16s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiConvert_16s32s_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

///@}
/** @name Copy Const Border
 * Methods for copying images and padding borders with a constant, user-specifiable color.
 */
///@{
/**
 * <pre>
 * 8-bit unsigned image copy width constant border color.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param oSrcSizeROI Size of the source region of pixels.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oDstSizeROI Size (width, height) of the destination region, i.e. the region that gets filled with
 *      data from the source image (inner part) and constant border color (outer part).
 * @param nTopBorderHeight Height (in pixels) of the top border. The height of the border at the bottom of the
 *      destination ROI is implicitly defined by the size of the source ROI: 
 *      nBottomBorderHeight = oDstSizeROI.height - nTopBorderHeight - oSrcSizeROI.height.
 * @param nLeftBorderWidth Width (in pixels) of the left border. The width of the border at the right side of the
 *      destination ROI is implicitly defined by the size of the source ROI:
 *      nRightBorderWidth = oDstSizeROI.width - nLeftBorderWidth - oSrcSizeROI.width.
 * @param nValue The pixel value to be set for border pixels.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopyConstBorder_18u_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject oSrcSizeROI, jobject pDst, jint nDstStep, jobject oDstSizeROI, jint nTopBorderHeight, jint nLeftBorderWidth, jbyte nValue)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopyConstBorder_8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (oSrcSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSrcSizeROI' is null for nppiCopyConstBorder_8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopyConstBorder_8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oDstSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oDstSizeROI' is null for nppiCopyConstBorder_8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nTopBorderHeight is primitive
    // nLeftBorderWidth is primitive
    // nValue is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopyConstBorder_8u_C1R(pSrc=%p, nSrcStep=%d, oSrcSizeROI=%p, pDst=%p, nDstStep=%d, oDstSizeROI=%p, nTopBorderHeight=%d, nLeftBorderWidth=%d, nValue=%c)\n",
        pSrc, nSrcStep, oSrcSizeROI, pDst, nDstStep, oDstSizeROI, nTopBorderHeight, nLeftBorderWidth, nValue);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    NppiSize oSrcSizeROI_native;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oDstSizeROI_native;
    int nTopBorderHeight_native = 0;
    int nLeftBorderWidth_native = 0;
    Npp8u nValue_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, oSrcSizeROI, oSrcSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oDstSizeROI, oDstSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nTopBorderHeight_native = (int)nTopBorderHeight;
    nLeftBorderWidth_native = (int)nLeftBorderWidth;
    nValue_native = (Npp8u)nValue;

    // Native function call
    NppStatus jniResult_native = nppiCopyConstBorder_8u_C1R(pSrc_native, nSrcStep_native, oSrcSizeROI_native, pDst_native, nDstStep_native, oDstSizeROI_native, nTopBorderHeight_native, nLeftBorderWidth_native, nValue_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // oSrcSizeROI is a read-only structure
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oDstSizeROI is a read-only structure
    // nTopBorderHeight is primitive
    // nLeftBorderWidth is primitive
    // nValue is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4channel 8-bit unsigned image copy with constant border color.
 * See nppiCopyConstBorder_8u_C1R() for detailed documentation.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param oSrcSizeROI Size of the source region-of-interest.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oDstSizeROI Size of the destination region-of-interest.
 * @param nTopBorderHeight Height of top border.
 * @param nLeftBorderWidth Width of left border.
 * @param aValue Vector of the RGBA values of the border pixels to be set.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopyConstBorder_18u_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject oSrcSizeROI, jobject pDst, jint nDstStep, jobject oDstSizeROI, jint nTopBorderHeight, jint nLeftBorderWidth, jbyteArray aValue)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopyConstBorder_8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (oSrcSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSrcSizeROI' is null for nppiCopyConstBorder_8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopyConstBorder_8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oDstSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oDstSizeROI' is null for nppiCopyConstBorder_8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nTopBorderHeight is primitive
    // nLeftBorderWidth is primitive
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiCopyConstBorder_8u_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopyConstBorder_8u_C4R(pSrc=%p, nSrcStep=%d, oSrcSizeROI=%p, pDst=%p, nDstStep=%d, oDstSizeROI=%p, nTopBorderHeight=%d, nLeftBorderWidth=%d, aValue=%p)\n",
        pSrc, nSrcStep, oSrcSizeROI, pDst, nDstStep, oDstSizeROI, nTopBorderHeight, nLeftBorderWidth, aValue);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    NppiSize oSrcSizeROI_native;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oDstSizeROI_native;
    int nTopBorderHeight_native = 0;
    int nLeftBorderWidth_native = 0;
    Npp8u aValue_native[4];

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, oSrcSizeROI, oSrcSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oDstSizeROI, oDstSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nTopBorderHeight_native = (int)nTopBorderHeight;
    nLeftBorderWidth_native = (int)nLeftBorderWidth;
    if (!initNative(env, aValue, aValue_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopyConstBorder_8u_C4R(pSrc_native, nSrcStep_native, oSrcSizeROI_native, pDst_native, nDstStep_native, oDstSizeROI_native, nTopBorderHeight_native, nLeftBorderWidth_native, aValue_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // oSrcSizeROI is a read-only structure
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oDstSizeROI is a read-only structure
    // nTopBorderHeight is primitive
    // nLeftBorderWidth is primitive
    if (!releaseNative(env, aValue_native, aValue, 4)) return JNPP_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 4 channel 8-bit unsigned image copy with constant border color.
 * See nppiCopyConstBorder_8u_C1R() for detailed documentation.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param oSrcSizeROI Size of the source region-of-interest.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oDstSizeROI Size of the destination region-of-interest.
 * @param nTopBorderHeight Height of top border.
 * @param nLeftBorderWidth Width of left border.
 * @param aValue Vector of the RGB values of the border pixels. Because this method does not
 *      affect the destination image's alpha channel, only three components of the border color
 *      are needed.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopyConstBorder_18u_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject oSrcSizeROI, jobject pDst, jint nDstStep, jobject oDstSizeROI, jint nTopBorderHeight, jint nLeftBorderWidth, jbyteArray aValue)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopyConstBorder_8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (oSrcSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSrcSizeROI' is null for nppiCopyConstBorder_8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopyConstBorder_8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oDstSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oDstSizeROI' is null for nppiCopyConstBorder_8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nTopBorderHeight is primitive
    // nLeftBorderWidth is primitive
    if (aValue == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aValue' is null for nppiCopyConstBorder_8u_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopyConstBorder_8u_AC4R(pSrc=%p, nSrcStep=%d, oSrcSizeROI=%p, pDst=%p, nDstStep=%d, oDstSizeROI=%p, nTopBorderHeight=%d, nLeftBorderWidth=%d, aValue=%p)\n",
        pSrc, nSrcStep, oSrcSizeROI, pDst, nDstStep, oDstSizeROI, nTopBorderHeight, nLeftBorderWidth, aValue);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    NppiSize oSrcSizeROI_native;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oDstSizeROI_native;
    int nTopBorderHeight_native = 0;
    int nLeftBorderWidth_native = 0;
    Npp8u aValue_native[3];

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, oSrcSizeROI, oSrcSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oDstSizeROI, oDstSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nTopBorderHeight_native = (int)nTopBorderHeight;
    nLeftBorderWidth_native = (int)nLeftBorderWidth;
    if (!initNative(env, aValue, aValue_native, 3)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiCopyConstBorder_8u_AC4R(pSrc_native, nSrcStep_native, oSrcSizeROI_native, pDst_native, nDstStep_native, oDstSizeROI_native, nTopBorderHeight_native, nLeftBorderWidth_native, aValue_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // oSrcSizeROI is a read-only structure
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oDstSizeROI is a read-only structure
    // nTopBorderHeight is primitive
    // nLeftBorderWidth is primitive
    if (!releaseNative(env, aValue_native, aValue, 3)) return JNPP_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * 32-bit image copy with constant border color.
 * See nppiCopyConstBorder_8u_C1R() for detailed documentation.
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param oSrcSizeROI Size of the source region-of-interest.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oDstSizeROI Size of the destination region-of-interest.
 * @param nTopBorderHeight Height of top border.
 * @param nLeftBorderWidth Width of left border.
 * @param nValue Border luminance value.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiCopyConstBorder_132s_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject oSrcSizeROI, jobject pDst, jint nDstStep, jobject oDstSizeROI, jint nTopBorderHeight, jint nLeftBorderWidth, jint nValue)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiCopyConstBorder_32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (oSrcSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSrcSizeROI' is null for nppiCopyConstBorder_32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiCopyConstBorder_32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oDstSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oDstSizeROI' is null for nppiCopyConstBorder_32s_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nTopBorderHeight is primitive
    // nLeftBorderWidth is primitive
    // nValue is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiCopyConstBorder_32s_C1R(pSrc=%p, nSrcStep=%d, oSrcSizeROI=%p, pDst=%p, nDstStep=%d, oDstSizeROI=%p, nTopBorderHeight=%d, nLeftBorderWidth=%d, nValue=%d)\n",
        pSrc, nSrcStep, oSrcSizeROI, pDst, nDstStep, oDstSizeROI, nTopBorderHeight, nLeftBorderWidth, nValue);

    // Native variable declarations
    Npp32s* pSrc_native = NULL;
    int nSrcStep_native = 0;
    NppiSize oSrcSizeROI_native;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oDstSizeROI_native;
    int nTopBorderHeight_native = 0;
    int nLeftBorderWidth_native = 0;
    Npp32s nValue_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp32s*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    if (!initNative(env, oSrcSizeROI, oSrcSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oDstSizeROI, oDstSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nTopBorderHeight_native = (int)nTopBorderHeight;
    nLeftBorderWidth_native = (int)nLeftBorderWidth;
    nValue_native = (Npp32s)nValue;

    // Native function call
    NppStatus jniResult_native = nppiCopyConstBorder_32s_C1R(pSrc_native, nSrcStep_native, oSrcSizeROI_native, pDst_native, nDstStep_native, oDstSizeROI_native, nTopBorderHeight_native, nLeftBorderWidth_native, nValue_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    // oSrcSizeROI is a read-only structure
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oDstSizeROI is a read-only structure
    // nTopBorderHeight is primitive
    // nLeftBorderWidth is primitive
    // nValue is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

///@}
/**
 * <pre>
 * @name Image Transpose
 * Methods for transposing images of various types. Like matrix transpose, image transpose is a mirror along the image's
 * diagonal (upper-left to lower-right corner).
 * </pre>
 */
///@{
/**
 * <pre>
 * 8-bit image transpose.
 *
 * @param pSrc \ref source_image_pointer.
 * @param nSrcStep \ref source_image_line_step.
 * @param pDst Pointer to the destination ROI.
 * @param nDstStep \ref destination_image_line_step.
 * @param oROI \ref roi_specification.
 *
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiTranspose_18u_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc, jint nSrcStep, jobject pDst, jint nDstStep, jobject oROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc' is null for nppiTranspose_8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcStep is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiTranspose_8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oROI' is null for nppiTranspose_8u_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiTranspose_8u_C1R(pSrc=%p, nSrcStep=%d, pDst=%p, nDstStep=%d, oROI=%p)\n",
        pSrc, nSrcStep, pDst, nDstStep, oROI);

    // Native variable declarations
    Npp8u* pSrc_native = NULL;
    int nSrcStep_native = 0;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oROI_native;

    // Obtain native variable values
    PointerData *pSrc_pointerData = initPointerData(env, pSrc);
    if (pSrc_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc_native = (Npp8u*)pSrc_pointerData->getPointer(env);
    nSrcStep_native = (int)nSrcStep;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oROI, oROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiTranspose_8u_C1R(pSrc_native, nSrcStep_native, pDst_native, nDstStep_native, oROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcStep is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

///@}
/**
 * <pre>
 * @name Image Color Channel Swap
 * Methods for exchanging the color channels of an image. The methods support arbitrary permutations of the original
 * channels, including replication.
 * </pre>
 */
///@{
/**
 * <pre>
 * 4 channel 8-bit unsigned swap channels, in-place.
 *
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param aDstOrder Integer array describing how channel values are permutated. The n-th entry
 *      of the array contains the number of the channel that is stored in the n-th channel of
 *      the output image. E.g. Given an RGBA image, aDstOrder = [3,2,1,0] converts this to ABGR
 *      channel order.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiSwapChannels_18u_1C4IRNative(JNIEnv *env, jclass cls, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jintArray aDstOrder)
{
    // Null-checks for non-primitive arguments
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiSwapChannels_8u_C4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiSwapChannels_8u_C4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (aDstOrder == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'aDstOrder' is null for nppiSwapChannels_8u_C4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiSwapChannels_8u_C4IR(pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, aDstOrder=%p)\n",
        pSrcDst, nSrcDstStep, oSizeROI, aDstOrder);

    // Native variable declarations
    Npp8u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int aDstOrder_native[4];

    // Obtain native variable values
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp8u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!initNative(env, aDstOrder, aDstOrder_native, 4)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiSwapChannels_8u_C4IR(pSrcDst_native, nSrcDstStep_native, oSizeROI_native, aDstOrder_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    if (!releaseNative(env, aDstOrder_native, aDstOrder, 4)) return JNPP_STATUS_INTERNAL_ERROR;

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

///@}
///@} image_data_exchange_and_initialization
/** @defgroup image_arithmetic Arithmetic and Logical Operations
 */
///@{
/** @name AddC
 * Adds a constant value to each pixel of an image.
 */
///@{
/**
 * <pre>
 * One 8-bit unsigned char channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_18u_1C1RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jbyte nConstant, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_8u_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    // nConstant is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_8u_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_8u_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_8u_C1RSfs(pSrc1=%p, nSrc1Step=%d, nConstant=%c, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp8u nConstant_native;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp8u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    nConstant_native = (Npp8u)nConstant;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_8u_C1RSfs(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    // nConstant is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 8-bit unsigned char channel in place image add constant, scale, then clamp to saturated value.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_18u_1C1IRSfsNative(JNIEnv *env, jclass cls, jbyte nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    // nConstant is primitive
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_8u_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_8u_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_8u_C1IRSfs(nConstant=%c, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u nConstant_native;
    Npp8u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    nConstant_native = (Npp8u)nConstant;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp8u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_8u_C1IRSfs(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    // nConstant is primitive
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 8-bit unsigned char channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel..
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_18u_1C3RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_8u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_8u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_8u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_8u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_8u_C3RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp8u* pConstants_native = NULL;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp8u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp8u*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_8u_C3RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 8-bit unsigned char channel 8-bit unsigned char in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel..
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_18u_1C3IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_8u_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_8u_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_8u_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_8u_C3IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pConstants_native = NULL;
    Npp8u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp8u*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp8u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_8u_C3IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 8-bit unsigned char channel with unmodified alpha image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel..
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_18u_1AC4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_8u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_8u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_8u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_8u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_8u_AC4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp8u* pConstants_native = NULL;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp8u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp8u*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_8u_AC4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 8-bit unsigned char channel with unmodified alpha in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel..
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_18u_1AC4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_8u_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_8u_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_8u_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_8u_AC4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pConstants_native = NULL;
    Npp8u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp8u*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp8u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_8u_AC4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 8-bit unsigned char channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel..
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_18u_1C4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_8u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_8u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_8u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_8u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_8u_C4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp8u* pConstants_native = NULL;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp8u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp8u*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_8u_C4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 8-bit unsigned char channel in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_18u_1C4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_8u_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_8u_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_8u_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_8u_C4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pConstants_native = NULL;
    Npp8u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp8u*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp8u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_8u_C4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 16-bit unsigned short channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116u_1C1RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jshort nConstant, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_16u_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    // nConstant is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_16u_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16u_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16u_C1RSfs(pSrc1=%p, nSrc1Step=%d, nConstant=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16u nConstant_native;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    nConstant_native = (Npp16u)nConstant;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16u_C1RSfs(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    // nConstant is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 16-bit unsigned short channel in place image add constant, scale, then clamp to saturated value.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116u_1C1IRSfsNative(JNIEnv *env, jclass cls, jshort nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    // nConstant is primitive
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_16u_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16u_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16u_C1IRSfs(nConstant=%d, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u nConstant_native;
    Npp16u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    nConstant_native = (Npp16u)nConstant;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16u_C1IRSfs(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    // nConstant is primitive
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 16-bit unsigned short channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116u_1C3RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_16u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_16u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16u_C3RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16u* pConstants_native = NULL;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16u*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16u_C3RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 16-bit unsigned short channel in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116u_1C3IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16u_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_16u_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16u_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16u_C3IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pConstants_native = NULL;
    Npp16u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16u*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16u_C3IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit unsigned short channel with unmodified alpha image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116u_1AC4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_16u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_16u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16u_AC4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16u* pConstants_native = NULL;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16u*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16u_AC4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit unsigned short channel with unmodified alpha in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116u_1AC4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16u_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_16u_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16u_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16u_AC4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pConstants_native = NULL;
    Npp16u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16u*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16u_AC4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit unsigned short channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116u_1C4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_16u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_16u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16u_C4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16u* pConstants_native = NULL;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16u*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16u_C4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit unsigned short channel in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116u_1C4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16u_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_16u_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16u_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16u_C4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pConstants_native = NULL;
    Npp16u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16u*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16u_C4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 16-bit signed short channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116s_1C1RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jshort nConstant, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_16s_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    // nConstant is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_16s_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16s_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16s_C1RSfs(pSrc1=%p, nSrc1Step=%d, nConstant=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16s nConstant_native;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16s*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    nConstant_native = (Npp16s)nConstant;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16s_C1RSfs(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    // nConstant is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 16-bit signed short channel in place image add constant, scale, then clamp to saturated value.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116s_1C1IRSfsNative(JNIEnv *env, jclass cls, jshort nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    // nConstant is primitive
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_16s_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16s_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16s_C1IRSfs(nConstant=%d, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s nConstant_native;
    Npp16s* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    nConstant_native = (Npp16s)nConstant;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16s*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16s_C1IRSfs(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    // nConstant is primitive
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 16-bit signed short channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116s_1C3RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_16s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_16s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16s_C3RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16s* pConstants_native = NULL;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16s*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16s*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16s_C3RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 16-bit signed short channel in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116s_1C3IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16s_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_16s_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16s_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16s_C3IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pConstants_native = NULL;
    Npp16s* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16s*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16s*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16s_C3IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit signed short channel with unmodified alpha image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116s_1AC4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_16s_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16s_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_16s_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16s_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16s_AC4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16s* pConstants_native = NULL;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16s*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16s*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16s_AC4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit signed short channel with unmodified alpha in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116s_1AC4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16s_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_16s_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16s_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16s_AC4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pConstants_native = NULL;
    Npp16s* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16s*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16s*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16s_AC4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit signed short channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116s_1C4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_16s_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16s_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_16s_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16s_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16s_C4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16s* pConstants_native = NULL;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16s*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16s*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16s_C4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit signed short channel in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116s_1C4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16s_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_16s_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16s_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16s_C4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pConstants_native = NULL;
    Npp16s* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16s*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16s*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16s_C4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116sc_1C1RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject nConstant, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_16sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (nConstant == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nConstant' is null for nppiAddC_16sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_16sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16sc_C1RSfs(pSrc1=%p, nSrc1Step=%d, nConstant=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16sc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16sc nConstant_native;
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16sc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    if (!initNative(env, nConstant, nConstant_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16sc_C1RSfs(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releaseNative(env, nConstant_native, nConstant)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image add constant, scale, then clamp to saturated value.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116sc_1C1IRSfsNative(JNIEnv *env, jclass cls, jobject nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (nConstant == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nConstant' is null for nppiAddC_16sc_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_16sc_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16sc_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16sc_C1IRSfs(nConstant=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16sc nConstant_native;
    Npp16sc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    if (!initNative(env, nConstant, nConstant_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16sc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16sc_C1IRSfs(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releaseNative(env, nConstant_native, nConstant)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116sc_1C3RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_16sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_16sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16sc_C3RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16sc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16sc* pConstants_native = NULL;
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16sc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16sc*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16sc_C3RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116sc_1C3IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16sc_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_16sc_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16sc_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16sc_C3IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16sc* pConstants_native = NULL;
    Npp16sc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16sc*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16sc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16sc_C3IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116sc_1AC4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_16sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_16sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16sc_AC4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16sc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16sc* pConstants_native = NULL;
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16sc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16sc*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16sc_AC4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_116sc_1AC4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_16sc_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_16sc_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_16sc_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_16sc_AC4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16sc* pConstants_native = NULL;
    Npp16sc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16sc*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16sc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_16sc_AC4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit signed integer channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132s_1C1RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jint nConstant, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32s_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    // nConstant is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32s_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32s_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32s_C1RSfs(pSrc1=%p, nSrc1Step=%d, nConstant=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32s* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32s nConstant_native;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32s*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    nConstant_native = (Npp32s)nConstant;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32s_C1RSfs(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    // nConstant is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit signed integer channel in place image add constant, scale, then clamp to saturated value.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132s_1C1IRSfsNative(JNIEnv *env, jclass cls, jint nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    // nConstant is primitive
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32s_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32s_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32s_C1IRSfs(nConstant=%d, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32s nConstant_native;
    Npp32s* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    nConstant_native = (Npp32s)nConstant;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32s*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32s_C1IRSfs(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    // nConstant is primitive
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit signed integer channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132s_1C3RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32s_C3RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32s* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32s* pConstants_native = NULL;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32s*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32s*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32s_C3RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit signed integer channel in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132s_1C3IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32s_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32s_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32s_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32s_C3IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32s* pConstants_native = NULL;
    Npp32s* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32s*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32s*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32s_C3IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132sc_1C1RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject nConstant, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (nConstant == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nConstant' is null for nppiAddC_32sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32sc_C1RSfs(pSrc1=%p, nSrc1Step=%d, nConstant=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32sc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32sc nConstant_native;
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32sc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    if (!initNative(env, nConstant, nConstant_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32sc_C1RSfs(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releaseNative(env, nConstant_native, nConstant)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel in place image add constant, scale, then clamp to saturated value.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132sc_1C1IRSfsNative(JNIEnv *env, jclass cls, jobject nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (nConstant == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nConstant' is null for nppiAddC_32sc_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32sc_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32sc_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32sc_C1IRSfs(nConstant=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32sc nConstant_native;
    Npp32sc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    if (!initNative(env, nConstant, nConstant_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32sc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32sc_C1IRSfs(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releaseNative(env, nConstant_native, nConstant)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132sc_1C3RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32sc_C3RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32sc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32sc* pConstants_native = NULL;
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32sc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32sc*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32sc_C3RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132sc_1C3IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32sc_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32sc_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32sc_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32sc_C3IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32sc* pConstants_native = NULL;
    Npp32sc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32sc*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32sc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32sc_C3IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel with unmodified alpha image add constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132sc_1AC4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32sc_AC4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32sc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32sc* pConstants_native = NULL;
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32sc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32sc*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32sc_AC4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image add constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132sc_1AC4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32sc_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32sc_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32sc_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32sc_AC4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32sc* pConstants_native = NULL;
    Npp32sc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32sc*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32sc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32sc_AC4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit floating point channel image add constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132f_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jfloat nConstant, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    // nConstant is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32f_C1R(pSrc1=%p, nSrc1Step=%d, nConstant=%f, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32f nConstant_native;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32f*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    nConstant_native = (Npp32f)nConstant;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32f_C1R(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    // nConstant is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit floating point channel in place image add constant.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132f_1C1IRNative(JNIEnv *env, jclass cls, jfloat nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nConstant is primitive
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32f_C1IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32f_C1IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32f_C1IR(nConstant=%f, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI);

    // Native variable declarations
    Npp32f nConstant_native;
    Npp32f* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nConstant_native = (Npp32f)nConstant;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32f*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32f_C1IR(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nConstant is primitive
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit floating point channel image add constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132f_1C3RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32f_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32f_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32f_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32f_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32f_C3R(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32f* pConstants_native = NULL;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32f*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32f*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32f_C3R(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit floating point channel in place image add constant.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132f_1C3IRNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32f_C3IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32f_C3IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32f_C3IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32f_C3IR(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pConstants_native = NULL;
    Npp32f* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32f*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32f*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32f_C3IR(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit floating point channel with unmodified alpha image add constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132f_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32f_AC4R(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32f* pConstants_native = NULL;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32f*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32f*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32f_AC4R(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit floating point channel with unmodified alpha in place image add constant.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132f_1AC4IRNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32f_AC4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32f_AC4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32f_AC4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32f_AC4IR(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pConstants_native = NULL;
    Npp32f* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32f*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32f*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32f_AC4IR(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit floating point channel image add constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132f_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32f_C4R(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32f* pConstants_native = NULL;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32f*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32f*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32f_C4R(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit floating point channel in place image add constant.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132f_1C4IRNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32f_C4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32f_C4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32f_C4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32f_C4IR(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pConstants_native = NULL;
    Npp32f* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32f*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32f*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32f_C4IR(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image add constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132fc_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject nConstant, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (nConstant == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nConstant' is null for nppiAddC_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32fc_C1R(pSrc1=%p, nSrc1Step=%d, nConstant=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32fc nConstant_native;
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32fc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    if (!initNative(env, nConstant, nConstant_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32fc_C1R(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releaseNative(env, nConstant_native, nConstant)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image add constant.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132fc_1C1IRNative(JNIEnv *env, jclass cls, jobject nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (nConstant == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nConstant' is null for nppiAddC_32fc_C1IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32fc_C1IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32fc_C1IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32fc_C1IR(nConstant=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc nConstant_native;
    Npp32fc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    if (!initNative(env, nConstant, nConstant_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32fc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32fc_C1IR(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releaseNative(env, nConstant_native, nConstant)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image add constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132fc_1C3RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32fc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32fc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32fc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32fc_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32fc_C3R(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32fc* pConstants_native = NULL;
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32fc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32fc*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32fc_C3R(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image add constant.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132fc_1C3IRNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32fc_C3IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32fc_C3IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32fc_C3IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32fc_C3IR(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pConstants_native = NULL;
    Npp32fc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32fc*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32fc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32fc_C3IR(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel with unmodified alpha image add constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132fc_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32fc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32fc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32fc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32fc_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32fc_AC4R(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32fc* pConstants_native = NULL;
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32fc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32fc*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32fc_AC4R(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel with unmodified alpha in place image add constant.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132fc_1AC4IRNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32fc_AC4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32fc_AC4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32fc_AC4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32fc_AC4IR(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pConstants_native = NULL;
    Npp32fc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32fc*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32fc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32fc_AC4IR(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image add constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132fc_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiAddC_32fc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32fc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiAddC_32fc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32fc_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32fc_C4R(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32fc* pConstants_native = NULL;
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32fc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32fc*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32fc_C4R(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image add constant.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiAddC_132fc_1C4IRNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiAddC_32fc_C4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiAddC_32fc_C4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiAddC_32fc_C4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiAddC_32fc_C4IR(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pConstants_native = NULL;
    Npp32fc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32fc*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32fc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiAddC_32fc_C4IR(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

// end of AddC
///@}
/** @name MulC
 * Multiplies each pixel of an image by a constant value.
 */
///@{
/**
 * <pre>
 * One 8-bit unsigned char channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_18u_1C1RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jbyte nConstant, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_8u_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    // nConstant is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_8u_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_8u_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_8u_C1RSfs(pSrc1=%p, nSrc1Step=%d, nConstant=%c, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp8u nConstant_native;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp8u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    nConstant_native = (Npp8u)nConstant;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_8u_C1RSfs(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    // nConstant is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 8-bit unsigned char channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_18u_1C1IRSfsNative(JNIEnv *env, jclass cls, jbyte nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    // nConstant is primitive
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_8u_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_8u_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_8u_C1IRSfs(nConstant=%c, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u nConstant_native;
    Npp8u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    nConstant_native = (Npp8u)nConstant;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp8u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_8u_C1IRSfs(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    // nConstant is primitive
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 8-bit unsigned char channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_18u_1C3RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_8u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_8u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_8u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_8u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_8u_C3RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp8u* pConstants_native = NULL;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp8u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp8u*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_8u_C3RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 8-bit unsigned char channel 8-bit unsigned char in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_18u_1C3IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_8u_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_8u_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_8u_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_8u_C3IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pConstants_native = NULL;
    Npp8u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp8u*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp8u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_8u_C3IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 8-bit unsigned char channel with unmodified alpha image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_18u_1AC4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_8u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_8u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_8u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_8u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_8u_AC4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp8u* pConstants_native = NULL;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp8u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp8u*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_8u_AC4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 8-bit unsigned char channel with unmodified alpha in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_18u_1AC4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_8u_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_8u_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_8u_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_8u_AC4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pConstants_native = NULL;
    Npp8u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp8u*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp8u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_8u_AC4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 8-bit unsigned char channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_18u_1C4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_8u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_8u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_8u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_8u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_8u_C4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp8u* pConstants_native = NULL;
    Npp8u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp8u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp8u*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp8u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_8u_C4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 8-bit unsigned char channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_18u_1C4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_8u_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_8u_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_8u_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_8u_C4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp8u* pConstants_native = NULL;
    Npp8u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp8u*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp8u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_8u_C4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 16-bit unsigned short channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116u_1C1RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jshort nConstant, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_16u_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    // nConstant is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_16u_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16u_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16u_C1RSfs(pSrc1=%p, nSrc1Step=%d, nConstant=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16u nConstant_native;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    nConstant_native = (Npp16u)nConstant;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16u_C1RSfs(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    // nConstant is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 16-bit unsigned short channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116u_1C1IRSfsNative(JNIEnv *env, jclass cls, jshort nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    // nConstant is primitive
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_16u_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16u_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16u_C1IRSfs(nConstant=%d, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u nConstant_native;
    Npp16u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    nConstant_native = (Npp16u)nConstant;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16u_C1IRSfs(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    // nConstant is primitive
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 16-bit unsigned short channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116u_1C3RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_16u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_16u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16u_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16u_C3RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16u* pConstants_native = NULL;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16u*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16u_C3RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 16-bit unsigned short channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116u_1C3IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16u_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_16u_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16u_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16u_C3IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pConstants_native = NULL;
    Npp16u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16u*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16u_C3IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit unsigned short channel with unmodified alpha image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116u_1AC4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_16u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_16u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16u_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16u_AC4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16u* pConstants_native = NULL;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16u*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16u_AC4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit unsigned short channel with unmodified alpha in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116u_1AC4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16u_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_16u_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16u_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16u_AC4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pConstants_native = NULL;
    Npp16u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16u*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16u_AC4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit unsigned short channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116u_1C4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_16u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_16u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16u_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16u_C4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16u* pConstants_native = NULL;
    Npp16u* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16u*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16u*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16u*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16u_C4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit unsigned short channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116u_1C4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16u_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_16u_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16u_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16u_C4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16u* pConstants_native = NULL;
    Npp16u* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16u*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16u*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16u_C4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 16-bit signed short channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116s_1C1RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jshort nConstant, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_16s_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    // nConstant is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_16s_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16s_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16s_C1RSfs(pSrc1=%p, nSrc1Step=%d, nConstant=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16s nConstant_native;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16s*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    nConstant_native = (Npp16s)nConstant;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16s_C1RSfs(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    // nConstant is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 16-bit signed short channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116s_1C1IRSfsNative(JNIEnv *env, jclass cls, jshort nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    // nConstant is primitive
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_16s_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16s_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16s_C1IRSfs(nConstant=%d, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s nConstant_native;
    Npp16s* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    nConstant_native = (Npp16s)nConstant;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16s*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16s_C1IRSfs(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    // nConstant is primitive
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 16-bit signed short channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116s_1C3RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_16s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_16s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16s_C3RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16s* pConstants_native = NULL;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16s*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16s*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16s_C3RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 16-bit signed short channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116s_1C3IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16s_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_16s_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16s_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16s_C3IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pConstants_native = NULL;
    Npp16s* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16s*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16s*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16s_C3IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit signed short channel with unmodified alpha image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116s_1AC4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_16s_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16s_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_16s_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16s_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16s_AC4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16s* pConstants_native = NULL;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16s*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16s*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16s_AC4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit signed short channel with unmodified alpha in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116s_1AC4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16s_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_16s_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16s_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16s_AC4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pConstants_native = NULL;
    Npp16s* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16s*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16s*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16s_AC4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit signed short channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116s_1C4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_16s_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16s_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_16s_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16s_C4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16s_C4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16s* pConstants_native = NULL;
    Npp16s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16s*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16s*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16s_C4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit signed short channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116s_1C4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16s_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_16s_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16s_C4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16s_C4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16s* pConstants_native = NULL;
    Npp16s* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16s*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16s*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16s_C4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116sc_1C1RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject nConstant, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_16sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (nConstant == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nConstant' is null for nppiMulC_16sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_16sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16sc_C1RSfs(pSrc1=%p, nSrc1Step=%d, nConstant=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16sc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16sc nConstant_native;
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16sc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    if (!initNative(env, nConstant, nConstant_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16sc_C1RSfs(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releaseNative(env, nConstant_native, nConstant)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116sc_1C1IRSfsNative(JNIEnv *env, jclass cls, jobject nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (nConstant == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nConstant' is null for nppiMulC_16sc_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_16sc_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16sc_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16sc_C1IRSfs(nConstant=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16sc nConstant_native;
    Npp16sc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    if (!initNative(env, nConstant, nConstant_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16sc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16sc_C1IRSfs(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releaseNative(env, nConstant_native, nConstant)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116sc_1C3RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_16sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_16sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16sc_C3RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16sc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16sc* pConstants_native = NULL;
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16sc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16sc*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16sc_C3RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116sc_1C3IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16sc_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_16sc_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16sc_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16sc_C3IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16sc* pConstants_native = NULL;
    Npp16sc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16sc*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16sc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16sc_C3IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116sc_1AC4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_16sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_16sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16sc_AC4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16sc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp16sc* pConstants_native = NULL;
    Npp16sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp16sc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16sc*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp16sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16sc_AC4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_116sc_1AC4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_16sc_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_16sc_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_16sc_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_16sc_AC4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp16sc* pConstants_native = NULL;
    Npp16sc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp16sc*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp16sc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_16sc_AC4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit signed integer channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132s_1C1RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jint nConstant, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_32s_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    // nConstant is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_32s_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32s_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32s_C1RSfs(pSrc1=%p, nSrc1Step=%d, nConstant=%d, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32s* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32s nConstant_native;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32s*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    nConstant_native = (Npp32s)nConstant;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32s_C1RSfs(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    // nConstant is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit signed integer channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132s_1C1IRSfsNative(JNIEnv *env, jclass cls, jint nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    // nConstant is primitive
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_32s_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32s_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32s_C1IRSfs(nConstant=%d, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32s nConstant_native;
    Npp32s* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    nConstant_native = (Npp32s)nConstant;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32s*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32s_C1IRSfs(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    // nConstant is primitive
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit signed integer channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132s_1C3RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_32s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_32s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_32s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32s_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32s_C3RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32s* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32s* pConstants_native = NULL;
    Npp32s* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32s*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32s*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32s*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32s_C3RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit signed integer channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132s_1C3IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_32s_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_32s_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32s_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32s_C3IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32s* pConstants_native = NULL;
    Npp32s* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32s*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32s*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32s_C3IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132sc_1C1RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject nConstant, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_32sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (nConstant == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nConstant' is null for nppiMulC_32sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_32sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32sc_C1RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32sc_C1RSfs(pSrc1=%p, nSrc1Step=%d, nConstant=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32sc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32sc nConstant_native;
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32sc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    if (!initNative(env, nConstant, nConstant_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32sc_C1RSfs(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releaseNative(env, nConstant_native, nConstant)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132sc_1C1IRSfsNative(JNIEnv *env, jclass cls, jobject nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (nConstant == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nConstant' is null for nppiMulC_32sc_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_32sc_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32sc_C1IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32sc_C1IRSfs(nConstant=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32sc nConstant_native;
    Npp32sc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    if (!initNative(env, nConstant, nConstant_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32sc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32sc_C1IRSfs(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releaseNative(env, nConstant_native, nConstant)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132sc_1C3RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_32sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_32sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_32sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32sc_C3RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32sc_C3RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32sc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32sc* pConstants_native = NULL;
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32sc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32sc*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32sc_C3RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132sc_1C3IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_32sc_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_32sc_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32sc_C3IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32sc_C3IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32sc* pConstants_native = NULL;
    Npp32sc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32sc*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32sc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32sc_C3IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel with unmodified alpha image multiply by constant, scale, then clamp to saturated value.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132sc_1AC4RSfsNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_32sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_32sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_32sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32sc_AC4RSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32sc_AC4RSfs(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32sc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32sc* pConstants_native = NULL;
    Npp32sc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32sc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32sc*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32sc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32sc_AC4RSfs(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image multiply by constant, scale, then clamp to saturated value.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @param nScaleFactor \ref integer_result_scaling.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132sc_1AC4IRSfsNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI, jint nScaleFactor)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_32sc_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_32sc_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32sc_AC4IRSfs");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nScaleFactor is primitive

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32sc_AC4IRSfs(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p, nScaleFactor=%d)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor);

    // Native variable declarations
    Npp32sc* pConstants_native = NULL;
    Npp32sc* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;
    int nScaleFactor_native = 0;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32sc*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32sc*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;
    nScaleFactor_native = (int)nScaleFactor;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32sc_AC4IRSfs(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native, nScaleFactor_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure
    // nScaleFactor is primitive

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit floating point channel image multiply by constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132f_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jfloat nConstant, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    // nConstant is primitive
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32f_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32f_C1R(pSrc1=%p, nSrc1Step=%d, nConstant=%f, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32f nConstant_native;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32f*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    nConstant_native = (Npp32f)nConstant;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32f_C1R(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    // nConstant is primitive
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit floating point channel in place image multiply by constant.
 * @param nConstant Constant.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132f_1C1IRNative(JNIEnv *env, jclass cls, jfloat nConstant, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    // nConstant is primitive
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_32f_C1IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32f_C1IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32f_C1IR(nConstant=%f, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p)\n",
        nConstant, pSrcDst, nSrcDstStep, oSizeROI);

    // Native variable declarations
    Npp32f nConstant_native;
    Npp32f* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    nConstant_native = (Npp32f)nConstant;
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32f*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32f_C1IR(nConstant_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native);

    // Write back native variable values
    // nConstant is primitive
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit floating point channel image multiply by constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132f_1C3RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_32f_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_32f_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_32f_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32f_C3R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32f_C3R(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32f* pConstants_native = NULL;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32f*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32f*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32f_C3R(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Three 32-bit floating point channel in place image multiply by constant.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132f_1C3IRNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_32f_C3IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_32f_C3IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32f_C3IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32f_C3IR(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pConstants_native = NULL;
    Npp32f* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32f*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32f*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32f_C3IR(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit floating point channel with unmodified alpha image multiply by constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132f_1AC4RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32f_AC4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32f_AC4R(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32f* pConstants_native = NULL;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32f*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32f*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32f_AC4R(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit floating point channel with unmodified alpha in place image multiply by constant.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132f_1AC4IRNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_32f_AC4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_32f_AC4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32f_AC4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32f_AC4IR(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pConstants_native = NULL;
    Npp32f* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32f*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32f*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32f_AC4IR(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit floating point channel image multiply by constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132f_1C4RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject pConstants, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32f_C4R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32f_C4R(pSrc1=%p, nSrc1Step=%d, pConstants=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32f* pConstants_native = NULL;
    Npp32f* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32f*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32f*)pConstants_pointerData->getPointer(env);
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32f*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32f_C4R(pSrc1_native, nSrc1Step_native, pConstants_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrc1Step is primitive
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * Four 32-bit floating point channel in place image multiply by constant.
 * @param pConstants pointer to a list of constant values, one per channel.
 * @param pSrcDst \ref in_place_image_pointer.
 * @param nSrcDstStep \ref in_place_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132f_1C4IRNative(JNIEnv *env, jclass cls, jobject pConstants, jobject pSrcDst, jint nSrcDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pConstants == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pConstants' is null for nppiMulC_32f_C4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pSrcDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrcDst' is null for nppiMulC_32f_C4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrcDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32f_C4IR");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32f_C4IR(pConstants=%p, pSrcDst=%p, nSrcDstStep=%d, oSizeROI=%p)\n",
        pConstants, pSrcDst, nSrcDstStep, oSizeROI);

    // Native variable declarations
    Npp32f* pConstants_native = NULL;
    Npp32f* pSrcDst_native = NULL;
    int nSrcDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pConstants_pointerData = initPointerData(env, pConstants);
    if (pConstants_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pConstants_native = (Npp32f*)pConstants_pointerData->getPointer(env);
    PointerData *pSrcDst_pointerData = initPointerData(env, pSrcDst);
    if (pSrcDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrcDst_native = (Npp32f*)pSrcDst_pointerData->getPointer(env);
    nSrcDstStep_native = (int)nSrcDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32f_C4IR(pConstants_native, pSrcDst_native, nSrcDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pConstants_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    if (!releasePointerData(env, pSrcDst_pointerData, 0)) return JNPP_STATUS_INTERNAL_ERROR;
    // nSrcDstStep is primitive
    // oSizeROI is a read-only structure

    // Return the result
    jint jniResult;
    jniResult = (jint)jniResult_native;
    return jniResult;
}

/**
 * <pre>
 * One 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image multiply by constant.
 * @param pSrc1 \ref source_image_pointer.
 * @param nSrc1Step \ref source_image_line_step.
 * @param nConstant Constant.
 * @param pDst \ref destination_image_pointer.
 * @param nDstStep \ref destination_image_line_step.
 * @param oSizeROI \ref roi_specification.
 * @return
 *  \ref image_data_error_codes, \ref roi_error_codes
 * </pre>
 */
JNIEXPORT jint JNICALL Java_jcuda_jnpp_JNppi_nppiMulC_132fc_1C1RNative(JNIEnv *env, jclass cls, jobject pSrc1, jint nSrc1Step, jobject nConstant, jobject pDst, jint nDstStep, jobject oSizeROI)
{
    // Null-checks for non-primitive arguments
    if (pSrc1 == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pSrc1' is null for nppiMulC_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nSrc1Step is primitive
    if (nConstant == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'nConstant' is null for nppiMulC_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    if (pDst == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'pDst' is null for nppiMulC_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    // nDstStep is primitive
    if (oSizeROI == NULL)
    {
        ThrowByName(env, "java/lang/NullPointerException", "Parameter 'oSizeROI' is null for nppiMulC_32fc_C1R");
        return JNPP_STATUS_INTERNAL_ERROR;
    }

    // Log message
    Logger::log(LOG_TRACE, "Executing nppiMulC_32fc_C1R(pSrc1=%p, nSrc1Step=%d, nConstant=%p, pDst=%p, nDstStep=%d, oSizeROI=%p)\n",
        pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI);

    // Native variable declarations
    Npp32fc* pSrc1_native = NULL;
    int nSrc1Step_native = 0;
    Npp32fc nConstant_native;
    Npp32fc* pDst_native = NULL;
    int nDstStep_native = 0;
    NppiSize oSizeROI_native;

    // Obtain native variable values
    PointerData *pSrc1_pointerData = initPointerData(env, pSrc1);
    if (pSrc1_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pSrc1_native = (Npp32fc*)pSrc1_pointerData->getPointer(env);
    nSrc1Step_native = (int)nSrc1Step;
    if (!initNative(env, nConstant, nConstant_native)) return JNPP_STATUS_INTERNAL_ERROR;
    PointerData *pDst_pointerData = initPointerData(env, pDst);
    if (pDst_pointerData == NULL)
    {
        return JNPP_STATUS_INTERNAL_ERROR;
    }
    pDst_native = (Npp32fc*)pDst_pointerData->getPointer(env);
    nDstStep_native = (int)nDstStep;
    if (!initNative(env, oSizeROI, oSizeROI_native)) return JNPP_STATUS_INTERNAL_ERROR;

    // Native function call
    NppStatus jniResult_native = nppiMulC_32fc_C1R(pSrc1_native, nSrc1Step_native, nConstant_native, pDst_native, nDstStep_native, oSizeROI_native);

    // Write back native variable values
    if (!releasePointerData(env, pSrc1_pointerData, 0)) return JN