/*
 * JNpp - Java bindings for NPP, to be used with JCuda
 *
 * Copyright (c) 2010-2012 Marco Hutter - http://www.jcuda.org
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

package jcuda.jnpp;


/**
 * Java bindings for NPP
 */
public class JNpp
{
    /**
     * Major version number
     */
    public static final int NPP_VERSION_MAJOR = 1;

    /**
     * Minor version number
     */
    public static final int NPP_VERSION_MINOR = 0;

    /**
     * Build number. Reflects the number of the nightly build.
     */
    public static final int NPP_VERSION_BUILD = 0;



    /**
     * Maximum 8-bit unsigned integer
     */
    public static final byte NPP_MAX_8U = (byte)( 0xFF );

    /**
     * Maximum 16-bit unsigned integer
     */
    public static final short NPP_MAX_16U    = (short)( 0xFFFF );

    /**
     * Maximum 32-bit unsigned integer
     */
    public static final int NPP_MAX_32U    = ( 0xFFFFFFFF );

    /**
     * Minimum 8-bit unsigned integer
     */
    public static final byte NPP_MIN_8U     = ( 0 );

    /**
     * Minimum 16-bit unsigned integer
     */
    public static final short NPP_MIN_16U    = ( 0 );

    /**
     * Minimum 32-bit unsigned integer
     */
    public static final int NPP_MIN_32U    = ( 0 );

    /**
     * Minimum 8-bit signed integer
     */
    public static final byte NPP_MIN_8S     = (-128 );

    /**
     * Maximum 8-bit signed integer
     */
    public static final byte NPP_MAX_8S     = ( 127 );

    /**
     * Minimum 16-bit signed integer
     */
    public static final short NPP_MIN_16S    = (short)(-32768 );

    /**
     * Maximum 16-bit signed integer
     */
    public static final short NPP_MAX_16S    = ( 32767 );

    /**
     * Minimum 32-bit signed integer
     */
    public static final int NPP_MIN_32S    = (-2147483647 - 1 );

    /**
     * Maximum 32-bit signed integer
     */
    public static final int NPP_MAX_32S    = ( 2147483647 );


    /**
     * NPP Core - Basic functions for library management,
     * in particular library version and device property
     * query functions.
     */
    public static class Core
    {
        /**
         * Private constructor to prevent instantiation
         */
        private Core()
        {
        }
        
        /**
         * Get the NPP library version.
         *
         * @return A struct containing separate values for major and minor revision 
         *      and build number.
         */
        public static NppLibraryVersion nppGetLibVersion()
        {
            return nppGetLibVersionNative();
        }
        private static native NppLibraryVersion nppGetLibVersionNative();

        /**
         * What CUDA compute model is supported by the default CUDA device?
         * 
         * Before trying to call any NPP functions, the user should make a call
         * this function to ensure that the current machine has a CUDA capable device.
         *
         * @return An enum value representing if a CUDA capable device was found and what
         *      level of compute capabilities it supports.
         */
        public static int nppGetGpuComputeCapability()
        {
            return nppGetGpuComputeCapabilityNative();
        }
        private static native int nppGetGpuComputeCapabilityNative();

        /**
         * Get the number of Streaming Multiprocessors (SM) on the default CUDA device.
         *
         * @return Number of SMs of the default CUDA device.
         */
        public static int nppGetGpuNumSMs()
        {
            return nppGetGpuNumSMsNative();
        }
        private static native int nppGetGpuNumSMsNative();

        /** 
         * Get the name of the default CUDA device.
         *
         * @return Name string of the graphics-card/compute device in a system.
         */
        public static String nppGetGpuName()
        {
            return nppGetGpuNameNative();
        }
        private static native String nppGetGpuNameNative();
    }
    

}
