/*
 * JNpp - Java bindings for NPP, to be used with JCuda
 *
 * Copyright (c) 2010-2012 Marco Hutter - http://www.jcuda.org
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */
package jcuda.jnpp;

import jcuda.*;
import jcuda.jnpp.types.*;

/**
 * Java bindings for NPPI. Most of the documentation is extracted
 * from the NPPI header files.
 */
public class JNppi
{

    /**
     * The flag that indicates whether the native library has been
     * loaded
     */
    private static boolean initialized = false;

    /**
     * Whether a CudaException should be thrown if a method is about
     * to set a result code that indicates an error
     */
    private static boolean exceptionsEnabled = false;

    /**
     * Private constructor to prevent instantiation
     */
    private JNppi()
    {
    }

    // Initialize the native library.
    static
    {
        initialize();
    }

    /**
     * Initializes the native library. Note that this method
     * does not have to be called explicitly, since it will
     * be called automatically when this class is loaded.
     */
    public static void initialize()
    {
        if (!initialized)
        {
            LibUtils.loadLibrary("Jnppi");
            initialized = true;
        }
    }



    /**
     * Set the specified log level for the JNppi library.<br />
     * <br />
     * Currently supported log levels:
     * <br />
     * LOG_QUIET: Never print anything <br />
     * LOG_ERROR: Print error messages <br />
     * LOG_TRACE: Print a trace of all native function calls <br />
     *
     * @param logLevel The log level to use.
     */
    public static void setLogLevel(LogLevel logLevel)
    {
        setLogLevelNative(logLevel.ordinal());
    }

    private static native void setLogLevelNative(int logLevel);


    /**
     * Enables or disables exceptions. By default, the methods of this class
     * only return the {@link NppStatus} from the native method.
     * If exceptions are enabled, a CudaException with a detailed error
     * message will be thrown if a method is about to return a result code
     * that indicates an error
     *
     * @param enabled Whether exceptions are enabled
     */
    public static void setExceptionsEnabled(boolean enabled)
    {
        exceptionsEnabled = enabled;
    }

    /**
     * If the given result is a NppStatus that indicates an error
     * and exceptions have been enabled, this method will throw a
     * CudaException with an error message that corresponds to the
     * given result code. Otherwise, the given result is simply
     * returned.
     *
     * @param result The result to check
     * @return The result that was given as the parameter
     * @throws CudaException If exceptions have been enabled and
     * the given result code indicates an error
     */
    private static int checkResult(int result)
    {
        if (exceptionsEnabled && result < 0)
        {
            throw new CudaException(NppStatus.stringFor(result));
        }
        return result;
    }

    /**
     * If the given result is <code>null</code> and exceptions are
     * enabled, then this method will throw a CudaException with
     * an error message stating that a memory allocation failed.
     * Otherwise, the result is simply returned.
     *
     * @param <T> The type of the pointer
     * @param result The result
     * @return The result that was given as a parameter
     * @throws CudaException If exceptions have been enabled and
     * the given result was <code>null</code>
     */
    private static <T extends TypedPointer<?>> T checkResult(T result)
    {
        if (exceptionsEnabled && result == null)
        {
            throw new CudaException("Failed to allocate memory");
        }
        return result;
    }




    //=== Auto-generated part ================================================

    /**
     * <pre>
     * 8-bit unsigned image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp8u> nppiMalloc_8u_C1(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_8u_C1Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp8u> nppiMalloc_8u_C1Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 2 channel 8-bit unsigned image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp8u> nppiMalloc_8u_C2(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_8u_C2Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp8u> nppiMalloc_8u_C2Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 3 channel 8-bit unsigned image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp8u> nppiMalloc_8u_C3(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_8u_C3Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp8u> nppiMalloc_8u_C3Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 4 channel 8-bit unsigned image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp8u> nppiMalloc_8u_C4(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_8u_C4Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp8u> nppiMalloc_8u_C4Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 16-bit unsigned image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp16u> nppiMalloc_16u_C1(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_16u_C1Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp16u> nppiMalloc_16u_C1Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 2 channel 16-bit unsigned image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp16u> nppiMalloc_16u_C2(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_16u_C2Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp16u> nppiMalloc_16u_C2Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 3 channel 16-bit unsigned image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp16u> nppiMalloc_16u_C3(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_16u_C3Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp16u> nppiMalloc_16u_C3Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 4 channel 16-bit unsigned image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp16u> nppiMalloc_16u_C4(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_16u_C4Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp16u> nppiMalloc_16u_C4Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 16-bit signed image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp16s> nppiMalloc_16s_C1(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_16s_C1Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp16s> nppiMalloc_16s_C1Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    public static TypedPointer<Npp16s> nppiMalloc_16s_C2(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_16s_C2Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp16s> nppiMalloc_16s_C2Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 4 channel 16-bit signed image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp16s> nppiMalloc_16s_C4(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_16s_C4Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp16s> nppiMalloc_16s_C4Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 1 channel 16-bit signed complex image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp16s> nppiMalloc_16sc_C1(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_16sc_C1Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp16s> nppiMalloc_16sc_C1Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 2 channel 16-bit signed complex image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp16s> nppiMalloc_16sc_C2(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_16sc_C2Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp16s> nppiMalloc_16sc_C2Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 3 channel 16-bit signed complex image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp16s> nppiMalloc_16sc_C3(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_16sc_C3Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp16s> nppiMalloc_16sc_C3Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 4 channel 16-bit signed complex image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp16s> nppiMalloc_16sc_C4(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_16sc_C4Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp16s> nppiMalloc_16sc_C4Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 32-bit signed image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32s> nppiMalloc_32s_C1(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32s_C1Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32s> nppiMalloc_32s_C1Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 3 channel 32-bit signed image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32s> nppiMalloc_32s_C3(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32s_C3Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32s> nppiMalloc_32s_C3Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 4 channel 32-bit signed image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32s> nppiMalloc_32s_C4(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32s_C4Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32s> nppiMalloc_32s_C4Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 32-bit integer complex image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32s> nppiMalloc_32sc_C1(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32sc_C1Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32s> nppiMalloc_32sc_C1Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 2 channel 32-bit integer complex image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32s> nppiMalloc_32sc_C2(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32sc_C2Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32s> nppiMalloc_32sc_C2Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 3 channel 32-bit integer complex image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32s> nppiMalloc_32sc_C3(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32sc_C3Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32s> nppiMalloc_32sc_C3Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 4 channel 32-bit integer complex image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32s> nppiMalloc_32sc_C4(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32sc_C4Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32s> nppiMalloc_32sc_C4Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 32-bit floating point image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32f> nppiMalloc_32f_C1(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32f_C1Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32f> nppiMalloc_32f_C1Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 2 channel 32-bit floating point image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32f> nppiMalloc_32f_C2(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32f_C2Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32f> nppiMalloc_32f_C2Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 3 channel 32-bit floating point image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32f> nppiMalloc_32f_C3(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32f_C3Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32f> nppiMalloc_32f_C3Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 4 channel 32-bit floating point image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32f> nppiMalloc_32f_C4(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32f_C4Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32f> nppiMalloc_32f_C4Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    public static TypedPointer<Npp32f> nppiMalloc_32fc_C1(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32fc_C1Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32f> nppiMalloc_32fc_C1Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 2 channel 32-bit float complex image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32f> nppiMalloc_32fc_C2(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32fc_C2Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32f> nppiMalloc_32fc_C2Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 3 channel 32-bit float complex image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32f> nppiMalloc_32fc_C3(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32fc_C3Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32f> nppiMalloc_32fc_C3Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    /**
     * <pre>
     * 4 channel 32-bit float complex image memory allocator.
     * @param nWidthPixels Image width.
     * @param nHeightPixels Image height.
     * @param pStepBytes \ref line_step.
     * @return
     *  Pointer to new image data.
     * </pre>
     */
    public static TypedPointer<Npp32f> nppiMalloc_32fc_C4(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes)
    {
        return checkResult(nppiMalloc_32fc_C4Native(nWidthPixels, nHeightPixels, pStepBytes));
    }
    private static native TypedPointer<Npp32f> nppiMalloc_32fc_C4Native(
        int nWidthPixels, 
        int nHeightPixels, 
        int[] pStepBytes);


    ///@} Malloc
    /**
     * <pre>
     * Free method for any 2D allocated memory.
     * This method should be used to free memory allocated with any of the nppiMalloc_<modifier> methods.
     * @param pData A pointer to memory allocated using nppiMalloc_<modifier>.
     * </pre>
     */
    public static void nppiFree(
        Pointer pData)
    {
        nppiFreeNative(pData);
    }
    private static native void nppiFreeNative(
        Pointer pData);


    ///@} Image-Memory Management
    /**
     * <pre>
     * @defgroup image_data_exchange_and_initialization Data-Exchange and Initialization
     *
     * Primitives for initializtion, copying and converting image data.
     *
     * </pre>
     */
    ///@{
    /**
     * <pre>
     * @name Image-Memory Set
     * Set methods for images of various types. Images are passed to these primitives via a pointer
     * to the image data (first pixel in the ROI) and a step-width, i.e. the number of bytes between
     * successive lines. The size of the area to be set (region-of-interest, ROI) is specified via
     * a NppiSize struct. 
     * In addition to the image data and ROI, all methods have a parameter to specify the value being
     * set. In case of single channel images this is a single value, in case of multi-channel, an 
     * array of values is passed. 
     * </pre>
     */
    ///@{
    /**
     * <pre>
     * 8-bit image set.
     * @param nValue The pixel value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_8s_C1R(
        byte nValue, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_8s_C1RNative(nValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_8s_C1RNative(
        byte nValue, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 8-bit two-channel image set.
     * @param aValue The pixel value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_8s_C2R(
        byte[] aValue, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_8s_C2RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_8s_C2RNative(
        byte[] aValue, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 8-bit three-channel image set.
     * @param aValue The pixel value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_8s_C3R(
        byte[] aValue, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_8s_C3RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_8s_C3RNative(
        byte[] aValue, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 8-bit four-channel image set.
     * @param aValue The pixel value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_8s_C4R(
        byte[] aValue, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_8s_C4RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_8s_C4RNative(
        byte[] aValue, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 8-bit four-channel image set ignoring alpha channel.
     * @param aValue The pixel value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_8s_AC4R(
        byte[] aValue, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_8s_AC4RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_8s_AC4RNative(
        byte[] aValue, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 8-bit unsigned image set.
     * @param nValue The pixel value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_8u_C1R(
        byte nValue, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_8u_C1RNative(nValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_8u_C1RNative(
        byte nValue, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 8-bit unsigned image set. 
     * The 8-bit mask image affects setting of the respective pixels in the destination image.
     * If the mask value is zero (0) the pixel is not set, if the mask is non-zero, the corresponding
     * destination pixel is set to specified value.
     * @param nValue The pixel value to be set.
     * @param pDst Pointer \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_8u_C1MR(
        byte nValue, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_8u_C1MRNative(nValue, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_8u_C1MRNative(
        byte nValue, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 8-bit unsigned image set.
     * @param aValues Four-channel array containing the pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_8u_C4R(
        byte[] aValues, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_8u_C4RNative(aValues, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_8u_C4RNative(
        byte[] aValues, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 4 channel 8-bit unsigned image set.
     * @param aValues Four-channel array containing the pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_8u_C4MR(
        byte[] aValues, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_8u_C4MRNative(aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_8u_C4MRNative(
        byte[] aValues, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 8-bit unsigned image set method, not affecting Alpha channel.
     * For RGBA images, this method allows setting of the RGB values without changing the contents
     * of the alpha-channel (fourth channel).
     * @param aValues Three-channel array containing the pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_8u_AC4R(
        byte[] aValues, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_8u_AC4RNative(aValues, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_8u_AC4RNative(
        byte[] aValues, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 4 channel 8-bit unsigned image set method, not affecting Alpha channel.
     * For RGBA images, this method allows setting of the RGB values without changing the contents
     * of the alpha-channel (fourth channel).
     * @param aValues Three-channel array containing the pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_8u_AC4MR(
        byte[] aValues, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_8u_AC4MRNative(aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_8u_AC4MRNative(
        byte[] aValues, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 8-bit unsigned image set affecting only single channel.
     * For RGBA images, this method allows setting of a single of the four (RGBA) values 
     * without changing the contents of the other three channels. The channel is selected
     * via the pDst pointer. The pointer needs to point to the actual first value to be set,
     * e.g. in order to set the R-channel (first channel), one would pass pDst unmodified, since
     * its value actually points to the r channel. If one wanted to modify the B channel (second
     * channel), one would pass pDst + 2 to the function.
     * @param nValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_8u_C4CR(
        byte nValue, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_8u_C4CRNative(nValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_8u_C4CRNative(
        byte nValue, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit unsigned image set.
     * @param nValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16u_C1R(
        short nValue, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16u_C1RNative(nValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16u_C1RNative(
        short nValue, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 16-bit unsigned image set.
     * @param nValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16u_C1MR(
        short nValue, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_16u_C1MRNative(nValue, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_16u_C1MRNative(
        short nValue, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 2 channel 16-bit unsigned image set.
     * @param aValues New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16u_C2R(
        short[] aValues, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16u_C2RNative(aValues, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16u_C2RNative(
        short[] aValues, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 16-bit unsigned image set.
     * @param aValues New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16u_C4R(
        short[] aValues, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16u_C4RNative(aValues, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16u_C4RNative(
        short[] aValues, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 4 channel 16-bit unsigned image set.
     * @param aValues New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16u_C4MR(
        short[] aValues, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_16u_C4MRNative(aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_16u_C4MRNative(
        short[] aValues, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 16-bit unsigned image set method, not affecting Alpha channel.
     * For RGBA images, this method allows setting of the RGB values without changing the contents
     * of the alpha-channel (fourth channel).
     * @param aValues Three-channel array containing the pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16u_AC4R(
        short[] aValues, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16u_AC4RNative(aValues, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16u_AC4RNative(
        short[] aValues, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 4 channel 16-bit unsigned image set method, not affecting Alpha channel.
     * For RGBA images, this method allows setting of the RGB values without changing the contents
     * of the alpha-channel (fourth channel).
     * @param aValues Three-channel array containing the pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16u_AC4MR(
        short[] aValues, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_16u_AC4MRNative(aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_16u_AC4MRNative(
        short[] aValues, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 16-bit unsigned image set affecting only single channel.
     * For RGBA images, this method allows setting of a single of the four (RGBA) values 
     * without changing the contents of the other three channels. The channel is selected
     * via the pDst pointer. The pointer needs to point to the actual first value to be set,
     * e.g. in order to set the R-channel (first channel), one would pass pDst unmodified, since
     * its value actually points to the r channel. If one wanted to modify the B channel (second
     * channel), one would pass pDst + 2 to the function.
     * @param nValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16u_C4CR(
        short nValue, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16u_C4CRNative(nValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16u_C4CRNative(
        short nValue, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit image set.
     * @param nValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16s_C1R(
        short nValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16s_C1RNative(nValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16s_C1RNative(
        short nValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 16-bit image set.
     * @param nValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16s_C1MR(
        short nValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_16s_C1MRNative(nValue, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_16s_C1MRNative(
        short nValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 2 channel 16-bit image set.
     * @param aValues New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16s_C2R(
        short[] aValues, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16s_C2RNative(aValues, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16s_C2RNative(
        short[] aValues, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 16-bit image set.
     * @param aValues New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16s_C4R(
        short[] aValues, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16s_C4RNative(aValues, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16s_C4RNative(
        short[] aValues, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 4 channel 16-bit image set.
     * @param aValues New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16s_C4MR(
        short[] aValues, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_16s_C4MRNative(aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_16s_C4MRNative(
        short[] aValues, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 16-bit image set method, not affecting Alpha channel.
     * For RGBA images, this method allows setting of the RGB values without changing the contents
     * of the alpha-channel (fourth channel).
     * @param aValues Three-channel array containing the pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16s_AC4R(
        short[] aValues, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16s_AC4RNative(aValues, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16s_AC4RNative(
        short[] aValues, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 4 channel 16-bit image set method, not affecting Alpha channel.
     * For RGBA images, this method allows setting of the RGB values without changing the contents
     * of the alpha-channel (fourth channel).
     * @param aValues Three-channel array containing the pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16s_AC4MR(
        short[] aValues, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_16s_AC4MRNative(aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_16s_AC4MRNative(
        short[] aValues, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 16-bit unsigned image set affecting only single channel.
     * For RGBA images, this method allows setting of a single of the four (RGBA) values 
     * without changing the contents of the other three channels. The channel is selected
     * via the pDst pointer. The pointer needs to point to the actual first value to be set,
     * e.g. in order to set the R-channel (first channel), one would pass pDst unmodified, since
     * its value actually points to the r channel. If one wanted to modify the B channel (second
     * channel), one would pass pDst + 2 to the function.
     * @param nValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16s_C4CR(
        short nValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16s_C4CRNative(nValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16s_C4CRNative(
        short nValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit complex integer image set.
     * @param oValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16sc_C1R(
        Npp16sc oValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16sc_C1RNative(oValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16sc_C1RNative(
        Npp16sc oValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit complex integer two-channel image set.
     * @param aValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16sc_C2R(
        Npp16sc[] aValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16sc_C2RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16sc_C2RNative(
        Npp16sc[] aValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit complex integer three-channel image set.
     * @param aValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16sc_C3R(
        Npp16sc[] aValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16sc_C3RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16sc_C3RNative(
        Npp16sc[] aValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit complex integer four-channel image set ignoring alpha.
     * @param aValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16sc_AC4R(
        Npp16sc[] aValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16sc_AC4RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16sc_AC4RNative(
        Npp16sc[] aValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit complex integer four-channel image set.
     * @param aValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_16sc_C4R(
        Npp16sc[] aValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_16sc_C4RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_16sc_C4RNative(
        Npp16sc[] aValue, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 32-bit image set.
     * @param nValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32s_C1R(
        int nValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32s_C1RNative(nValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32s_C1RNative(
        int nValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 32-bit image set.
     * @param nValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32s_C1MR(
        int nValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_32s_C1MRNative(nValue, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_32s_C1MRNative(
        int nValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 32-bit image set.
     * @param aValues New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32s_C4R(
        int[] aValues, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32s_C4RNative(aValues, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32s_C4RNative(
        int[] aValues, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 4 channel 32-bit image set.
     * @param aValues New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32s_C4MR(
        int[] aValues, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_32s_C4MRNative(aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_32s_C4MRNative(
        int[] aValues, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 16-bit image set method, not affecting Alpha channel.
     * For RGBA images, this method allows setting of the RGB values without changing the contents
     * of the alpha-channel (fourth channel).
     * @param aValues Three-channel array containing the pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32s_AC4R(
        int[] aValues, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32s_AC4RNative(aValues, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32s_AC4RNative(
        int[] aValues, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 4 channel 16-bit image set method, not affecting Alpha channel.
     * For RGBA images, this method allows setting of the RGB values without changing the contents
     * of the alpha-channel (fourth channel).
     * @param aValues Three-channel array containing the pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32s_AC4MR(
        int[] aValues, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_32s_AC4MRNative(aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_32s_AC4MRNative(
        int[] aValues, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 32-bit unsigned image set affecting only single channel.
     * For RGBA images, this method allows setting of a single of the four (RGBA) values 
     * without changing the contents of the other three channels. The channel is selected
     * via the pDst pointer. The pointer needs to point to the actual first value to be set,
     * e.g. in order to set the R-channel (first channel), one would pass pDst unmodified, since
     * its value actually points to the r channel. If one wanted to modify the B channel (second
     * channel), one would pass pDst + 2 to the function.
     * @param nValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32s_C4CR(
        int nValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32s_C4CRNative(nValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32s_C4CRNative(
        int nValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Single channel 32-bit complex integer image set.
     * @param oValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32sc_C1R(
        Npp32sc oValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32sc_C1RNative(oValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32sc_C1RNative(
        Npp32sc oValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Two channel 32-bit complex integer image set.
     * @param aValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32sc_C2R(
        Npp32sc[] aValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32sc_C2RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32sc_C2RNative(
        Npp32sc[] aValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three channel 32-bit complex integer image set.
     * @param aValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32sc_C3R(
        Npp32sc[] aValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32sc_C3RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32sc_C3RNative(
        Npp32sc[] aValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four channel 32-bit complex integer image set.
     * @param aValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32sc_C4R(
        Npp32sc[] aValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32sc_C4RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32sc_C4RNative(
        Npp32sc[] aValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 32-bit complex integer four-channel image set ignoring alpha.
     * @param aValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32sc_AC4R(
        Npp32sc[] aValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32sc_AC4RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32sc_AC4RNative(
        Npp32sc[] aValue, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 32-bit floating point image set.
     * @param nValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32f_C1R(
        float nValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32f_C1RNative(nValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32f_C1RNative(
        float nValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 32-bit floating point image set.
     * @param nValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32f_C1MR(
        float nValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_32f_C1MRNative(nValue, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_32f_C1MRNative(
        float nValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 32-bit floating point image set.
     * @param aValues New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32f_C4R(
        float[] aValues, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32f_C4RNative(aValues, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32f_C4RNative(
        float[] aValues, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 4 channel 32-bit floating point image set.
     * @param aValues New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32f_C4MR(
        float[] aValues, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_32f_C4MRNative(aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_32f_C4MRNative(
        float[] aValues, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 32-bit floating point image set method, not affecting Alpha channel.
     * For RGBA images, this method allows setting of the RGB values without changing the contents
     * of the alpha-channel (fourth channel).
     * @param aValues Three-channel array containing the pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32f_AC4R(
        float[] aValues, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32f_AC4RNative(aValues, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32f_AC4RNative(
        float[] aValues, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Masked 4 channel 32-bit floating point image set method, not affecting Alpha channel.
     * For RGBA images, this method allows setting of the RGB values without changing the contents
     * of the alpha-channel (fourth channel).
     * @param aValues Three-channel array containing the pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param pMask Pointer to the mask image. This is a single channel 8-bit unsigned int image.
     * @param nMaskStep Number of bytes between line starts of successive lines in the mask image.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32f_AC4MR(
        float[] aValues, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiSet_32f_AC4MRNative(aValues, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiSet_32f_AC4MRNative(
        float[] aValues, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * 4 channel 32-bit floating point image set affecting only single channel.
     * For RGBA images, this method allows setting of a single of the four (RGBA) values 
     * without changing the contents of the other three channels. The channel is selected
     * via the pDst pointer. The pointer needs to point to the actual first value to be set,
     * e.g. in order to set the R-channel (first channel), one would pass pDst unmodified, since
     * its value actually points to the r channel. If one wanted to modify the B channel (second
     * channel), one would pass pDst + 2 to the function.
     * @param nValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32f_C4CR(
        float nValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32f_C4CRNative(nValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32f_C4CRNative(
        float nValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Single channel 32-bit complex image set.
     * @param oValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32fc_C1R(
        Npp32fc oValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32fc_C1RNative(oValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32fc_C1RNative(
        Npp32fc oValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Two channel 32-bit complex image set.
     * @param aValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32fc_C2R(
        Npp32fc[] aValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32fc_C2RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32fc_C2RNative(
        Npp32fc[] aValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three channel 32-bit complex image set.
     * @param aValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32fc_C3R(
        Npp32fc[] aValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32fc_C3RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32fc_C3RNative(
        Npp32fc[] aValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four channel 32-bit complex image set.
     * @param aValue The pixel-value to be set.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32fc_C4R(
        Npp32fc[] aValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32fc_C4RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32fc_C4RNative(
        Npp32fc[] aValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 32-bit complex four-channel image set ignoring alpha.
     * @param aValue New pixel value.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSet_32fc_AC4R(
        Npp32fc[] aValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSet_32fc_AC4RNative(aValue, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSet_32fc_AC4RNative(
        Npp32fc[] aValue, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    ///@}
    /**
     * <pre>
     * @name Image-Memory Copy
     * Copy methods for images of various types. In addition to routines for
     * copying pixels of identical layout from one image to another, there are
     * copy routines for select channels as well as packed-planar conversions:
     * - Select channel to multi-channel copy. E.g. given a three-channel source
     *      and destination image one may copy the second channel of the source
     *      to the third channel of the destination.
     * - Single channel to multi-channel copy. E.g. given a single-channel source
     *      and a four-channel destination, one may copy the contents of the single-
     *      channel source to the second channel of the destination.
     * - Select channel to single-channel copy. E.g. given a three-channel source
     *      and a single-channel destination one may copy the third channel of the 
     *      source to the destination.
     * - Multi-channel to planar copy. These copy operations split a multi-channel
     *      image into a set of single-channel images.
     * - Planar image to multi-channel copy. These copy routines combine separate
     *      color-planes (single channel images) into a single multi-channel image.
     * 
     * </pre>
     */
    ///@{
    /**
     * <pre>
     * 8-bit image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8s_C1R(
        TypedPointer<Npp8s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8s_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8s_C1RNative(
        TypedPointer<Npp8s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Two-channel 8-bit image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8s_C2R(
        TypedPointer<Npp8s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8s_C2RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8s_C2RNative(
        TypedPointer<Npp8s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 8-bit image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8s_C3R(
        TypedPointer<Npp8s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8s_C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8s_C3RNative(
        TypedPointer<Npp8s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 8-bit image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8s_C4R(
        TypedPointer<Npp8s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8s_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8s_C4RNative(
        TypedPointer<Npp8s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 8-bit image copy, ignoring alpha channel.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8s_AC4R(
        TypedPointer<Npp8s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8s_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8s_AC4RNative(
        TypedPointer<Npp8s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 8-bit unsigned image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C1R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_C1RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 8-bit unsigned image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C4R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_C4RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 8-bit unsigned image copy, not affecting Alpha channel.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_AC4R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_AC4RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit unsigned image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C1R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_C1RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 16-bit unsigned image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C4R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_C4RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 16-bit unsigned image copy, not affecting Alpha channel.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_AC4R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_AC4RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C1R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_C1RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 16-bit image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C4R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_C4RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 16-bit image copy, not affecting Alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_AC4R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_AC4RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit complex image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16sc_C1R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16sc_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16sc_C1RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Two-channel 16-bit complex image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16sc_C2R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16sc_C2RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16sc_C2RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 16-bit complex image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16sc_C3R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16sc_C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16sc_C3RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 16-bit complex image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16sc_C4R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16sc_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16sc_C4RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 16-bit complex image copy, ignoring alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16sc_AC4R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16sc_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16sc_AC4RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 32-bit image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C1R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_C1RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 32-bit image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C4R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_C4RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 32-bit image copy, not affecting Alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_AC4R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_AC4RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 32-bit complex image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32sc_C1R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32sc_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32sc_C1RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Two-channel 32-bit complex image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32sc_C2R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32sc_C2RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32sc_C2RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 32-bit complex image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32sc_C3R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32sc_C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32sc_C3RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 32-bit complex image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32sc_C4R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32sc_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32sc_C4RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 32-bit complex image copy, ignoring alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32sc_AC4R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32sc_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32sc_AC4RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 32-bit floating point image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C1R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_C1RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 32-bit floating point image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C4R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_C4RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 32-bit floating point image copy, not affecting Alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_AC4R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_AC4RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 32-bit floating-point complex image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32fc_C1R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32fc_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32fc_C1RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Two-channel 32-bit floating-point complex image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32fc_C2R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32fc_C2RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32fc_C2RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 32-bit floating-point complex image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32fc_C3R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32fc_C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32fc_C3RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 32-bit floating-point complex image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32fc_C4R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32fc_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32fc_C4RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 32-bit floating-point complex image copy, ignoring alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32fc_AC4R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32fc_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32fc_AC4RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * \ref masked_operation 8-bit unsigned image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C1MR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_8u_C1MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_8u_C1MRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation three channel 8-bit unsigned image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C3MR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_8u_C3MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_8u_C3MRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation four channel 8-bit unsigned image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C4MR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_8u_C4MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_8u_C4MRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation four channel 8-bit unsigned image copy, ignoring alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_AC4MR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_8u_AC4MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_8u_AC4MRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation 16-bit unsigned image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C1MR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_16u_C1MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_16u_C1MRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation three channel 16-bit unsigned image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C3MR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_16u_C3MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_16u_C3MRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation four channel 16-bit unsigned image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C4MR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_16u_C4MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_16u_C4MRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation four channel 16-bit unsigned image copy, ignoring alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_AC4MR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_16u_AC4MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_16u_AC4MRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation 16-bit signed image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C1MR(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_16s_C1MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_16s_C1MRNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation three channel 16-bit signed image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C3MR(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_16s_C3MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_16s_C3MRNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation four channel 16-bit signed image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C4MR(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_16s_C4MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_16s_C4MRNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation four channel 16-bit signed image copy, ignoring alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_AC4MR(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_16s_AC4MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_16s_AC4MRNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation 32-bit signed image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C1MR(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_32s_C1MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_32s_C1MRNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation three channel 32-bit signed image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C3MR(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_32s_C3MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_32s_C3MRNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation four channel 32-bit signed image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C4MR(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_32s_C4MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_32s_C4MRNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation four channel 32-bit signed image copy, ignoring alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_AC4MR(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_32s_AC4MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_32s_AC4MRNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation 32-bit float image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C1MR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_32f_C1MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_32f_C1MRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation three channel 32-bit float image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C3MR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_32f_C3MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_32f_C3MRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation four channel 32-bit float image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C4MR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_32f_C4MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_32f_C4MRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * \ref masked_operation four channel 32-bit float image copy, ignoring alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_AC4MR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep)
    {
        return checkResult(nppiCopy_32f_AC4MRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, pMask, nMaskStep));
    }
    private static native int nppiCopy_32f_AC4MRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep);


    /**
     * <pre>
     * Select-channel 8-bit unsigned image copy for three-channel images.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref select_destination_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C3CR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_C3CRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_C3CRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Select-channel 8-bit unsigned image copy for four-channel images.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref select_destination_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C4CR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_C4CRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_C4CRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Select-channel 16-bit signed image copy for three-channel images.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref select_destination_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C3CR(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_C3CRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_C3CRNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Select-channel 16-bit signed image copy for four-channel images.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref select_destination_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C4CR(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_C4CRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_C4CRNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Select-channel 16-bit unsigned image copy for three-channel images.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref select_destination_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C3CR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_C3CRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_C3CRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Select-channel 16-bit unsigned image copy for four-channel images.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref select_destination_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C4CR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_C4CRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_C4CRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Select-channel 32-bit signed image copy for three-channel images.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref select_destination_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C3CR(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_C3CRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_C3CRNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Select-channel 32-bit signed image copy for four-channel images.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref select_destination_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C4CR(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_C4CRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_C4CRNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Select-channel 32-bit float image copy for three-channel images.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref select_destination_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C3CR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_C3CRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_C3CRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Select-channel 32-bit float image copy for four-channel images.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref select_destination_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C4CR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_C4CRNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_C4CRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel to single-channel 8-bit unsigned image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C3C1R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_C3C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_C3C1RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel to single-channel 8-bit unsigned image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C4C1R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_C4C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_C4C1RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel to single-channel 16-bit signed image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C3C1R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_C3C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_C3C1RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel to single-channel 16-bit signed image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C4C1R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_C4C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_C4C1RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel to single-channel 16-bit unsigned image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C3C1R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_C3C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_C3C1RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel to single-channel 16-bit unsigned image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C4C1R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_C4C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_C4C1RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel to single-channel 32-bit signed image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C3C1R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_C3C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_C3C1RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel to single-channel 32-bit signed image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C4C1R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_C4C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_C4C1RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel to single-channel 32-bit float image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C3C1R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_C3C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_C3C1RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel to single-channel 32-bit float image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C4C1R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_C4C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_C4C1RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Single-channel to three-channel 8-bit unsigned image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C1C3R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_C1C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_C1C3RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Single-channel to four-channel 8-bit unsigned image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C1C4R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_C1C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_C1C4RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Single-channel to three-channel 16-bit signed image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C1C3R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_C1C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_C1C3RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Single-channel to four-channel 16-bit signed image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C1C4R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_C1C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_C1C4RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Single-channel to three-channel 16-bit unsigned image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C1C3R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_C1C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_C1C3RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Single-channel to four-channel 16-bit unsigned image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C1C4R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_C1C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_C1C4RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Single-channel to three-channel 32-bit signed image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C1C3R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_C1C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_C1C3RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Single-channel to four-channel 32-bit signed image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C1C4R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_C1C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_C1C4RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Single-channel to three-channel 32-bit float image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C1C3R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_C1C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_C1C3RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Single-channel to four-channel 32-bit float image copy.
     * @param pSrc \ref select_source_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C1C4R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_C1C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_C1C4RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 8-bit unsigned packed to planar image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param aDst Planar \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C3P3R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_C3P3RNative(pSrc, nSrcStep, aDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_C3P3RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 8-bit unsigned packed to planar image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param aDst Planar \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_C4P4R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_C4P4RNative(pSrc, nSrcStep, aDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_C4P4RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 16-bit signed packed to planar image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param aDst Planar \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C3P3R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_C3P3RNative(pSrc, nSrcStep, aDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_C3P3RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 16-bit signed packed to planar image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param aDst Planar \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_C4P4R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_C4P4RNative(pSrc, nSrcStep, aDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_C4P4RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 16-bit unsigned packed to planar image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param aDst Planar \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C3P3R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_C3P3RNative(pSrc, nSrcStep, aDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_C3P3RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 16-bit unsigned packed to planar image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param aDst Planar \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_C4P4R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_C4P4RNative(pSrc, nSrcStep, aDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_C4P4RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 32-bit signed packed to planar image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param aDst Planar \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C3P3R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_C3P3RNative(pSrc, nSrcStep, aDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_C3P3RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 32-bit signed packed to planar image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param aDst Planar \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_C4P4R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_C4P4RNative(pSrc, nSrcStep, aDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_C4P4RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 32-bit float packed to planar image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param aDst Planar \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C3P3R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_C3P3RNative(pSrc, nSrcStep, aDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_C3P3RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 32-bit float packed to planar image copy.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param aDst Planar \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_C4P4R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_C4P4RNative(pSrc, nSrcStep, aDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_C4P4RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f>[] aDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 8-bit unsigned planar to packed image copy.
     * @param aSrc Planar \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_P3C3R(
        TypedPointer<Npp8u>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_P3C3RNative(aSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_P3C3RNative(
        TypedPointer<Npp8u>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 8-bit unsigned planar to packed image copy.
     * @param aSrc Planar \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_8u_P4C4R(
        TypedPointer<Npp8u>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_8u_P4C4RNative(aSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_8u_P4C4RNative(
        TypedPointer<Npp8u>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 16-bit unsigned planar to packed image copy.
     * @param aSrc Planar \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_P3C3R(
        TypedPointer<Npp16u>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_P3C3RNative(aSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_P3C3RNative(
        TypedPointer<Npp16u>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 16-bit unsigned planar to packed image copy.
     * @param aSrc Planar \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16u_P4C4R(
        TypedPointer<Npp16u>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16u_P4C4RNative(aSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16u_P4C4RNative(
        TypedPointer<Npp16u>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 16-bit signed planar to packed image copy.
     * @param aSrc Planar \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_P3C3R(
        TypedPointer<Npp16s>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_P3C3RNative(aSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_P3C3RNative(
        TypedPointer<Npp16s>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 16-bit signed planar to packed image copy.
     * @param aSrc Planar \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_16s_P4C4R(
        TypedPointer<Npp16s>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_16s_P4C4RNative(aSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_16s_P4C4RNative(
        TypedPointer<Npp16s>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 32-bit signed planar to packed image copy.
     * @param aSrc Planar \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_P3C3R(
        TypedPointer<Npp32s>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_P3C3RNative(aSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_P3C3RNative(
        TypedPointer<Npp32s>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 32-bit signed planar to packed image copy.
     * @param aSrc Planar \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32s_P4C4R(
        TypedPointer<Npp32s>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32s_P4C4RNative(aSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32s_P4C4RNative(
        TypedPointer<Npp32s>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three-channel 32-bit float planar to packed image copy.
     * @param aSrc Planar \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_P3C3R(
        TypedPointer<Npp32f>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_P3C3RNative(aSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_P3C3RNative(
        TypedPointer<Npp32f>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four-channel 32-bit float planar to packed image copy.
     * @param aSrc Planar \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopy_32f_P4C4R(
        TypedPointer<Npp32f>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiCopy_32f_P4C4RNative(aSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiCopy_32f_P4C4RNative(
        TypedPointer<Npp32f>[] aSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    ///@}
    /**
     * <pre>
     * @name Bit-Depth Conversion
     * Convert bit-depth up and down.
     *
     * The integer conversion methods do not involve any scaling. Conversions that reduce bit-depth saturate
     * values exceeding the reduced range to the range's maximum/minimum value.
     * When converting from floating-point values to integer values, a rounding mode can be specified. After rounding
     * to integer values the values get saturated to the destination data type's range.
     * </pre>
     */
    ///@{
    /**
     * <pre>
     * 8-bit unsigned to 16-bit unsigned conversion.
     *
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_8u16u_C1R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_8u16u_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_8u16u_C1RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit unsigned to 8-bit unsigned conversion.
     * For detailed documentation see nppiConvert_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_16u8u_C1R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_16u8u_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_16u8u_C1RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 8-bit unsigned to 16-bit unsigned  conversion.
     * For detailed documentation see nppiConvert_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_8u16u_C4R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_8u16u_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_8u16u_C4RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 16-bit unsigned to 8-bit unsigned conversion.
     * For detailed documentation see nppiConvert_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_16u8u_C4R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_16u8u_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_16u8u_C4RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 8-bit unsigned to 16-bit unsigned conversion, not affecting Alpha.
     * For detailed documentation see nppiConvert_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_8u16u_AC4R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_8u16u_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_8u16u_AC4RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 16-bit unsigned to 8-bit unsigned conversion, not affecting Alpha.
     * For detailed documentation see nppiConvert_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_16u8u_AC4R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_16u8u_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_16u8u_AC4RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 8-bit unsigned to 16-bit signed conversion.
     * For detailed documentation see nppiConvert_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_8u16s_C1R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_8u16s_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_8u16s_C1RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 16-bit signed to 8-bit unsigned conversion.
     * For detailed documentation see nppiConvert_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_16s8u_C1R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_16s8u_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_16s8u_C1RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 8-bit unsigned to 16-bit signed conversion.
     * For detailed documentation see nppiConvert_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_8u16s_C4R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_8u16s_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_8u16s_C4RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 16-bit signed to 8-bit unsignedconversion, not affecting Alpha.
     * For detailed documentation see nppiConvert_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_16s8u_C4R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_16s8u_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_16s8u_C4RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 8-bit unsigned to 16-bit signed conversion, not affecting Alpha.
     * For detailed documentation see nppiConverte_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_8u16s_AC4R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_8u16s_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_8u16s_AC4RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 4 channel 16-bit signed to 8-bit unsigned conversion, not affecting Alpha.
     * For detailed documentation see nppiConverte_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_16s8u_AC4R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_16s8u_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_16s8u_AC4RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit singedto 32-bit floating point conversion.
     * For detailed documentation see nppiConverte_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_16s32f_C1R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_16s32f_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_16s32f_C1RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 32-bit floating point to 16-bit conversion.
     * For detailed documentation see nppiConverte_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param eRoundMode Flag specifying how fractional float values are rounded to integer values.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_32f16s_C1R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int eRoundMode)
    {
        return checkResult(nppiConvert_32f16s_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, eRoundMode));
    }
    private static native int nppiConvert_32f16s_C1RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int eRoundMode);


    /**
     * <pre>
     * 8-bit unsigned to 32-bit floating point conversion.
     * For detailed documentation see nppiConverte_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_8u32f_C1R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_8u32f_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_8u32f_C1RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit unsigned to 32-bit floating point conversion.
     * For detailed documentation see nppiConverte_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_16u32f_C1R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_16u32f_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_16u32f_C1RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 32-bit floating point to 16-bit unsigned conversion.
     * For detailed documentation see nppiConverte_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param eRoundMode Flag specifying how fractional float values are rounded to integer values.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_32f16u_C1R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int eRoundMode)
    {
        return checkResult(nppiConvert_32f16u_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, eRoundMode));
    }
    private static native int nppiConvert_32f16u_C1RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int eRoundMode);


    /**
     * <pre>
     * 32-bit floating point to 8-bit unsigned conversion.
     * For detailed documentation see nppiConverte_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param eRoundMode Flag specifying how fractional float values are rounded to integer values.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_32f8u_C1R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int eRoundMode)
    {
        return checkResult(nppiConvert_32f8u_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, eRoundMode));
    }
    private static native int nppiConvert_32f8u_C1RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int eRoundMode);


    /**
     * <pre>
     * 16-bit unsigned to 32-bit signed conversion.
     * For detailed documentation see nppiConverte_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_16u32s_C1R(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_16u32s_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_16u32s_C1RNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * 16-bit to 32-bit  conversion.
     * For detailed documentation see nppiConvert_8u16u_C1R().
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiConvert_16s32s_C1R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiConvert_16s32s_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiConvert_16s32s_C1RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    ///@}
    /** @name Copy Const Border
     * Methods for copying images and padding borders with a constant, user-specifiable color.
     */
    ///@{
    /**
     * <pre>
     * 8-bit unsigned image copy width constant border color.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param oSrcSizeROI Size of the source region of pixels.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oDstSizeROI Size (width, height) of the destination region, i.e. the region that gets filled with
     *      data from the source image (inner part) and constant border color (outer part).
     * @param nTopBorderHeight Height (in pixels) of the top border. The height of the border at the bottom of the
     *      destination ROI is implicitly defined by the size of the source ROI: 
     *      nBottomBorderHeight = oDstSizeROI.height - nTopBorderHeight - oSrcSizeROI.height.
     * @param nLeftBorderWidth Width (in pixels) of the left border. The width of the border at the right side of the
     *      destination ROI is implicitly defined by the size of the source ROI:
     *      nRightBorderWidth = oDstSizeROI.width - nLeftBorderWidth - oSrcSizeROI.width.
     * @param nValue The pixel value to be set for border pixels.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopyConstBorder_8u_C1R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        NppiSize oSrcSizeROI, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oDstSizeROI, 
        int nTopBorderHeight, 
        int nLeftBorderWidth, 
        byte nValue)
    {
        return checkResult(nppiCopyConstBorder_8u_C1RNative(pSrc, nSrcStep, oSrcSizeROI, pDst, nDstStep, oDstSizeROI, nTopBorderHeight, nLeftBorderWidth, nValue));
    }
    private static native int nppiCopyConstBorder_8u_C1RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        NppiSize oSrcSizeROI, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oDstSizeROI, 
        int nTopBorderHeight, 
        int nLeftBorderWidth, 
        byte nValue);


    /**
     * <pre>
     * 4channel 8-bit unsigned image copy with constant border color.
     * See nppiCopyConstBorder_8u_C1R() for detailed documentation.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param oSrcSizeROI Size of the source region-of-interest.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oDstSizeROI Size of the destination region-of-interest.
     * @param nTopBorderHeight Height of top border.
     * @param nLeftBorderWidth Width of left border.
     * @param aValue Vector of the RGBA values of the border pixels to be set.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopyConstBorder_8u_C4R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        NppiSize oSrcSizeROI, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oDstSizeROI, 
        int nTopBorderHeight, 
        int nLeftBorderWidth, 
        byte[] aValue)
    {
        return checkResult(nppiCopyConstBorder_8u_C4RNative(pSrc, nSrcStep, oSrcSizeROI, pDst, nDstStep, oDstSizeROI, nTopBorderHeight, nLeftBorderWidth, aValue));
    }
    private static native int nppiCopyConstBorder_8u_C4RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        NppiSize oSrcSizeROI, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oDstSizeROI, 
        int nTopBorderHeight, 
        int nLeftBorderWidth, 
        byte[] aValue);


    /**
     * <pre>
     * 4 channel 8-bit unsigned image copy with constant border color.
     * See nppiCopyConstBorder_8u_C1R() for detailed documentation.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param oSrcSizeROI Size of the source region-of-interest.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oDstSizeROI Size of the destination region-of-interest.
     * @param nTopBorderHeight Height of top border.
     * @param nLeftBorderWidth Width of left border.
     * @param aValue Vector of the RGB values of the border pixels. Because this method does not
     *      affect the destination image's alpha channel, only three components of the border color
     *      are needed.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopyConstBorder_8u_AC4R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        NppiSize oSrcSizeROI, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oDstSizeROI, 
        int nTopBorderHeight, 
        int nLeftBorderWidth, 
        byte[] aValue)
    {
        return checkResult(nppiCopyConstBorder_8u_AC4RNative(pSrc, nSrcStep, oSrcSizeROI, pDst, nDstStep, oDstSizeROI, nTopBorderHeight, nLeftBorderWidth, aValue));
    }
    private static native int nppiCopyConstBorder_8u_AC4RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        NppiSize oSrcSizeROI, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oDstSizeROI, 
        int nTopBorderHeight, 
        int nLeftBorderWidth, 
        byte[] aValue);


    /**
     * <pre>
     * 32-bit image copy with constant border color.
     * See nppiCopyConstBorder_8u_C1R() for detailed documentation.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param oSrcSizeROI Size of the source region-of-interest.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oDstSizeROI Size of the destination region-of-interest.
     * @param nTopBorderHeight Height of top border.
     * @param nLeftBorderWidth Width of left border.
     * @param nValue Border luminance value.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiCopyConstBorder_32s_C1R(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        NppiSize oSrcSizeROI, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oDstSizeROI, 
        int nTopBorderHeight, 
        int nLeftBorderWidth, 
        int nValue)
    {
        return checkResult(nppiCopyConstBorder_32s_C1RNative(pSrc, nSrcStep, oSrcSizeROI, pDst, nDstStep, oDstSizeROI, nTopBorderHeight, nLeftBorderWidth, nValue));
    }
    private static native int nppiCopyConstBorder_32s_C1RNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        NppiSize oSrcSizeROI, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oDstSizeROI, 
        int nTopBorderHeight, 
        int nLeftBorderWidth, 
        int nValue);


    ///@}
    /**
     * <pre>
     * @name Image Transpose
     * Methods for transposing images of various types. Like matrix transpose, image transpose is a mirror along the image's
     * diagonal (upper-left to lower-right corner).
     * </pre>
     */
    ///@{
    /**
     * <pre>
     * 8-bit image transpose.
     *
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst Pointer to the destination ROI.
     * @param nDstStep \ref destination_image_line_step.
     * @param oROI \ref roi_specification.
     *
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiTranspose_8u_C1R(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oROI)
    {
        return checkResult(nppiTranspose_8u_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oROI));
    }
    private static native int nppiTranspose_8u_C1RNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oROI);


    ///@}
    /**
     * <pre>
     * @name Image Color Channel Swap
     * Methods for exchanging the color channels of an image. The methods support arbitrary permutations of the original
     * channels, including replication.
     * </pre>
     */
    ///@{
    /**
     * <pre>
     * 4 channel 8-bit unsigned swap channels, in-place.
     *
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param aDstOrder Integer array describing how channel values are permutated. The n-th entry
     *      of the array contains the number of the channel that is stored in the n-th channel of
     *      the output image. E.g. Given an RGBA image, aDstOrder = [3,2,1,0] converts this to ABGR
     *      channel order.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSwapChannels_8u_C4IR(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int[] aDstOrder)
    {
        return checkResult(nppiSwapChannels_8u_C4IRNative(pSrcDst, nSrcDstStep, oSizeROI, aDstOrder));
    }
    private static native int nppiSwapChannels_8u_C4IRNative(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int[] aDstOrder);


    ///@}
    ///@} image_data_exchange_and_initialization
    /** @defgroup image_arithmetic Arithmetic and Logical Operations
     */
    ///@{
    /** @name AddC
     * Adds a constant value to each pixel of an image.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_8u_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image add constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_8u_C1IRSfs(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_8u_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_8u_C1IRSfsNative(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel..
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_8u_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel 8-bit unsigned char in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel..
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_8u_C3IRSfs(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_8u_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel..
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_8u_AC4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_8u_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_8u_AC4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel..
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_8u_AC4IRSfs(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_8u_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_8u_AC4IRSfsNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel..
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_8u_C4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_8u_C4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_8u_C4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_8u_C4IRSfs(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_8u_C4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_8u_C4IRSfsNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16u_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image add constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16u_C1IRSfs(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16u_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16u_C1IRSfsNative(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16u_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16u_C3IRSfs(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16u_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16u_AC4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16u_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16u_AC4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16u_AC4IRSfs(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16u_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16u_AC4IRSfsNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16u_C4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16u_C4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16u_C4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16u_C4IRSfs(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16u_C4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16u_C4IRSfsNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16s_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image add constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16s_C1IRSfs(
        short nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16s_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16s_C1IRSfsNative(
        short nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16s_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16s_C3IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16s_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16s_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16s_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16s_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16s_AC4IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16s_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16s_AC4IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16s_C4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16s_C4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16s_C4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16s_C4IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16s_C4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16s_C4IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16sc_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16sc_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16sc_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image add constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16sc_C1IRSfs(
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16sc_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16sc_C1IRSfsNative(
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16sc_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16sc_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16sc_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16sc_C3IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16sc_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16sc_C3IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16sc_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16sc_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16sc_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_16sc_AC4IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_16sc_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_16sc_AC4IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32s_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        int nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_32s_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_32s_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        int nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image add constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32s_C1IRSfs(
        int nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_32s_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_32s_C1IRSfsNative(
        int nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32s_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_32s_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_32s_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer channel in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32s_C3IRSfs(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_32s_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_32s_C3IRSfsNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32sc_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_32sc_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_32sc_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel in place image add constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32sc_C1IRSfs(
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_32sc_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_32sc_C1IRSfsNative(
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32sc_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_32sc_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_32sc_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32sc_C3IRSfs(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_32sc_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_32sc_C3IRSfsNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel with unmodified alpha image add constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32sc_AC4RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_32sc_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_32sc_AC4RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image add constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32sc_AC4IRSfs(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAddC_32sc_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAddC_32sc_AC4IRSfsNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit floating point channel image add constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32f_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        float nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32f_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAddC_32f_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        float nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image add constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32f_C1IR(
        float nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32f_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddC_32f_C1IRNative(
        float nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image add constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32f_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32f_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAddC_32f_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel in place image add constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32f_C3IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32f_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddC_32f_C3IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha image add constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32f_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32f_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAddC_32f_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha in place image add constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32f_AC4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32f_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddC_32f_AC4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image add constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32f_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32f_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAddC_32f_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image add constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32f_C4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32f_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddC_32f_C4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image add constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32fc_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32fc_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAddC_32fc_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image add constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32fc_C1IR(
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32fc_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddC_32fc_C1IRNative(
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image add constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32fc_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32fc_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAddC_32fc_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image add constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32fc_C3IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32fc_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddC_32fc_C3IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel with unmodified alpha image add constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32fc_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32fc_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAddC_32fc_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel with unmodified alpha in place image add constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32fc_AC4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32fc_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddC_32fc_AC4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image add constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32fc_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32fc_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAddC_32fc_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image add constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddC_32fc_C4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddC_32fc_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddC_32fc_C4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of AddC
    ///@}
    /** @name MulC
     * Multiplies each pixel of an image by a constant value.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_8u_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_8u_C1IRSfs(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_8u_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_8u_C1IRSfsNative(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_8u_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel 8-bit unsigned char in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_8u_C3IRSfs(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_8u_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_8u_AC4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_8u_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_8u_AC4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_8u_AC4IRSfs(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_8u_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_8u_AC4IRSfsNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_8u_C4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_8u_C4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_8u_C4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_8u_C4IRSfs(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_8u_C4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_8u_C4IRSfsNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16u_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16u_C1IRSfs(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16u_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16u_C1IRSfsNative(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16u_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16u_C3IRSfs(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16u_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16u_AC4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16u_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16u_AC4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16u_AC4IRSfs(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16u_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16u_AC4IRSfsNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16u_C4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16u_C4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16u_C4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16u_C4IRSfs(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16u_C4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16u_C4IRSfsNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16s_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16s_C1IRSfs(
        short nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16s_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16s_C1IRSfsNative(
        short nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16s_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16s_C3IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16s_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16s_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16s_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16s_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16s_AC4IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16s_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16s_AC4IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16s_C4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16s_C4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16s_C4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16s_C4IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16s_C4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16s_C4IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16sc_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16sc_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16sc_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16sc_C1IRSfs(
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16sc_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16sc_C1IRSfsNative(
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16sc_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16sc_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16sc_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16sc_C3IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16sc_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16sc_C3IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16sc_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16sc_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16sc_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_16sc_AC4IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_16sc_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_16sc_AC4IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32s_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        int nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_32s_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_32s_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        int nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32s_C1IRSfs(
        int nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_32s_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_32s_C1IRSfsNative(
        int nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32s_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_32s_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_32s_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32s_C3IRSfs(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_32s_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_32s_C3IRSfsNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32sc_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_32sc_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_32sc_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32sc_C1IRSfs(
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_32sc_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_32sc_C1IRSfsNative(
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32sc_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_32sc_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_32sc_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32sc_C3IRSfs(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_32sc_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_32sc_C3IRSfsNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel with unmodified alpha image multiply by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32sc_AC4RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_32sc_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_32sc_AC4RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image multiply by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32sc_AC4IRSfs(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMulC_32sc_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMulC_32sc_AC4IRSfsNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit floating point channel image multiply by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32f_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        float nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32f_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulC_32f_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        float nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image multiply by constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32f_C1IR(
        float nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32f_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulC_32f_C1IRNative(
        float nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image multiply by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32f_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32f_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulC_32f_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel in place image multiply by constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32f_C3IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32f_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulC_32f_C3IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha image multiply by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32f_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32f_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulC_32f_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha in place image multiply by constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32f_AC4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32f_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulC_32f_AC4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image multiply by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32f_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32f_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulC_32f_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image multiply by constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32f_C4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32f_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulC_32f_C4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image multiply by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32fc_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32fc_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulC_32fc_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image multiply by constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32fc_C1IR(
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32fc_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulC_32fc_C1IRNative(
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image multiply by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32fc_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32fc_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulC_32fc_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image multiply by constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32fc_C3IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32fc_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulC_32fc_C3IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel with unmodified alpha image multiply by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32fc_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32fc_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulC_32fc_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel with unmodified alpha in place image multiply by constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32fc_AC4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32fc_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulC_32fc_AC4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image multiply by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32fc_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32fc_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulC_32fc_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image multiply by constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulC_32fc_C4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulC_32fc_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulC_32fc_C4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of MulC
    ///@}
    /** @name MulCScale
     * Multiplies each pixel of an image by a constant value then scales the result by the maximum value for the data bit width.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image multiply by constant and scale by max bit width value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_8u_C1R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_8u_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_8u_C1RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image multiply by constant and scale by max bit width value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_8u_C1IR(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_8u_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_8u_C1IRNative(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image multiply by constant and scale by max bit width value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_8u_C3R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_8u_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_8u_C3RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 8-bit unsigned char channel 8-bit unsigned char in place image multiply by constant and scale by max bit width value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_8u_C3IR(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_8u_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_8u_C3IRNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha image multiply by constant and scale by max bit width value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_8u_AC4R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_8u_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_8u_AC4RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha in place image multiply by constant, scale and scale by max bit width value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_8u_AC4IR(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_8u_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_8u_AC4IRNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image multiply by constant and scale by max bit width value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_8u_C4R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_8u_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_8u_C4RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image multiply by constant and scale by max bit width value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_8u_C4IR(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_8u_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_8u_C4IRNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel image multiply by constant and scale by max bit width value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_16u_C1R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_16u_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_16u_C1RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image multiply by constant and scale by max bit width value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_16u_C1IR(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_16u_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_16u_C1IRNative(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image multiply by constant and scale by max bit width value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_16u_C3R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_16u_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_16u_C3RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 16-bit unsigned short channel in place image multiply by constant and scale by max bit width value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_16u_C3IR(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_16u_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_16u_C3IRNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha image multiply by constant and scale by max bit width value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_16u_AC4R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_16u_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_16u_AC4RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha in place image multiply by constant and scale by max bit width value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_16u_AC4IR(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_16u_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_16u_AC4IRNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image multiply by constant and scale by max bit width value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_16u_C4R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_16u_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_16u_C4RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image multiply by constant and scale by max bit width value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulCScale_16u_C4IR(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulCScale_16u_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulCScale_16u_C4IRNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of MulCScale
    ///@}
    /** @name SubC
     * Subtracts a constant value from each pixel of an image.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_8u_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image subtract constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_8u_C1IRSfs(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_8u_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_8u_C1IRSfsNative(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_8u_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel 8-bit unsigned char in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_8u_C3IRSfs(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_8u_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_8u_AC4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_8u_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_8u_AC4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_8u_AC4IRSfs(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_8u_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_8u_AC4IRSfsNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_8u_C4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_8u_C4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_8u_C4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_8u_C4IRSfs(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_8u_C4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_8u_C4IRSfsNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16u_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image subtract constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16u_C1IRSfs(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16u_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16u_C1IRSfsNative(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16u_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16u_C3IRSfs(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16u_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16u_AC4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16u_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16u_AC4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16u_AC4IRSfs(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16u_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16u_AC4IRSfsNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16u_C4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16u_C4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16u_C4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16u_C4IRSfs(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16u_C4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16u_C4IRSfsNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16s_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image subtract constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16s_C1IRSfs(
        short nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16s_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16s_C1IRSfsNative(
        short nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16s_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16s_C3IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16s_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16s_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16s_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16s_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16s_AC4IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16s_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16s_AC4IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16s_C4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16s_C4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16s_C4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16s_C4IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16s_C4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16s_C4IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16sc_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16sc_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16sc_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image subtract constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16sc_C1IRSfs(
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16sc_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16sc_C1IRSfsNative(
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16sc_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16sc_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16sc_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16sc_C3IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16sc_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16sc_C3IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16sc_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16sc_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16sc_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_16sc_AC4IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_16sc_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_16sc_AC4IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32s_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        int nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_32s_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_32s_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        int nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image subtract constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32s_C1IRSfs(
        int nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_32s_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_32s_C1IRSfsNative(
        int nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32s_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_32s_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_32s_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer channel in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32s_C3IRSfs(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_32s_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_32s_C3IRSfsNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32sc_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_32sc_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_32sc_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel in place image subtract constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32sc_C1IRSfs(
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_32sc_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_32sc_C1IRSfsNative(
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32sc_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_32sc_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_32sc_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32sc_C3IRSfs(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_32sc_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_32sc_C3IRSfsNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel with unmodified alpha image subtract constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32sc_AC4RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_32sc_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_32sc_AC4RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image subtract constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32sc_AC4IRSfs(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSubC_32sc_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSubC_32sc_AC4IRSfsNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit floating point channel image subtract constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32f_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        float nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32f_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSubC_32f_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        float nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image subtract constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32f_C1IR(
        float nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32f_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSubC_32f_C1IRNative(
        float nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image subtract constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32f_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32f_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSubC_32f_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel in place image subtract constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32f_C3IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32f_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSubC_32f_C3IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha image subtract constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32f_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32f_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSubC_32f_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha in place image subtract constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32f_AC4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32f_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSubC_32f_AC4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image subtract constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32f_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32f_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSubC_32f_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image subtract constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32f_C4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32f_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSubC_32f_C4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image subtract constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32fc_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32fc_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSubC_32fc_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image subtract constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32fc_C1IR(
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32fc_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSubC_32fc_C1IRNative(
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image subtract constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32fc_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32fc_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSubC_32fc_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image subtract constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32fc_C3IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32fc_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSubC_32fc_C3IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel with unmodified alpha image subtract constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32fc_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32fc_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSubC_32fc_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel with unmodified alpha in place image subtract constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32fc_AC4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32fc_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSubC_32fc_AC4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image subtract constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32fc_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32fc_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSubC_32fc_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image subtract constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSubC_32fc_C4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSubC_32fc_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSubC_32fc_C4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of SubC
    ///@}
    /** @name DivC
     * Divides each pixel of an image by a constant value.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_8u_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image divided by constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_8u_C1IRSfs(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_8u_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_8u_C1IRSfsNative(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_8u_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel 8-bit unsigned char in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_8u_C3IRSfs(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_8u_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_8u_AC4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_8u_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_8u_AC4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_8u_AC4IRSfs(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_8u_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_8u_AC4IRSfsNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_8u_C4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_8u_C4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_8u_C4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_8u_C4IRSfs(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_8u_C4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_8u_C4IRSfsNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16u_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image divided by constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16u_C1IRSfs(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16u_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16u_C1IRSfsNative(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16u_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16u_C3IRSfs(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16u_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16u_AC4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16u_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16u_AC4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16u_AC4IRSfs(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16u_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16u_AC4IRSfsNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16u_C4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16u_C4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16u_C4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16u_C4IRSfs(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16u_C4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16u_C4IRSfsNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16s_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image divided by constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16s_C1IRSfs(
        short nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16s_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16s_C1IRSfsNative(
        short nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16s_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16s_C3IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16s_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16s_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16s_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16s_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16s_AC4IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16s_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16s_AC4IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16s_C4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16s_C4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16s_C4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16s_C4IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16s_C4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16s_C4IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16sc_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16sc_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16sc_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image divided by constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16sc_C1IRSfs(
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16sc_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16sc_C1IRSfsNative(
        Npp16sc nConstant, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16sc_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16sc_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16sc_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16sc_C3IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16sc_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16sc_C3IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16sc_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16sc_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16sc_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_16sc_AC4IRSfs(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_16sc_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_16sc_AC4IRSfsNative(
        TypedPointer<Npp16s> pConstants, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32s_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        int nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_32s_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_32s_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        int nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image divided by constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32s_C1IRSfs(
        int nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_32s_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_32s_C1IRSfsNative(
        int nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32s_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_32s_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_32s_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer channel in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32s_C3IRSfs(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_32s_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_32s_C3IRSfsNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32sc_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_32sc_C1RSfsNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_32sc_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel in place image divided by constant, scale, then clamp to saturated value.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32sc_C1IRSfs(
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_32sc_C1IRSfsNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_32sc_C1IRSfsNative(
        Npp32sc nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32sc_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_32sc_C3RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_32sc_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32sc_C3IRSfs(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_32sc_C3IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_32sc_C3IRSfsNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel with unmodified alpha image divided by constant, scale, then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32sc_AC4RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_32sc_AC4RSfsNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_32sc_AC4RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed complex integer (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image divided by constant, scale, then clamp to saturated value.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32sc_AC4IRSfs(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDivC_32sc_AC4IRSfsNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDivC_32sc_AC4IRSfsNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit floating point channel image divided by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32f_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        float nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32f_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDivC_32f_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        float nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image divided by constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32f_C1IR(
        float nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32f_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDivC_32f_C1IRNative(
        float nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image divided by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32f_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32f_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDivC_32f_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel in place image divided by constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32f_C3IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32f_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDivC_32f_C3IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha image divided by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32f_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32f_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDivC_32f_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha in place image divided by constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32f_AC4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32f_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDivC_32f_AC4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image divided by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32f_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32f_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDivC_32f_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image divided by constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32f_C4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32f_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDivC_32f_C4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image divided by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32fc_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32fc_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDivC_32fc_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image divided by constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32fc_C1IR(
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32fc_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDivC_32fc_C1IRNative(
        Npp32fc nConstant, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image divided by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32fc_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32fc_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDivC_32fc_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image divided by constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32fc_C3IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32fc_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDivC_32fc_C3IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel with unmodified alpha image divided by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32fc_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32fc_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDivC_32fc_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel with unmodified alpha in place image divided by constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32fc_AC4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32fc_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDivC_32fc_AC4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel image divided by constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32fc_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32fc_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDivC_32fc_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit complex floating point (32-bit floating point real, 32-bit floating point imaginary) channel in place image divided by constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDivC_32fc_C4IR(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDivC_32fc_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDivC_32fc_C4IRNative(
        TypedPointer<Npp32f> pConstants, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of DivC
    ///@}
    /** @name AbsDiffC
     * Determines absolute difference between each pixel of an image and a constant value.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image absolute difference with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbsDiffC_8u_C1R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        byte nConstant)
    {
        return checkResult(nppiAbsDiffC_8u_C1RNative(pSrc1, nSrc1Step, pDst, nDstStep, oSizeROI, nConstant));
    }
    private static native int nppiAbsDiffC_8u_C1RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        byte nConstant);


    /**
     * <pre>
     * One 16-bit unsigned short channel image absolute difference with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbsDiffC_16u_C1R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nConstant)
    {
        return checkResult(nppiAbsDiffC_16u_C1RNative(pSrc1, nSrc1Step, pDst, nDstStep, oSizeROI, nConstant));
    }
    private static native int nppiAbsDiffC_16u_C1RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nConstant);


    /**
     * <pre>
     * One 32-bit floating point channel image absolute difference with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbsDiffC_32f_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        float nConstant)
    {
        return checkResult(nppiAbsDiffC_32f_C1RNative(pSrc1, nSrc1Step, pDst, nDstStep, oSizeROI, nConstant));
    }
    private static native int nppiAbsDiffC_32f_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        float nConstant);


    // end of AbsDiffC
    ///@}
    /** @name Add Image
     * Pixel by pixel addition of two images.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_8u_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_8u_C1IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_8u_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_8u_C1IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_8u_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_8u_C3IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_8u_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_8u_AC4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_8u_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_8u_AC4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_8u_AC4IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_8u_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_8u_AC4IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_8u_C4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_8u_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_8u_C4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_8u_C4IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_8u_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_8u_C4IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16u_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16u_C1IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16u_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16u_C1IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16u_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16u_C3IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16u_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16u_AC4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16u_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16u_AC4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16u_AC4IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16u_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16u_AC4IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16u_C4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16u_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16u_C4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16u_C4IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16u_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16u_C4IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16s_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16s_C1IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16s_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16s_C1IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16s_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16s_C3IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16s_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16s_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16s_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16s_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16s_AC4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16s_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16s_AC4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16s_C4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16s_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16s_C4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16s_C4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16s_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16s_C4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16sc_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16sc_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16sc_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16sc_C1IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16sc_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16sc_C1IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16sc_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16sc_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16sc_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16sc_C3IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16sc_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16sc_C3IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16sc_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16sc_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16sc_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_16sc_AC4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_16sc_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_16sc_AC4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32s_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_32s_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_32s_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Note: This function is to be deprecated in future NPP releases, use the function above with a scale factor of 0 instead. 
     * 32-bit image add.
     * Add the pixel values of corresponding pixels in the ROI and write them to the output image.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32s_C1R(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32s_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAdd_32s_C1RNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32s_C1IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_32s_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_32s_C1IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32s_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_32s_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_32s_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32s_C3IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_32s_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_32s_C3IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32sc_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_32sc_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_32sc_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32sc_C1IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_32sc_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_32sc_C1IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32sc_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_32sc_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_32sc_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32sc_C3IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_32sc_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_32sc_C3IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32sc_AC4RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_32sc_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_32sc_AC4RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image addition, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32sc_AC4IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiAdd_32sc_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiAdd_32sc_AC4IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit floating point channel image addition.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32f_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32f_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAdd_32f_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image addition.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32f_C1IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32f_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAdd_32f_C1IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image addition.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32f_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32f_C3RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAdd_32f_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image addition.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32f_C3IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32f_C3IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAdd_32f_C3IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha image addition.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32f_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32f_AC4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAdd_32f_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha in place image addition.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32f_AC4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32f_AC4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAdd_32f_AC4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image addition.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32f_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32f_C4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAdd_32f_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image addition.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32f_C4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32f_C4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAdd_32f_C4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel image addition.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32fc_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32fc_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAdd_32fc_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel in place image addition.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32fc_C1IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32fc_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAdd_32fc_C1IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel image addition.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32fc_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32fc_C3RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAdd_32fc_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel in place image addition.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32fc_C3IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32fc_C3IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAdd_32fc_C3IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha image addition.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32fc_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32fc_AC4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAdd_32fc_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image addition.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32fc_AC4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32fc_AC4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAdd_32fc_AC4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel image addition.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32fc_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32fc_C4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAdd_32fc_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel in place image addition.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAdd_32fc_C4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAdd_32fc_C4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAdd_32fc_C4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of Add Image
    ///@}
    /** @name Add Square Image
     * Pixel by pixel addition of squared pixels from source image to floating point pixel values of destination image.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image squared then added to in place floating point destination image using filter mask (updates destination when mask is non-zero).
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddSquare_8u32f_C1IMR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddSquare_8u32f_C1IMRNative(pSrc, nSrcStep, pMask, nMaskStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddSquare_8u32f_C1IMRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 8-bit unsigned char channel image squared then added to in place floating point destination image.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddSquare_8u32f_C1IR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddSquare_8u32f_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddSquare_8u32f_C1IRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel image squared then added to in place floating point destination image using filter mask (updates destination when mask is non-zero).
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddSquare_16u32f_C1IMR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddSquare_16u32f_C1IMRNative(pSrc, nSrcStep, pMask, nMaskStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddSquare_16u32f_C1IMRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel image squared then added to in place floating point destination image.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddSquare_16u32f_C1IR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddSquare_16u32f_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddSquare_16u32f_C1IRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel image squared then added to in place floating point destination image using filter mask (updates destination when mask is non-zero).
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddSquare_32f_C1IMR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddSquare_32f_C1IMRNative(pSrc, nSrcStep, pMask, nMaskStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddSquare_32f_C1IMRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel image squared then added to in place floating point destination image.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddSquare_32f_C1IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddSquare_32f_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddSquare_32f_C1IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of Add Square Image
    ///@}
    /** @name Add Product Image
     * Pixel by pixel addition of product of pixels from two source images to floating point pixel values of destination image.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image product added to in place floating point destination image using filter mask (updates destination when mask is non-zero).
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddProduct_8u32f_C1IMR(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddProduct_8u32f_C1IMRNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pMask, nMaskStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddProduct_8u32f_C1IMRNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 8-bit unsigned char channel image product added to in place floating point destination image.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddProduct_8u32f_C1IR(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddProduct_8u32f_C1IRNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddProduct_8u32f_C1IRNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel image product added to in place floating point destination image using filter mask (updates destination when mask is non-zero).
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddProduct_16u32f_C1IMR(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddProduct_16u32f_C1IMRNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pMask, nMaskStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddProduct_16u32f_C1IMRNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel image product added to in place floating point destination image.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddProduct_16u32f_C1IR(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddProduct_16u32f_C1IRNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddProduct_16u32f_C1IRNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel image product added to in place floating point destination image using filter mask (updates destination when mask is non-zero).
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddProduct_32f_C1IMR(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddProduct_32f_C1IMRNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pMask, nMaskStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddProduct_32f_C1IMRNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel image product added to in place floating point destination image.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddProduct_32f_C1IR(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAddProduct_32f_C1IRNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAddProduct_32f_C1IRNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of Add Product Image
    ///@}
    /** @name Add Weighted Image
     * Pixel by pixel addition of alpha weighted pixel values from a source image to floating point pixel values of destination image.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel alpha weighted image added to in place floating point destination image using filter mask (updates destination when mask is non-zero).
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nAlpha Alpha weight to be applied to source image pixels (0.0F to 1.0F)
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddWeighted_8u32f_C1IMR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        float nAlpha)
    {
        return checkResult(nppiAddWeighted_8u32f_C1IMRNative(pSrc, nSrcStep, pMask, nMaskStep, pSrcDst, nSrcDstStep, oSizeROI, nAlpha));
    }
    private static native int nppiAddWeighted_8u32f_C1IMRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        float nAlpha);


    /**
     * <pre>
     * One 8-bit unsigned char channel alpha weighted image added to in place floating point destination image.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nAlpha Alpha weight to be applied to source image pixels (0.0F to 1.0F)
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddWeighted_8u32f_C1IR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        float nAlpha)
    {
        return checkResult(nppiAddWeighted_8u32f_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nAlpha));
    }
    private static native int nppiAddWeighted_8u32f_C1IRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        float nAlpha);


    /**
     * <pre>
     * One 16-bit unsigned short channel alpha weighted image added to in place floating point destination image using filter mask (updates destination when mask is non-zero).
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nAlpha Alpha weight to be applied to source image pixels (0.0F to 1.0F)
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddWeighted_16u32f_C1IMR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        float nAlpha)
    {
        return checkResult(nppiAddWeighted_16u32f_C1IMRNative(pSrc, nSrcStep, pMask, nMaskStep, pSrcDst, nSrcDstStep, oSizeROI, nAlpha));
    }
    private static native int nppiAddWeighted_16u32f_C1IMRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        float nAlpha);


    /**
     * <pre>
     * One 16-bit unsigned short channel alpha weighted image added to in place floating point destination image.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nAlpha Alpha weight to be applied to source image pixels (0.0F to 1.0F)
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddWeighted_16u32f_C1IR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        float nAlpha)
    {
        return checkResult(nppiAddWeighted_16u32f_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nAlpha));
    }
    private static native int nppiAddWeighted_16u32f_C1IRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        float nAlpha);


    /**
     * <pre>
     * One 32-bit floating point channel alpha weighted image added to in place floating point destination image using filter mask (updates destination when mask is non-zero).
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pMask \ref mask_image_pointer.
     * @param nMaskStep \ref mask_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nAlpha Alpha weight to be applied to source image pixels (0.0F to 1.0F)
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddWeighted_32f_C1IMR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        float nAlpha)
    {
        return checkResult(nppiAddWeighted_32f_C1IMRNative(pSrc, nSrcStep, pMask, nMaskStep, pSrcDst, nSrcDstStep, oSizeROI, nAlpha));
    }
    private static native int nppiAddWeighted_32f_C1IMRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pMask, 
        int nMaskStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        float nAlpha);


    /**
     * <pre>
     * One 32-bit floating point channel alpha weighted image added to in place floating point destination image.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nAlpha Alpha weight to be applied to source image pixels (0.0F to 1.0F)
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAddWeighted_32f_C1IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        float nAlpha)
    {
        return checkResult(nppiAddWeighted_32f_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nAlpha));
    }
    private static native int nppiAddWeighted_32f_C1IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        float nAlpha);


    // end of Add Weighted Image
    ///@}
    /** @name Mul Image
     * Pixel by pixel multiply of two images.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_8u_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_8u_C1IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_8u_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_8u_C1IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_8u_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_8u_C3IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_8u_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_8u_AC4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_8u_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_8u_AC4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_8u_AC4IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_8u_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_8u_AC4IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_8u_C4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_8u_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_8u_C4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_8u_C4IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_8u_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_8u_C4IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16u_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16u_C1IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16u_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16u_C1IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16u_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16u_C3IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16u_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16u_AC4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16u_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16u_AC4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16u_AC4IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16u_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16u_AC4IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16u_C4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16u_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16u_C4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16u_C4IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16u_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16u_C4IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16s_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16s_C1IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16s_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16s_C1IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16s_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16s_C3IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16s_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16s_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16s_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16s_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16s_AC4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16s_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16s_AC4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16s_C4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16s_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16s_C4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16s_C4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16s_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16s_C4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16sc_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16sc_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16sc_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16sc_C1IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16sc_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16sc_C1IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16sc_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16sc_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16sc_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16sc_C3IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16sc_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16sc_C3IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16sc_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16sc_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16sc_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_16sc_AC4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_16sc_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_16sc_AC4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32s_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_32s_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_32s_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Note: This function is to be deprecated in future NPP releases, use the function above with a scale factor of 0 instead.
     * 1 channel 32-bit image multiplication.
     * Multiply corresponding pixels in ROI. 
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32s_C1R(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32s_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMul_32s_C1RNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32s_C1IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_32s_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_32s_C1IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32s_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_32s_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_32s_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32s_C3IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_32s_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_32s_C3IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32sc_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_32sc_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_32sc_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32sc_C1IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_32sc_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_32sc_C1IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32sc_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_32sc_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_32sc_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32sc_C3IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_32sc_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_32sc_C3IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32sc_AC4RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_32sc_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_32sc_AC4RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image multiplication, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32sc_AC4IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiMul_32sc_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiMul_32sc_AC4IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit floating point channel image multiplication.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32f_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32f_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMul_32f_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image multiplication.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32f_C1IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32f_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMul_32f_C1IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image multiplication.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32f_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32f_C3RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMul_32f_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image multiplication.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32f_C3IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32f_C3IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMul_32f_C3IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha image multiplication.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32f_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32f_AC4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMul_32f_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha in place image multiplication.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32f_AC4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32f_AC4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMul_32f_AC4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image multiplication.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32f_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32f_C4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMul_32f_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image multiplication.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32f_C4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32f_C4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMul_32f_C4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel image multiplication.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32fc_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32fc_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMul_32fc_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel in place image multiplication.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32fc_C1IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32fc_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMul_32fc_C1IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel image multiplication.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32fc_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32fc_C3RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMul_32fc_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel in place image multiplication.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32fc_C3IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32fc_C3IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMul_32fc_C3IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha image multiplication.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32fc_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32fc_AC4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMul_32fc_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image multiplication.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32fc_AC4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32fc_AC4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMul_32fc_AC4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel image multiplication.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32fc_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32fc_C4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMul_32fc_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel in place image multiplication.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMul_32fc_C4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMul_32fc_C4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMul_32fc_C4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of Mul Image
    ///@}
    /** @name MulScale Image
     * Pixel by pixel multiplies each pixel of two images then scales the result by the maximum value for the data bit width.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_8u_C1R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_8u_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulScale_8u_C1RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_8u_C1IR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_8u_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulScale_8u_C1IRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_8u_C3R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_8u_C3RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulScale_8u_C3RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_8u_C3IR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_8u_C3IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulScale_8u_C3IRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_8u_AC4R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_8u_AC4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulScale_8u_AC4RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha in place image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_8u_AC4IR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_8u_AC4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulScale_8u_AC4IRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_8u_C4R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_8u_C4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulScale_8u_C4RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_8u_C4IR(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_8u_C4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulScale_8u_C4IRNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_16u_C1R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_16u_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulScale_16u_C1RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_16u_C1IR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_16u_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulScale_16u_C1IRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_16u_C3R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_16u_C3RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulScale_16u_C3RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_16u_C3IR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_16u_C3IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulScale_16u_C3IRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_16u_AC4R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_16u_AC4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulScale_16u_AC4RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha in place image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_16u_AC4IR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_16u_AC4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulScale_16u_AC4IRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_16u_C4R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_16u_C4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiMulScale_16u_C4RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image multiplication then scale by maximum value for pixel bit width.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiMulScale_16u_C4IR(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiMulScale_16u_C4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiMulScale_16u_C4IRNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of MulScale Image
    ///@}
    /** @name Sub Image
     * Pixel by pixel subtraction of two images.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_8u_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_8u_C1IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_8u_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_8u_C1IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_8u_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_8u_C3IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_8u_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_8u_AC4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_8u_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_8u_AC4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_8u_AC4IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_8u_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_8u_AC4IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_8u_C4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_8u_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_8u_C4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_8u_C4IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_8u_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_8u_C4IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16u_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16u_C1IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16u_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16u_C1IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16u_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16u_C3IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16u_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16u_AC4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16u_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16u_AC4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16u_AC4IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16u_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16u_AC4IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16u_C4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16u_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16u_C4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16u_C4IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16u_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16u_C4IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16s_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16s_C1IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16s_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16s_C1IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16s_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16s_C3IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16s_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16s_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16s_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16s_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16s_AC4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16s_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16s_AC4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16s_C4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16s_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16s_C4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16s_C4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16s_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16s_C4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16sc_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16sc_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16sc_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16sc_C1IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16sc_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16sc_C1IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16sc_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16sc_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16sc_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16sc_C3IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16sc_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16sc_C3IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16sc_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16sc_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16sc_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_16sc_AC4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_16sc_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_16sc_AC4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32s_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_32s_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_32s_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Note: This function is to be deprecated in future NPP releases, use the function above with a scale factor of 0 instead. 
     * 32-bit image subtraction.
     * Subtract pSrc1's pixels from corresponding pixels in pSrc2. 
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32s_C1R(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32s_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSub_32s_C1RNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32s_C1IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_32s_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_32s_C1IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32s_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_32s_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_32s_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32s_C3IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_32s_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_32s_C3IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32sc_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_32sc_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_32sc_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32sc_C1IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_32sc_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_32sc_C1IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32sc_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_32sc_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_32sc_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32sc_C3IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_32sc_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_32sc_C3IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32sc_AC4RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_32sc_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_32sc_AC4RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image subtraction, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32sc_AC4IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSub_32sc_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSub_32sc_AC4IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit floating point channel image subtraction.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32f_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32f_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSub_32f_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image subtraction.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32f_C1IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32f_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSub_32f_C1IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image subtraction.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32f_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32f_C3RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSub_32f_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image subtraction.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32f_C3IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32f_C3IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSub_32f_C3IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha image subtraction.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32f_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32f_AC4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSub_32f_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha in place image subtraction.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32f_AC4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32f_AC4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSub_32f_AC4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image subtraction.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32f_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32f_C4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSub_32f_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image subtraction.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32f_C4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32f_C4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSub_32f_C4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel image subtraction.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32fc_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32fc_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSub_32fc_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel in place image subtraction.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32fc_C1IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32fc_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSub_32fc_C1IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel image subtraction.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32fc_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32fc_C3RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSub_32fc_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel in place image subtraction.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32fc_C3IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32fc_C3IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSub_32fc_C3IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha image subtraction.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32fc_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32fc_AC4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSub_32fc_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image subtraction.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32fc_AC4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32fc_AC4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSub_32fc_AC4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel image subtraction.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32fc_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32fc_C4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSub_32fc_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel in place image subtraction.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSub_32fc_C4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSub_32fc_C4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSub_32fc_C4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of Sub Image
    ///@}
    /** @name Div Image
     * Pixel by pixel division of two images.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_8u_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_8u_C1IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_8u_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_8u_C1IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_8u_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_8u_C3IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_8u_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_8u_AC4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_8u_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_8u_AC4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel with unmodified alpha in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_8u_AC4IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_8u_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_8u_AC4IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_8u_C4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_8u_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_8u_C4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_8u_C4IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_8u_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_8u_C4IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16u_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16u_C1IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16u_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16u_C1IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16u_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16u_C3IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16u_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16u_AC4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16u_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16u_AC4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel with unmodified alpha in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16u_AC4IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16u_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16u_AC4IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16u_C4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16u_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16u_C4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16u_C4IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16u_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16u_C4IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16s_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16s_C1IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16s_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16s_C1IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16s_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16s_C3IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16s_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16s_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16s_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16s_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel with unmodified alpha in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16s_AC4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16s_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16s_AC4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16s_C4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16s_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16s_C4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16s_C4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16s_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16s_C4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16sc_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16sc_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16sc_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16sc_C1IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16sc_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16sc_C1IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16sc_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16sc_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16sc_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16sc_C3IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16sc_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16sc_C3IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16sc_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16sc_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16sc_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short complex number (16-bit real, 16-bit imaginary) channel with unmodified alpha in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_16sc_AC4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_16sc_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_16sc_AC4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32s_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_32s_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_32s_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Note: This function is to be deprecated in future NPP releases, use the function above with a scale factor of 0 instead. 
     * 32-bit image division.
     * Divide pixels in pSrc2 by pSrc1's pixels. 
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32s_C1R(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32s_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDiv_32s_C1RNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32s_C1IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_32s_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_32s_C1IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32s_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_32s_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_32s_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32s_C3IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_32s_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_32s_C3IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32sc_C1RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_32sc_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_32sc_C1RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32sc_C1IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_32sc_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_32sc_C1IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32sc_C3RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_32sc_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_32sc_C3RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32sc_C3IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_32sc_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_32sc_C3IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32sc_AC4RSfs(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_32sc_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_32sc_AC4RSfsNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 32-bit signed integer complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32sc_AC4IRSfs(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_32sc_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiDiv_32sc_AC4IRSfsNative(
        TypedPointer<Npp32s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit floating point channel image division.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32f_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32f_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDiv_32f_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image division.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32f_C1IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32f_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDiv_32f_C1IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image division.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32f_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32f_C3RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDiv_32f_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image division.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32f_C3IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32f_C3IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDiv_32f_C3IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha image division.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32f_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32f_AC4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDiv_32f_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel with unmodified alpha in place image division.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32f_AC4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32f_AC4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDiv_32f_AC4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image division.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32f_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32f_C4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDiv_32f_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image division.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32f_C4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32f_C4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDiv_32f_C4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel image division.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32fc_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32fc_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDiv_32fc_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel in place image division.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32fc_C1IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32fc_C1IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDiv_32fc_C1IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel image division.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32fc_C3R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32fc_C3RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDiv_32fc_C3RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel in place image division.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32fc_C3IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32fc_C3IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDiv_32fc_C3IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha image division.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32fc_AC4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32fc_AC4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDiv_32fc_AC4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel with unmodified alpha in place image division.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32fc_AC4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32fc_AC4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDiv_32fc_AC4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel image division.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32fc_C4R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32fc_C4RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiDiv_32fc_C4RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point complex number (32-bit real, 32-bit imaginary) channel in place image division.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_32fc_C4IR(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiDiv_32fc_C4IRNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiDiv_32fc_C4IRNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of Div Image
    ///@}
    /** @name Div_Round Image
     * Pixel by pixel division of two images using result rounding modes.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_8u_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_8u_C1IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_8u_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_8u_C1IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_8u_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_8u_C3IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_8u_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image division with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_8u_AC4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_8u_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_8u_AC4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image division with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_8u_AC4IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_8u_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_8u_AC4IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_8u_C4RSfs(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_8u_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_8u_C4RSfsNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_8u_C4IRSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_8u_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_8u_C4IRSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16u_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16u_C1IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16u_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16u_C1IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16u_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16u_C3IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16u_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image division with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16u_AC4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16u_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16u_AC4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image division with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16u_AC4IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16u_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16u_AC4IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16u_C4RSfs(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16u_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16u_C4RSfsNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16u_C4IRSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16u_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16u_C4IRSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16s_C1RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16s_C1IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16s_C1IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16s_C1IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16s_C3RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16s_C3IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16s_C3IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image division with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16s_AC4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16s_AC4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16s_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image division with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16s_AC4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16s_AC4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16s_AC4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16s_C4RSfs(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16s_C4RSfsNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16s_C4RSfsNative(
        TypedPointer<Npp16s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16s> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image division, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param rndMode Result Rounding mode to be used (NPP_RND_ZERO, NPP_RND_NEAR, or NP_RND_FINANCIAL)
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiDiv_Round_16s_C4IRSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor)
    {
        return checkResult(nppiDiv_Round_16s_C4IRSfsNative(pSrc, nSrcStep, pSrcDst, nSrcDstStep, oSizeROI, rndMode, nScaleFactor));
    }
    private static native int nppiDiv_Round_16s_C4IRSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int rndMode, 
        int nScaleFactor);


    // end of Div_Round Image
    ///@}
    /** @name Abs Image
     * Absolute value of each pixel value in an image.
     */
    ///@{
    /**
     * <pre>
     * One 16-bit signed short channel image absolute value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_16s_C1R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_16s_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAbs_16s_C1RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit signed short channel in place image absolute value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_16s_C1IR(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_16s_C1IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAbs_16s_C1IRNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 16-bit signed short channel image absolute value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_16s_C3R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_16s_C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAbs_16s_C3RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 16-bit signed short channel in place image absolute value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_16s_C3IR(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_16s_C3IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAbs_16s_C3IRNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit signed short channel image absolute value with unmodified alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_16s_AC4R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_16s_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAbs_16s_AC4RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image absolute value with unmodified alpha.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_16s_AC4IR(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_16s_AC4IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAbs_16s_AC4IRNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit signed short channel image absolute value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_16s_C4R(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_16s_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAbs_16s_C4RNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image absolute value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_16s_C4IR(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_16s_C4IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAbs_16s_C4IRNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel image absolute value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_32f_C1R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_32f_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAbs_32f_C1RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image absolute value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_32f_C1IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_32f_C1IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAbs_32f_C1IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image absolute value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_32f_C3R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_32f_C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAbs_32f_C3RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel in place image absolute value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_32f_C3IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_32f_C3IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAbs_32f_C3IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image absolute value with unmodified alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_32f_AC4R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_32f_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAbs_32f_AC4RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image absolute value with unmodified alpha.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_32f_AC4IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_32f_AC4IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAbs_32f_AC4IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image absolute value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_32f_C4R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_32f_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAbs_32f_C4RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image absolute value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbs_32f_C4IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbs_32f_C4IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAbs_32f_C4IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of Abs Image
    ///@}
    /** @name AbsDiff Image
     * Pixel by pixel absolute difference between two images.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel absolute difference of image1 minus image2.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbsDiff_8u_C1R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbsDiff_8u_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAbsDiff_8u_C1RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel absolute difference of image1 minus image2.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbsDiff_16u_C1R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbsDiff_16u_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAbsDiff_16u_C1RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel absolute difference of image1 minus image2.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pSrc2 \ref source_image_pointer.
     * @param nSrc2Step \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAbsDiff_32f_C1R(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAbsDiff_32f_C1RNative(pSrc1, nSrc1Step, pSrc2, nSrc2Step, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAbsDiff_32f_C1RNative(
        TypedPointer<Npp32f> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32f> pSrc2, 
        int nSrc2Step, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    // end of AbsDiff Image
    ///@}
    /** @name Sqr Image
     * Square each pixel in an image.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_8u_C1RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_8u_C1IRSfs(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_8u_C1IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_8u_C1IRSfsNative(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_8u_C3RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel in place image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_8u_C3IRSfs(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_8u_C3IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image squared with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_8u_AC4RSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_8u_AC4RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_8u_AC4RSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image squared with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_8u_AC4IRSfs(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_8u_AC4IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_8u_AC4IRSfsNative(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_8u_C4RSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_8u_C4RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_8u_C4RSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_8u_C4IRSfs(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_8u_C4IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_8u_C4IRSfsNative(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16u_C1RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16u_C1IRSfs(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16u_C1IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16u_C1IRSfsNative(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16u_C3RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel in place image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16u_C3IRSfs(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16u_C3IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image squared with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16u_AC4RSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16u_AC4RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16u_AC4RSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image squared with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16u_AC4IRSfs(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16u_AC4IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16u_AC4IRSfsNative(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16u_C4RSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16u_C4RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16u_C4RSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16u_C4IRSfs(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16u_C4IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16u_C4IRSfsNative(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16s_C1RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16s_C1IRSfs(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16s_C1IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16s_C1IRSfsNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16s_C3RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel in place image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16s_C3IRSfs(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16s_C3IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image squared with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16s_AC4RSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16s_AC4RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16s_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image squared with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16s_AC4IRSfs(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16s_AC4IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16s_AC4IRSfsNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16s_C4RSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16s_C4RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16s_C4RSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image squared, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_16s_C4IRSfs(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqr_16s_C4IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqr_16s_C4IRSfsNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit floating point channel image squared.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_32f_C1R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqr_32f_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSqr_32f_C1RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image squared.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_32f_C1IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqr_32f_C1IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSqr_32f_C1IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image squared.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_32f_C3R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqr_32f_C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSqr_32f_C3RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel in place image squared.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_32f_C3IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqr_32f_C3IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSqr_32f_C3IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image squared with unmodified alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_32f_AC4R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqr_32f_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSqr_32f_AC4RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image squared with unmodified alpha.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_32f_AC4IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqr_32f_AC4IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSqr_32f_AC4IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image squared.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_32f_C4R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqr_32f_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSqr_32f_C4RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image squared.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqr_32f_C4IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqr_32f_C4IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSqr_32f_C4IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of Sqr Image
    ///@}
    /** @name Sqrt Image
     * Pixel by pixel square root of each pixel in an image.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image square root, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_8u_C1RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image square root, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_8u_C1IRSfs(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_8u_C1IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_8u_C1IRSfsNative(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image square root, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_8u_C3RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel in place image square root, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_8u_C3IRSfs(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_8u_C3IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image square root with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_8u_AC4RSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_8u_AC4RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_8u_AC4RSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image square root with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_8u_AC4IRSfs(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_8u_AC4IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_8u_AC4IRSfsNative(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image square root, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_16u_C1RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image square root, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_16u_C1IRSfs(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_16u_C1IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_16u_C1IRSfsNative(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image square root, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_16u_C3RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel in place image square root, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_16u_C3IRSfs(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_16u_C3IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image square root with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_16u_AC4RSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_16u_AC4RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_16u_AC4RSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image square root with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_16u_AC4IRSfs(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_16u_AC4IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_16u_AC4IRSfsNative(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image square root, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_16s_C1RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image square root, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_16s_C1IRSfs(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_16s_C1IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_16s_C1IRSfsNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image square root, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_16s_C3RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel in place image square root, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_16s_C3IRSfs(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_16s_C3IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel image square root with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_16s_AC4RSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_16s_AC4RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_16s_AC4RSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Four 16-bit signed short channel in place image square root with unmodified alpha, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_16s_AC4IRSfs(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiSqrt_16s_AC4IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiSqrt_16s_AC4IRSfsNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit floating point channel image square root.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_32f_C1R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqrt_32f_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSqrt_32f_C1RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image square root.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_32f_C1IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqrt_32f_C1IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSqrt_32f_C1IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image square root.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_32f_C3R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqrt_32f_C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSqrt_32f_C3RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel in place image square root.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_32f_C3IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqrt_32f_C3IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSqrt_32f_C3IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image square root with unmodified alpha.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_32f_AC4R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqrt_32f_AC4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSqrt_32f_AC4RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image square root with unmodified alpha.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_32f_AC4IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqrt_32f_AC4IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSqrt_32f_AC4IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel image square root.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_32f_C4R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqrt_32f_C4RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiSqrt_32f_C4RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit floating point channel in place image square root.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiSqrt_32f_C4IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiSqrt_32f_C4IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiSqrt_32f_C4IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of Sqrt Image
    ///@}
    /** @name Ln Image
     * Pixel by pixel natural logarithm of each pixel in an image.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image natural logarithm, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiLn_8u_C1RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiLn_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image natural logarithm, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_8u_C1IRSfs(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiLn_8u_C1IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiLn_8u_C1IRSfsNative(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image natural logarithm, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiLn_8u_C3RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiLn_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel in place image natural logarithm, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_8u_C3IRSfs(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiLn_8u_C3IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiLn_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image natural logarithm, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiLn_16u_C1RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiLn_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image natural logarithm, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_16u_C1IRSfs(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiLn_16u_C1IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiLn_16u_C1IRSfsNative(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image natural logarithm, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiLn_16u_C3RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiLn_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel in place image natural logarithm, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_16u_C3IRSfs(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiLn_16u_C3IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiLn_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image natural logarithm, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiLn_16s_C1RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiLn_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image natural logarithm, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_16s_C1IRSfs(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiLn_16s_C1IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiLn_16s_C1IRSfsNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image natural logarithm, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiLn_16s_C3RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiLn_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel in place image natural logarithm, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_16s_C3IRSfs(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiLn_16s_C3IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiLn_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit floating point channel image natural logarithm.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_32f_C1R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiLn_32f_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiLn_32f_C1RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image natural logarithm.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_32f_C1IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiLn_32f_C1IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiLn_32f_C1IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image natural logarithm.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_32f_C3R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiLn_32f_C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiLn_32f_C3RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel in place image natural logarithm.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiLn_32f_C3IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiLn_32f_C3IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiLn_32f_C3IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of Ln Image
    ///@}
    /** @name Exp Image
     * Exponential value of each pixel in an image.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image exponential, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_8u_C1RSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiExp_8u_C1RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiExp_8u_C1RSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image exponential, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_8u_C1IRSfs(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiExp_8u_C1IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiExp_8u_C1IRSfsNative(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image exponential, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_8u_C3RSfs(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiExp_8u_C3RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiExp_8u_C3RSfsNative(
        TypedPointer<Npp8u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 8-bit unsigned char channel in place image exponential, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_8u_C3IRSfs(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiExp_8u_C3IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiExp_8u_C3IRSfsNative(
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel image exponential, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_16u_C1RSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiExp_16u_C1RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiExp_16u_C1RSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image exponential, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_16u_C1IRSfs(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiExp_16u_C1IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiExp_16u_C1IRSfsNative(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image exponential, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_16u_C3RSfs(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiExp_16u_C3RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiExp_16u_C3RSfsNative(
        TypedPointer<Npp16u> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit unsigned short channel in place image exponential, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_16u_C3IRSfs(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiExp_16u_C3IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiExp_16u_C3IRSfsNative(
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel image exponential, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_16s_C1RSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiExp_16s_C1RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiExp_16s_C1RSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 16-bit signed short channel in place image exponential, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_16s_C1IRSfs(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiExp_16s_C1IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiExp_16s_C1IRSfsNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel image exponential, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_16s_C3RSfs(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiExp_16s_C3RSfsNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiExp_16s_C3RSfsNative(
        TypedPointer<Npp16s> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp16s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * Three 16-bit signed short channel in place image exponential, scale by 2^(-nScaleFactor), then clamp to saturated value.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @param nScaleFactor \ref integer_result_scaling.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_16s_C3IRSfs(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor)
    {
        return checkResult(nppiExp_16s_C3IRSfsNative(pSrcDst, nSrcDstStep, oSizeROI, nScaleFactor));
    }
    private static native int nppiExp_16s_C3IRSfsNative(
        TypedPointer<Npp16s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI, 
        int nScaleFactor);


    /**
     * <pre>
     * One 32-bit floating point channel image exponential.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_32f_C1R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiExp_32f_C1RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiExp_32f_C1RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit floating point channel in place image exponential.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_32f_C1IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiExp_32f_C1IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiExp_32f_C1IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel image exponential.
     * @param pSrc \ref source_image_pointer.
     * @param nSrcStep \ref source_image_line_step.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_32f_C3R(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiExp_32f_C3RNative(pSrc, nSrcStep, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiExp_32f_C3RNative(
        TypedPointer<Npp32f> pSrc, 
        int nSrcStep, 
        TypedPointer<Npp32f> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit floating point channel in place image exponential.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiExp_32f_C3IR(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiExp_32f_C3IRNative(pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiExp_32f_C3IRNative(
        TypedPointer<Npp32f> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of Exp Image
    ///@}
    /** @name AndC Image
     * Pixel by pixel logical and of an image with a constant.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image logical and with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_8u_C1R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_8u_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAndC_8u_C1RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image logical and with constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_8u_C1IR(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_8u_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAndC_8u_C1IRNative(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image logical and with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_8u_C3R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_8u_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAndC_8u_C3RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 8-bit unsigned char channel in place image logical and with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_8u_C3IR(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_8u_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAndC_8u_C3IRNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image logical and with constant with unmodified alpha.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_8u_AC4R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_8u_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAndC_8u_AC4RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image logical and with constant with unmodified alpha.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_8u_AC4IR(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_8u_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAndC_8u_AC4IRNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image logical and with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_8u_C4R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_8u_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAndC_8u_C4RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image logical and with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_8u_C4IR(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_8u_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAndC_8u_C4IRNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel image logical and with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_16u_C1R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_16u_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAndC_16u_C1RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image logical and with constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_16u_C1IR(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_16u_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAndC_16u_C1IRNative(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image logical and with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_16u_C3R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_16u_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAndC_16u_C3RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 16-bit unsigned short channel in place image logical and with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_16u_C3IR(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_16u_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAndC_16u_C3IRNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image logical and with constant with unmodified alpha.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_16u_AC4R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_16u_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAndC_16u_AC4RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image logical and with constant with unmodified alpha.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_16u_AC4IR(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_16u_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAndC_16u_AC4IRNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image logical and with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_16u_C4R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_16u_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAndC_16u_C4RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image logical and with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_16u_C4IR(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_16u_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAndC_16u_C4IRNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit signed integer channel image logical and with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_32s_C1R(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        int nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_32s_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAndC_32s_C1RNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        int nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image logical and with constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_32s_C1IR(
        int nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_32s_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAndC_32s_C1IRNative(
        int nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit signed integer channel image logical and with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_32s_C3R(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_32s_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAndC_32s_C3RNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit signed integer channel in place image logical and with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_32s_C3IR(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_32s_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAndC_32s_C3IRNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit signed integer channel image logical and with constant with unmodified alpha.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_32s_AC4R(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_32s_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAndC_32s_AC4RNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit signed integer channel in place image logical and with constant with unmodified alpha.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_32s_AC4IR(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_32s_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAndC_32s_AC4IRNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit signed integer channel image logical and with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_32s_C4R(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_32s_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiAndC_32s_C4RNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit signed integer channel in place image logical and with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiAndC_32s_C4IR(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiAndC_32s_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiAndC_32s_C4IRNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of AndC Image
    ///@}
    /** @name OrC Image
     * Pixel by pixel logical or of an image with a constant.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image logical or with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_8u_C1R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_8u_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiOrC_8u_C1RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image logical or with constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_8u_C1IR(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_8u_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiOrC_8u_C1IRNative(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image logical or with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_8u_C3R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_8u_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiOrC_8u_C3RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 8-bit unsigned char channel in place image logical or with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_8u_C3IR(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_8u_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiOrC_8u_C3IRNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image logical or with constant with unmodified alpha.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_8u_AC4R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_8u_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiOrC_8u_AC4RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image logical or with constant with unmodified alpha.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_8u_AC4IR(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_8u_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiOrC_8u_AC4IRNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel image logical or with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_8u_C4R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_8u_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiOrC_8u_C4RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 8-bit unsigned char channel in place image logical or with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_8u_C4IR(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_8u_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiOrC_8u_C4IRNative(
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel image logical or with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_16u_C1R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_16u_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiOrC_16u_C1RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        short nConstant, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 16-bit unsigned short channel in place image logical or with constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_16u_C1IR(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_16u_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiOrC_16u_C1IRNative(
        short nConstant, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 16-bit unsigned short channel image logical or with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_16u_C3R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_16u_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiOrC_16u_C3RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 16-bit unsigned short channel in place image logical or with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_16u_C3IR(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_16u_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiOrC_16u_C3IRNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image logical or with constant with unmodified alpha.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_16u_AC4R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_16u_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiOrC_16u_AC4RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image logical or with constant with unmodified alpha.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_16u_AC4IR(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_16u_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiOrC_16u_AC4IRNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel image logical or with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_16u_C4R(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_16u_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiOrC_16u_C4RNative(
        TypedPointer<Npp16u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 16-bit unsigned short channel in place image logical or with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_16u_C4IR(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_16u_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiOrC_16u_C4IRNative(
        TypedPointer<Npp16u> pConstants, 
        TypedPointer<Npp16u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit signed integer channel image logical or with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_32s_C1R(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        int nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_32s_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiOrC_32s_C1RNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        int nConstant, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 32-bit signed integer channel in place image logical or with constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_32s_C1IR(
        int nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_32s_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiOrC_32s_C1IRNative(
        int nConstant, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit signed integer channel image logical or with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_32s_C3R(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_32s_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiOrC_32s_C3RNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 32-bit signed integer channel in place image logical or with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_32s_C3IR(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_32s_C3IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiOrC_32s_C3IRNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit signed integer channel image logical or with constant with unmodified alpha.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_32s_AC4R(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_32s_AC4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiOrC_32s_AC4RNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit signed integer channel in place image logical or with constant with unmodified alpha.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_32s_AC4IR(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_32s_AC4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiOrC_32s_AC4IRNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit signed integer channel image logical or with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_32s_C4R(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_32s_C4RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiOrC_32s_C4RNative(
        TypedPointer<Npp32s> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Four 32-bit signed integer channel in place image logical or with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiOrC_32s_C4IR(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiOrC_32s_C4IRNative(pConstants, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiOrC_32s_C4IRNative(
        TypedPointer<Npp32s> pConstants, 
        TypedPointer<Npp32s> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    // end of OrC Image
    ///@}
    /** @name XorC Image
     * Pixel by pixel logical exclusive or of an image with a constant.
     */
    ///@{
    /**
     * <pre>
     * One 8-bit unsigned char channel image logical exclusive or with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param nConstant Constant.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiXorC_8u_C1R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiXorC_8u_C1RNative(pSrc1, nSrc1Step, nConstant, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiXorC_8u_C1RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        byte nConstant, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * One 8-bit unsigned char channel in place image logical exclusive or with constant.
     * @param nConstant Constant.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiXorC_8u_C1IR(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiXorC_8u_C1IRNative(nConstant, pSrcDst, nSrcDstStep, oSizeROI));
    }
    private static native int nppiXorC_8u_C1IRNative(
        byte nConstant, 
        TypedPointer<Npp8u> pSrcDst, 
        int nSrcDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 8-bit unsigned char channel image logical exclusive or with constant.
     * @param pSrc1 \ref source_image_pointer.
     * @param nSrc1Step \ref source_image_line_step.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pDst \ref destination_image_pointer.
     * @param nDstStep \ref destination_image_line_step.
     * @param oSizeROI \ref roi_specification.
     * @return
     *  \ref image_data_error_codes, \ref roi_error_codes
     * </pre>
     */
    public static int nppiXorC_8u_C3R(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI)
    {
        return checkResult(nppiXorC_8u_C3RNative(pSrc1, nSrc1Step, pConstants, pDst, nDstStep, oSizeROI));
    }
    private static native int nppiXorC_8u_C3RNative(
        TypedPointer<Npp8u> pSrc1, 
        int nSrc1Step, 
        TypedPointer<Npp8u> pConstants, 
        TypedPointer<Npp8u> pDst, 
        int nDstStep, 
        NppiSize oSizeROI);


    /**
     * <pre>
     * Three 8-bit unsigned char channel in place image logical exclusive or with constant.
     * @param pConstants pointer to a list of constant values, one per channel.
     * @param pSrcDst \ref in_place_image_pointer.
     * @param nSrcDstStep \ref in_place_image_line_step.
     * @param oSize