/*
 * JNpp - Java bindings for NPP, to be used with JCuda
 *
 * Copyright (c) 2010-2012 Marco Hutter - http://www.jcuda.org
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

package jcuda.jnpp;
/**
 * Error Status Codes.
 *
 * Almost all NPP function return error-status information using
 * these return codes.
 * Negative return codes indicate errors, positive return codes indicate
 * warnings, a return code of 0 indicates success.
 */
class NppStatus
{

    // negative return-codes indicate errors

    public static final int NPP_NOT_SUPPORTED_MODE_ERROR = -9999;
    public static final int NPP_ROUND_MODE_NOT_SUPPORTED_ERROR = -213;
    public static final int NPP_RESIZE_NO_OPERATION_ERROR = -50;
    public static final int NPP_BAD_ARG_ERROR = -26;
    public static final int NPP_LUT_NUMBER_OF_LEVELS_ERROR = -25;
    public static final int NPP_TEXTURE_BIND_ERROR = -24;
    public static final int NPP_COEFF_ERROR = -23;
    public static final int NPP_RECT_ERROR = -22;
    public static final int NPP_QUAD_ERROR = -21;
    public static final int NPP_WRONG_INTERSECTION_ROI_ERROR = -20;
    public static final int NPP_NOT_EVEN_STEP_ERROR = -19;
    public static final int NPP_INTERPOLATION_ERROR = -18;
    public static final int NPP_RESIZE_FACTOR_ERROR = -17;
    public static final int NPP_HAAR_CLASSIFIER_PIXEL_MATCH_ERROR = -16;
    public static final int NPP_MEMFREE_ERR = -15;
    public static final int NPP_MEMSET_ERR = -14;
    public static final int NPP_MEMCPY_ERROR = -13;
    public static final int NPP_MEM_ALLOC_ERR = -12;
    public static final int NPP_HISTO_NUMBER_OF_LEVELS_ERROR = -11;
    public static final int NPP_MIRROR_FLIP_ERR = -10;
    public static final int NPP_INVALID_INPUT = -9;
    public static final int NPP_ALIGNMENT_ERROR = -8;
    public static final int NPP_STEP_ERROR = -7;
    public static final int NPP_SIZE_ERROR = -6;
    public static final int NPP_POINTER_ERROR = -5;
    public static final int NPP_NULL_POINTER_ERROR = -4;
    public static final int NPP_CUDA_KERNEL_EXECUTION_ERROR = -3;
    public static final int NPP_NOT_IMPLEMENTED_ERROR = -2;
    public static final int NPP_ERROR = -1;


    // success

    /** Error free operation */
    public static final int NPP_NO_ERROR = 0;

    /** Successful operation (same as NPP_NO_ERROR) */
    public static final int NPP_SUCCESS = NPP_NO_ERROR;


    // positive return-codes indicate warnings

    public static final int NPP_WARNING = 1;
    public static final int NPP_WRONG_INTERSECTION_QUAD_WARNING = 2;

    /**
     * Speed reduction due to uncoalesced memory accesses warning.
     */
    public static final int NPP_MISALIGNED_DST_ROI_WARNING = 3;


    /**
     * Indicates that the quadrangle passed to one of affine warping
     * functions doesn't have necessary properties. First 3 vertices
     * are used, the fourth vertex discarded.
     */
    public static final int NPP_AFFINE_QUAD_INCORRECT_WARNING = 4;


    /**
     * Alias for NPP_AFFINE_QUAD_INCORRECT_WARNING resembling the IPP warning more closely.
     */
    public static final int NPP_AFFINE_QUAD_CHANGED_WARNING = NPP_AFFINE_QUAD_INCORRECT_WARNING;


    /**
     * Indicates that in case of 422/411/420 sampling the ROI width/height was modified for proper processing.
     */
    public static final int NPP_ADJUSTED_ROI_SIZE_WARNING = 5;


    /**
     * Alias for the ADJUSTED_ROI_SIZE_WARNING, this name is closer to IPP's original warning enum.
     */
    public static final int NPP_DOUBLE_SIZE_WARNING = NPP_ADJUSTED_ROI_SIZE_WARNING;


    /**
     * Indicates that for 422/411/420 sampling the ROI width/height was forced to even value.
     */
    public static final int NPP_ODD_ROI_WARNING = 6;



    /**
     * Returns the String identifying the given NppStatus
     *
     * @param m The NppStatus
     * @return The String identifying the given NppStatus
     */
    public static String stringFor(int m)
    {
        switch (m)
        {
            case NPP_NOT_SUPPORTED_MODE_ERROR: return "NPP_NOT_SUPPORTED_MODE_ERROR";
            case NPP_ROUND_MODE_NOT_SUPPORTED_ERROR: return "NPP_ROUND_MODE_NOT_SUPPORTED_ERROR";
            case NPP_RESIZE_NO_OPERATION_ERROR: return "NPP_RESIZE_NO_OPERATION_ERROR";
            case NPP_BAD_ARG_ERROR: return "NPP_BAD_ARG_ERROR";
            case NPP_LUT_NUMBER_OF_LEVELS_ERROR: return "NPP_LUT_NUMBER_OF_LEVELS_ERROR";
            case NPP_TEXTURE_BIND_ERROR: return "NPP_TEXTURE_BIND_ERROR";
            case NPP_COEFF_ERROR: return "NPP_COEFF_ERROR";
            case NPP_RECT_ERROR: return "NPP_RECT_ERROR";
            case NPP_QUAD_ERROR: return "NPP_QUAD_ERROR";
            case NPP_WRONG_INTERSECTION_ROI_ERROR: return "NPP_WRONG_INTERSECTION_ROI_ERROR";
            case NPP_NOT_EVEN_STEP_ERROR: return "NPP_NOT_EVEN_STEP_ERROR";
            case NPP_INTERPOLATION_ERROR: return "NPP_INTERPOLATION_ERROR";
            case NPP_RESIZE_FACTOR_ERROR: return "NPP_RESIZE_FACTOR_ERROR";
            case NPP_HAAR_CLASSIFIER_PIXEL_MATCH_ERROR: return "NPP_HAAR_CLASSIFIER_PIXEL_MATCH_ERROR";
            case NPP_MEMFREE_ERR: return "NPP_MEMFREE_ERR";
            case NPP_MEMSET_ERR: return "NPP_MEMSET_ERR";
            case NPP_MEMCPY_ERROR: return "NPP_MEMCPY_ERROR";
            case NPP_MEM_ALLOC_ERR: return "NPP_MEM_ALLOC_ERR";
            case NPP_HISTO_NUMBER_OF_LEVELS_ERROR: return "NPP_HISTO_NUMBER_OF_LEVELS_ERROR";
            case NPP_MIRROR_FLIP_ERR: return "NPP_MIRROR_FLIP_ERR";
            case NPP_INVALID_INPUT: return "NPP_INVALID_INPUT";
            case NPP_ALIGNMENT_ERROR: return "NPP_ALIGNMENT_ERROR";
            case NPP_STEP_ERROR: return "NPP_STEP_ERROR";
            case NPP_SIZE_ERROR: return "NPP_SIZE_ERROR";
            case NPP_POINTER_ERROR: return "NPP_POINTER_ERROR";
            case NPP_NULL_POINTER_ERROR: return "NPP_NULL_POINTER_ERROR";
            case NPP_CUDA_KERNEL_EXECUTION_ERROR: return "NPP_CUDA_KERNEL_EXECUTION_ERROR";
            case NPP_NOT_IMPLEMENTED_ERROR: return "NPP_NOT_IMPLEMENTED_ERROR";
            case NPP_ERROR: return "NPP_ERROR";
            case NPP_NO_ERROR: return "NPP_NO_ERROR/NPP_SUCCESS";
            //case NPP_SUCCESS: return "NPP_SUCCESS";
            case NPP_WARNING: return "NPP_WARNING";
            case NPP_WRONG_INTERSECTION_QUAD_WARNING: return "NPP_WRONG_INTERSECTION_QUAD_WARNING";
            case NPP_MISALIGNED_DST_ROI_WARNING: return "NPP_MISALIGNED_DST_ROI_WARNING";
            case NPP_AFFINE_QUAD_INCORRECT_WARNING: return "NPP_AFFINE_QUAD_INCORRECT_WARNING/NPP_AFFINE_QUAD_CHANGED_WARNING";
            //case NPP_AFFINE_QUAD_CHANGED_WARNING: return "NPP_AFFINE_QUAD_CHANGED_WARNING";
            case NPP_ADJUSTED_ROI_SIZE_WARNING: return "NPP_ADJUSTED_ROI_SIZE_WARNING/NPP_DOUBLE_SIZE_WARNING";
            //case NPP_DOUBLE_SIZE_WARNING: return "NPP_DOUBLE_SIZE_WARNING";
            case NPP_ODD_ROI_WARNING: return "NPP_ODD_ROI_WARNING";
        }
        return "INVALID NppStatus: " + m;
    }

    /**
     * Private constructor to prevent instantiation.
     */
    private NppStatus()
    {
    }

}
