/*
 * JNpp - Java bindings for NPP, to be used with JCuda
 *
 * Copyright (c) 2010-2012 Marco Hutter - http://www.jcuda.org
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

package jcuda.jnpp;
import java.nio.ByteBuffer;

import jcuda.Pointer;
import jcuda.jnpp.types.*;

/**
 * A typed pointer, that is, a usual {@link Pointer} that has 
 * a type parameter. 
 * 
 * @param <T> The type parameter
 */
public class TypedPointer<T extends NppType> extends Pointer
{
    /**
     * Create a new typed pointer from the given {@link Pointer}
     * 
     * @param <S> The type
     * @param pointer The pointer
     * @return The typed pointer
     */
    public static <S extends NppType> TypedPointer<S> from(Pointer pointer)
    {
        return new TypedPointer<S>(pointer);
    }
    
    /**
     * Creates a new typed pointer to the given buffer.
     * 
     * @param <S> The type 
     * @param buffer The buffer
     * @return The typed pointer
     */
    public static <S extends NppType> TypedPointer<S> to(ByteBuffer buffer)
    {
        return new TypedPointer<S>(Pointer.to(buffer));
    }

    
    /**
     * Creates a new typed pointer to the given array
     * 
     * @param array The array
     * @return The typed pointer
     */
    public static TypedPointer<Npp8u> to8u(byte array[])
    {
        return from(Pointer.to(array));
    }

    /**
     * Creates a new typed pointer to the given array
     * 
     * @param array The array
     * @return The typed pointer
     */
    public static TypedPointer<Npp8s> to8s(byte array[])
    {
        return from(Pointer.to(array));
    }

    /**
     * Creates a new typed pointer to the given array
     * 
     * @param array The array
     * @return The typed pointer
     */
    public static TypedPointer<Npp16u> to16u(short array[])
    {
        return from(Pointer.to(array));
    }

    /**
     * Creates a new typed pointer to the given array
     * 
     * @param array The array
     * @return The typed pointer
     */
    public static TypedPointer<Npp16s> to16s(short array[])
    {
        return from(Pointer.to(array));
    }
    
    /**
     * Creates a new typed pointer to the given array
     * 
     * @param array The array
     * @return The typed pointer
     */
    public static TypedPointer<Npp32u> to32u(int array[])
    {
        return from(Pointer.to(array));
    }

    /**
     * Creates a new typed pointer to the given array
     * 
     * @param array The array
     * @return The typed pointer
     */
    public static TypedPointer<Npp32s> to32s(int array[])
    {
        return from(Pointer.to(array));
    }

    
    /**
     * Creates a new typed pointer to the given array
     * 
     * @param array The array
     * @return The typed pointer
     */
    public static TypedPointer<Npp64u> to64u(long array[])
    {
        return from(Pointer.to(array));
    }

    /**
     * Creates a new typed pointer to the given array
     * 
     * @param array The array
     * @return The typed pointer
     */
    public static TypedPointer<Npp64s> to64s(long array[])
    {
        return from(Pointer.to(array));
    }

    
    /**
     * Creates a new typed pointer to the given array
     * 
     * @param array The array
     * @return The typed pointer
     */
    public static TypedPointer<Npp32f> to32f(float array[])
    {
        return from(Pointer.to(array));
    }

    /**
     * Creates a new typed pointer to the given array
     * 
     * @param array The array
     * @return The typed pointer
     */
    public static TypedPointer<Npp64f> to64f(double array[])
    {
        return from(Pointer.to(array));
    }
    
    
    
    /**
     * Creates a new (null) typed pointer.
     */
    public TypedPointer()
    {
    }

    /**
     * Creates a typed pointer from another pointer
     * 
     * @param other The other pointer
     */
    private TypedPointer(Pointer other)
    {
        super(other);
    }

    /**
     * Returns a new typed pointer with an offset of the given number
     * of bytes
     *
     * @param byteOffset The byte offset for the pointer
     * @return The new typed pointer with the given byte offset
     */
    @Override
    public TypedPointer<T> withByteOffset(long byteOffset)
    {
        return new TypedPointer<T>(super.withByteOffset(byteOffset));
    }
    
}
