/*
 * JNpp - Java bindings for NPP, to be used with JCuda
 *
 * Copyright (c) 2010-2012 Marco Hutter - http://www.jcuda.org
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

package jcuda.jnpp.utilnpp;
import jcuda.jnpp.TypedPointer;
import jcuda.jnpp.types.NppType;


/**
 * Abstract base implementation of an {@link ImagePacked}. 
 *
 * @param <T> The content type of the image
 */
abstract class AbstractImagePacked<T extends NppType> extends AbstractImage implements ImagePacked<T>
{
    /**
     * The pointer to the pixel data of this image
     */
    protected TypedPointer<T> aPixels_;
    
    /**
     * The pitch of this image
     */
    protected int nPitch_;
    
    /**
     * Creates a new, uninitialized AbstractImagePacked
     */
    protected AbstractImagePacked()
    {
        super(0,0);
        aPixels_ = null;
        nPitch_ = 0;
    }
    
    /**
     * Creates a new AbstractImagePacked with the given size
     * 
     * @param width The width
     * @param height The height
     */
    protected AbstractImagePacked(int width, int height)
    {
        super(width, height);
        int pPitch[] = new int[1];
        aPixels_ = allocator().Malloc2D(width, height, pPitch);
        nPitch_ = pPitch[0];
    }
    
    /**
     * Creates a new AbstractImagePacked with the given size
     * 
     * @param rSize The size
     */
    protected AbstractImagePacked(Size rSize)
    {
        this(rSize.nWidth(), rSize.nHeight());
    }
    
    /**
     * Creates a new AbstractImagePacked that is a copy of 
     * the given image.
     * 
     * @param rImage The other image
     */
    protected AbstractImagePacked(ImagePacked<T> rImage)
    {
        super(rImage.size());
        nPitch_ = rImage.pitch();
        allocator().Copy2D(aPixels_, nPitch_, rImage.data(), rImage.pitch(), width(), height());
    }
    
    /**
     * Destroy this AbstractImagePacked.
     * 
     * TODO: This corresponds to the destructor, but has to be called explicitly!
     */
    public void destroy()
    {
        allocator().Free2D(aPixels_);
    }
    
    /**
     * Returns the allocator that performs all memory operations
     * for this image.
     * 
     * @return The allocator.
     */
    protected abstract Allocator<T> allocator();
    
    
    @Override
    public void set(ImagePacked<T> rImage)
    {
        if (rImage == this)
        {
            return;
        }
            
        allocator().Free2D(aPixels_);
        aPixels_ = null;
        nPitch_ = 0;
            
        // assign parent class's data fields (width, height)
        super.set(rImage);
            
        int pPitch[] = new int[1];
        aPixels_  = allocator().Malloc2D(width(), height(), pPitch);
        nPitch_ = pPitch[0];
        allocator().Copy2D(aPixels_, nPitch_, rImage.data(), rImage.pitch(), width(), height());
    }
    
    
    @Override
    public int pitch()
    {
        return nPitch_;
    }

    @Override
    public ImagePacked.Pixel[] pixels(int nx, int ny)
    {
        // TODO Return Pixel array
        throw new UnsupportedOperationException("Not implemented yet");
    }

    @Override
    public TypedPointer<T> data()
    {
        return aPixels_;
    }
    
}
