/*
 * JNpp - Java bindings for NPP, to be used with JCuda
 *
 * Copyright (c) 2010-2012 Marco Hutter - http://www.jcuda.org
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

package jcuda.jnpp.utilnpp;

import jcuda.jnpp.TypedPointer;
import jcuda.jnpp.types.NppType;

/**
 * Interface for an allocator. An allocator manages all memory
 * operations for an image, be it a CPU (host) or NPP (device)
 * image. Implementations of this interface are defined in 
 * the {@link ImageAllocatorsCPU} and {@link ImageAllocatorsNPP}
 * class.
 * 
 * @param <T> The data type for this allocator
 */
public interface Allocator<T extends NppType>
{
    /**
     * Allocate the specified memory block for the data 
     * type of this allocator.
     * 
     * @param nWidth The width
     * @param nHeight The height
     * @param pPitch Will contain the pitch
     * @return The pointer to the memory
     */
    TypedPointer<T> Malloc2D(int nWidth, int nHeight, int pPitch[]);
    
    /**
     * Free the given memory
     * 
     * @param pPixels The memory
     */
    void Free2D(TypedPointer<T> pPixels);
    
    /**
     * 2D copy operation for the specified memory blocks consisting
     * of data with the type that is defined by this allocator.
     * 
     * @param pDst The destination pointer
     * @param nDstPitch The destination pitch
     * @param pSrc The source pointer
     * @param nSrcPitch The source pitch
     * @param nWidth The width
     * @param nHeight The height
     */
    void Copy2D(TypedPointer<T> pDst, long nDstPitch, TypedPointer<T> pSrc, long nSrcPitch, long nWidth, long nHeight);

    /**
     * 2D host-to-device copy for the specified memory blocks 
     * consisting of data with the type that is defined by 
     * this allocator.
     * 
     * @param pDst The destination pointer
     * @param nDstPitch The destination pitch
     * @param pSrc The source pointer
     * @param nSrcPitch The source pitch
     * @param nWidth The width
     * @param nHeight The height
     */
    void HostToDeviceCopy2D(TypedPointer<T> pDst, long nDstPitch, TypedPointer<T> pSrc, long nSrcPitch, long nWidth, long nHeight);

    /**
     * 2D device-to-host copy for the specified memory blocks 
     * consisting of data with the type that is defined by 
     * this allocator.
     * 
     * @param pDst The destination pointer
     * @param nDstPitch The destination pitch
     * @param pSrc The source pointer
     * @param nSrcPitch The source pitch
     * @param nWidth The width
     * @param nHeight The height
     */
    void DeviceToHostCopy2D(TypedPointer<T> pDst, long nDstPitch, TypedPointer<T> pSrc, long nSrcPitch, long nWidth, long nHeight);
}
