/*
 * JNpp - Java bindings for NPP, to be used with JCuda
 *
 * Copyright (c) 2010-2012 Marco Hutter - http://www.jcuda.org
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

package jcuda.jnpp.utilnpp;


import static jcuda.jnpp.JNppUtils.*;

import java.nio.*;

import jcuda.jnpp.TypedPointer;
import jcuda.jnpp.types.*;

/**
 * This class contains factory methods for creating CPU (host)
 * memory allocators for different data types. <br />
 * <br />
 * Note: These correspond to concrete template instantiations
 * of the class that is defined in the ImageAllocatorsCPU 
 * header file.  
 */
public class ImageAllocatorsCPU
{
    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp8u> allocator8u1()
    {
        return new AllocatorCPU<Npp8u>(Npp8u.class, 1);
    }
    
    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp8u> allocator8u2()
    {
        return new AllocatorCPU<Npp8u>(Npp8u.class, 2);
    }
    
    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp8u> allocator8u3()
    {
        return new AllocatorCPU<Npp8u>(Npp8u.class, 3);
    }
    
    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp8u> allocator8u4()
    {
        return new AllocatorCPU<Npp8u>(Npp8u.class, 4);
    }
    

    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp16u> allocator16u1()
    {
        return new AllocatorCPU<Npp16u>(Npp16u.class, 1);
    }
    
    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp16u> allocator16u3()
    {
        return new AllocatorCPU<Npp16u>(Npp16u.class, 3);
    }
    
    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp16u> allocator16u4()
    {
        return new AllocatorCPU<Npp16u>(Npp16u.class, 4);
    }
    

    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp16s> allocator16s1()
    {
        return new AllocatorCPU<Npp16s>(Npp16s.class, 1);
    }
    
    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp16s> allocator16s3()
    {
        return new AllocatorCPU<Npp16s>(Npp16s.class, 3);
    }
    
    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp16s> allocator16s4()
    {
        return new AllocatorCPU<Npp16s>(Npp16s.class, 4);
    }
    

    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp32s> allocator32s1()
    {
        return new AllocatorCPU<Npp32s>(Npp32s.class, 1);
    }
    
    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp32s> allocator32s3()
    {
        return new AllocatorCPU<Npp32s>(Npp32s.class, 3);
    }
    
    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp32s> allocator32s4()
    {
        return new AllocatorCPU<Npp32s>(Npp32s.class, 4);
    }
    

    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp32f> allocator32f1()
    {
        return new AllocatorCPU<Npp32f>(Npp32f.class, 1);
    }
    
    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp32f> allocator32f3()
    {
        return new AllocatorCPU<Npp32f>(Npp32f.class, 3);
    }
    
    /**
     * Creates and returns a new Allocator
     * 
     * @return The allocator
     */
    static Allocator<Npp32f> allocator32f4()
    {
        return new AllocatorCPU<Npp32f>(Npp32f.class, 4);
    }
    
    
    /**
     * Implementation of an Allocator for host memory
     */
    private static class AllocatorCPU<T extends NppType> implements Allocator<T>
    {
        private int N = 0;
        private Class<? extends NppType> typeClass;
        
        AllocatorCPU(Class<? extends NppType> typeClass, int channels)
        {
            this.typeClass = typeClass;
            this.N = channels;
        }
        
        @Override
        public TypedPointer<T> Malloc2D(int nWidth, int nHeight, int[] pPitch)
        {
            NPP_ASSERT(nWidth * nHeight > 0);
            
            int capacity = nWidth * N * nHeight * sizeof(typeClass);
            ByteBuffer buffer = 
                ByteBuffer.allocateDirect(capacity).order(ByteOrder.nativeOrder());
            TypedPointer<T> pResult = TypedPointer.to(buffer);
            pPitch[0] = nWidth * sizeof(typeClass) * N;
            
            return pResult;
        }
        
        @Override
        public void Free2D(TypedPointer<T> pixels)
        {
        }

        @Override
        public void Copy2D(TypedPointer<T> pDst, long nDstPitch,
            TypedPointer<T> pSrc, long nSrcPitch, long nWidth, long nHeight)
        {
            // TODO: Verify Copy2D in ImageAllocatorsCPU! (Maybe use Buffer slice?)
            
            long srcOffset = 0;
            long dstOffset = 0;
            long lineSize = nWidth * N * sizeof(typeClass);

            ByteBuffer srcBuffer = pSrc.getByteBuffer(srcOffset, lineSize);
            ByteBuffer dstBuffer = pDst.getByteBuffer(dstOffset, lineSize);

            for (int iLine = 0; iLine < nHeight; ++iLine)
            {
                dstBuffer.put(srcBuffer);
                
                srcOffset += nSrcPitch;
                dstOffset += nDstPitch;
                
                srcBuffer = pSrc.getByteBuffer(srcOffset, lineSize);
                dstBuffer = pDst.getByteBuffer(dstOffset, lineSize);
            }
        }

        
        @Override
        public void DeviceToHostCopy2D(TypedPointer<T> dst, long dstPitch,
            TypedPointer<T> src, long srcPitch, long width, long height)
        {
            throw new UnsupportedOperationException("CPU allocator can not copy from device");
        }


        @Override
        public void HostToDeviceCopy2D(TypedPointer<T> dst, long dstPitch,
            TypedPointer<T> src, long srcPitch, long width, long height)
        {
            throw new UnsupportedOperationException("CPU allocator can not copy to device");
        }
        
    }


    
}
