/*
 * JNpp - Java bindings for NPP, to be used with JCuda
 *
 * Copyright (c) 2010-2012 Marco Hutter - http://www.jcuda.org
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

package jcuda.jnpp.utilnpp;

import jcuda.jnpp.TypedPointer;
import jcuda.jnpp.types.*;
import jcuda.jnpp.utilnpp.ImagesCPU.ImageCPU;

/**
 * This class provides a set of static inner classes for images with
 * different types.<br />
 * <br /> 
 * Note: This class serves the same purpose as the ImagesNPP header file, 
 * which defines a set of "typedefs" in the form
 * <pre>
 * <code>
 * typedef ImageNPP&lt;Npp8u,  1&gt;   ImageNPP_8u_C1;
 * </code>
 * </pre>
 * This, the inner classes of this class correspond to concrete template 
 * instantiations with appropriate allocators for the image data.   
 */
public class ImagesNPP 
{
    /**
     * Private constructor to prevent instantiation
     */
    private ImagesNPP()
    {
    }
    
    /**
     * This class is the abstract base class for all NPP images. That is,
     * for all images that are handled by NPP and are using device memory.
     *
     * @param <T> The data type of this image
     */
    public static abstract class ImageNPP<T extends NppType> extends AbstractImagePacked<T> implements ImagePacked<T>
    {
        /**
         * Creates a new Image
         */
        protected ImageNPP()
        {
            super();
        }
    
        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        protected ImageNPP(int width, int height)
        {
            super(width, height);
        }
        
        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        protected ImageNPP(Size rSize)
        {
            super(rSize);
        }
    
        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        protected ImageNPP(ImageNPP<T> rImage)
        {
            super(rImage);
        }
        
        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        protected ImageNPP(ImageCPU<T> rImage)
        {
            super(rImage.width(), rImage.height());        
            allocator().HostToDeviceCopy2D(data(), pitch(), rImage.data(), rImage.pitch(), width(), height());
        }
        
        /**
         * Copy the data from this image to the given host pointer
         * 
         * @param pData The host pointer
         * @param nPitch The pitch
         */
        public void copyTo(TypedPointer<T> pData, int nPitch)
        {
            //NPP_ASSERT((ImagePacked<D, N, A>::width() * sizeof(npp::Pixel<D, N>) <= nPitch));
            allocator().DeviceToHostCopy2D(pData, nPitch, data(), pitch(), width(), height());
        }
    
        /**
         * Copy the data from the given host pointer into this image
         * 
         * @param pData The host pointer
         * @param nPitch The pitch
         */
        public void copyFrom(TypedPointer<T> pData, int nPitch) 
        {
            //NPP_ASSERT((ImagePacked<D, N, A>::width() * sizeof(npp::Pixel<D, N>) <= nPitch));
            allocator().HostToDeviceCopy2D(data(), pitch(), pData, nPitch, width(), height());
        }
    }
    
    
    
    //=== auto-generated: =====================================================

    /**
     * Implementation of an image 
     */
    public static class ImageNPP_8u_C1 extends ImageNPP<Npp8u>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp8u> allocator = ImageAllocatorsNPP.allocator8u1();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_8u_C1()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_8u_C1(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_8u_C1(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_8u_C1(ImageNPP<Npp8u> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_8u_C1(ImageCPU<Npp8u> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp8u> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_8u_C2 extends ImageNPP<Npp8u>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp8u> allocator = ImageAllocatorsNPP.allocator8u2();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_8u_C2()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_8u_C2(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_8u_C2(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_8u_C2(ImageNPP<Npp8u> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_8u_C2(ImageCPU<Npp8u> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp8u> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_8u_C3 extends ImageNPP<Npp8u>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp8u> allocator = ImageAllocatorsNPP.allocator8u3();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_8u_C3()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_8u_C3(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_8u_C3(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_8u_C3(ImageNPP<Npp8u> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_8u_C3(ImageCPU<Npp8u> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp8u> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_8u_C4 extends ImageNPP<Npp8u>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp8u> allocator = ImageAllocatorsNPP.allocator8u4();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_8u_C4()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_8u_C4(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_8u_C4(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_8u_C4(ImageNPP<Npp8u> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_8u_C4(ImageCPU<Npp8u> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp8u> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_16u_C1 extends ImageNPP<Npp16u>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp16u> allocator = ImageAllocatorsNPP.allocator16u1();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_16u_C1()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_16u_C1(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_16u_C1(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_16u_C1(ImageNPP<Npp16u> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_16u_C1(ImageCPU<Npp16u> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp16u> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_16u_C2 extends ImageNPP<Npp16u>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp16u> allocator = ImageAllocatorsNPP.allocator16u2();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_16u_C2()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_16u_C2(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_16u_C2(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_16u_C2(ImageNPP<Npp16u> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_16u_C2(ImageCPU<Npp16u> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp16u> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_16u_C3 extends ImageNPP<Npp16u>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp16u> allocator = ImageAllocatorsNPP.allocator16u3();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_16u_C3()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_16u_C3(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_16u_C3(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_16u_C3(ImageNPP<Npp16u> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_16u_C3(ImageCPU<Npp16u> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp16u> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_16u_C4 extends ImageNPP<Npp16u>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp16u> allocator = ImageAllocatorsNPP.allocator16u4();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_16u_C4()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_16u_C4(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_16u_C4(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_16u_C4(ImageNPP<Npp16u> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_16u_C4(ImageCPU<Npp16u> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp16u> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_16s_C1 extends ImageNPP<Npp16s>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp16s> allocator = ImageAllocatorsNPP.allocator16s1();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_16s_C1()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_16s_C1(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_16s_C1(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_16s_C1(ImageNPP<Npp16s> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_16s_C1(ImageCPU<Npp16s> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp16s> allocator()
        {
            return allocator;
        }
    }




    /**
     * Implementation of an image 
     */
    public static class ImageNPP_16s_C4 extends ImageNPP<Npp16s>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp16s> allocator = ImageAllocatorsNPP.allocator16s4();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_16s_C4()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_16s_C4(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_16s_C4(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_16s_C4(ImageNPP<Npp16s> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_16s_C4(ImageCPU<Npp16s> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp16s> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_32s_C1 extends ImageNPP<Npp32s>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp32s> allocator = ImageAllocatorsNPP.allocator32s1();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_32s_C1()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_32s_C1(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_32s_C1(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_32s_C1(ImageNPP<Npp32s> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_32s_C1(ImageCPU<Npp32s> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp32s> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_32s_C3 extends ImageNPP<Npp32s>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp32s> allocator = ImageAllocatorsNPP.allocator32s3();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_32s_C3()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_32s_C3(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_32s_C3(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_32s_C3(ImageNPP<Npp32s> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_32s_C3(ImageCPU<Npp32s> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp32s> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_32s_C4 extends ImageNPP<Npp32s>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp32s> allocator = ImageAllocatorsNPP.allocator32s4();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_32s_C4()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_32s_C4(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_32s_C4(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_32s_C4(ImageNPP<Npp32s> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_32s_C4(ImageCPU<Npp32s> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp32s> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_32f_C1 extends ImageNPP<Npp32f>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp32f> allocator = ImageAllocatorsNPP.allocator32f1();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_32f_C1()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_32f_C1(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_32f_C1(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_32f_C1(ImageNPP<Npp32f> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_32f_C1(ImageCPU<Npp32f> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp32f> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_32f_C2 extends ImageNPP<Npp32f>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp32f> allocator = ImageAllocatorsNPP.allocator32f2();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_32f_C2()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_32f_C2(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_32f_C2(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_32f_C2(ImageNPP<Npp32f> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_32f_C2(ImageCPU<Npp32f> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp32f> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_32f_C3 extends ImageNPP<Npp32f>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp32f> allocator = ImageAllocatorsNPP.allocator32f3();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_32f_C3()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_32f_C3(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_32f_C3(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_32f_C3(ImageNPP<Npp32f> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_32f_C3(ImageCPU<Npp32f> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp32f> allocator()
        {
            return allocator;
        }
    }


    /**
     * Implementation of an image 
     */
    public static class ImageNPP_32f_C4 extends ImageNPP<Npp32f>
    {
        /** The allocator used in this image class */
        private static Allocator<Npp32f> allocator = ImageAllocatorsNPP.allocator32f4();
        
        /**
         * Creates a new Image
         */
        public ImageNPP_32f_C4()
        {
            super();
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param width The width
         * @param height The height
         */
        public ImageNPP_32f_C4(int width, int height)
        {
            super(width, height);
        }

        /**
         * Creates a new Image with the given size
         * 
         * @param rSize The size
         */
        public ImageNPP_32f_C4(Size rSize)
        {
            super(rSize);
        }

        /**
         * Copy-constructor
         * 
         * @param rImage The other image
         */
        public ImageNPP_32f_C4(ImageNPP<Npp32f> rImage)
        {
            super(rImage);
        }

        /**
         * Copy-constructor from host images
         * 
         * @param rImage The other image
         */
        public ImageNPP_32f_C4(ImageCPU<Npp32f> rImage)
        {
            super(rImage);
        }

        @Override
        protected Allocator<Npp32f> allocator()
        {
            return allocator;
        }
    }


    
}
